﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using ToolDevelopmentKit;

    /// <summary>
    /// ドキュメントを閉じます。
    /// </summary>
    public class CloseDocumentHandler : CommandHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        public CloseDocumentHandler(IQueryCommandParameter queryParameter) :
            base(queryParameter)
        {
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return FileCommands.CloseDocument.ID; }
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <param name="bankServiceRef">対象バンクサービスへの参照。</param>
        public static bool Execute(BankServiceReference bankServiceRef, bool closeBankService)
        {
            Ensure.Argument.NotNull(bankServiceRef);

            AppMessageBoxResult result = ConfirmCloseBank(bankServiceRef.Target);

            switch (result)
            {
                case AppMessageBoxResult.Cancel:
                    return false;

                case AppMessageBoxResult.Yes:
                    Document document = bankServiceRef.Target.BankDocument;
                    if (SaveHelper.Save(() => ApplicationBase.Instance.DocumentService.SaveDocument(document), new string[] { document.Resource.Key }) == false)
                    {
                        return false;
                    }
                    break;

                case AppMessageBoxResult.OK:
                case AppMessageBoxResult.No:
                    break;

                default:
                    throw new Exception("invalid messege box result.");
            }

            // 暫定。
            // 他でバンクサービスを参照していない場合だけクローズする
            if (closeBankService)
            {
                CloseBankDocument(bankServiceRef);
            }

            return true;
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            if (GetTargetComponentServiceReference(parameters) is BankServiceReference)
            {
                return (GetTargetDocument(parameters) is BankDocument) ?
                    CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            BankServiceReference bankServiceRef = GetTargetComponentServiceReference(parameters);
            if (null == bankServiceRef) { return false; }

            return Execute(bankServiceRef, true);
        }

        /// <summary>
        /// バンクサービスを閉じてもよいか確認します。
        /// </summary>
        /// <param name="bankService">対象バンクサービス。</param>
        /// <returns>メッセージボックスの戻り値。</returns>
        private static AppMessageBoxResult ConfirmCloseBank(BankService bankService)
        {
            if (null == bankService) { throw new ArgumentNullException("bankService"); }
            if (null == bankService.BankDocument) { return AppMessageBoxResult.Cancel; }

            if (!bankService.BankDocument.IsDirty) { return AppMessageBoxResult.OK; }

            string messageText = MessageResource.Message_ConfirmCloseDocument +
                                    "\n " + Path.GetFileName(bankService.BankDocument.Resource.Key);

            return ApplicationBase.Instance.UIService.ShowMessageBox(messageText,
                                            AppMessageBoxButton.YesNoCancel,
                                            AppMessageBoxImage.Question, AppMessageBoxResult.Yes);
        }

        /// <summary>
        /// 開かれているバンクサービスを閉じます。
        /// </summary>
        private static void CloseBankDocument(BankServiceReference bankServiceReference)
        {
            Assertion.Argument.NotNull(bankServiceReference);

            bankServiceReference.Close();
        }

        /// <summary>
        /// 対象コンポーネントサービスを取得します。
        /// </summary>
        private BankServiceReference GetTargetComponentServiceReference(IQueryCommandParameter parameters)
        {
            Assertion.Argument.NotNull(parameters);

            if (!parameters.ContainsParameter(CommandParameterNames.TargetComponentService))
            {
                return null;
            }

            return parameters.GetParameter(CommandParameterNames.TargetComponentService)
                   as BankServiceReference;
        }

        /// <summary>
        /// 対象バンクドキュメントを取得します。
        /// </summary>
        private BankDocument GetTargetDocument(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            if (!parameters.ContainsParameter(CommandParameterNames.TargetDocuments))
            {
                return null;
            }

            return (parameters.GetParameter(CommandParameterNames.TargetDocuments)
                     as IEnumerable<BankDocument>).FirstOrDefault();
        }
    }
}
