﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using System.Text;
    using System.Threading;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.FileManagement;
    using NintendoWare.SoundMaker.Framework.Preset;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Preview.Service;
    using NintendoWare.SoundMaker.Profile;
    using NintendoWare.ToolDevelopmentKit;
    using Win32 = NintendoWare.SoundFoundation.Core.Win32;

    /// <summary>
    /// アプリケーション基本機能を提供します。
    /// このクラスは WindowsForms や WPF 等の WindowsアプリケーションUIフレームワークに依存しません。
    /// </summary>
    public abstract class ApplicationBase
        : ICommandTarget, IQueryCommandParameter, ICoreServices
    {
        private const string DefaultParametersFileExtension = ".defaultparam";

        private const string BuiltInWavePreprocessExeRelativePath = @"sox\\sox.exe";
        public const string DefaultPlayerName = "DEFAULT_PLAYER";

        private const int Mask_Modifiers = -65536;
        private const int Mask_KeyCode = 65535;
        private const int Key_Shift = 65536;
        private const int Key_Control = 131072;
        private const int Key_Alt = 262144;

        // シングルトン化のためのパラメータ
        private static ApplicationBase _instance;
        private static object _lock = new object();

        // アプリケーション特性
        private IApplicationTraits traits;

        // コンフィギュレーション
        private AppConfiguration _appConfiguration;
        private SoundProjectConfiguration _projectConfiguration;
        private PresetListColumnsConfiguration _presetListColumnsConfiguration;
        private BookmarkConfiguration _bookMarkConfiguration;
        private bool isExistSndEditInSettingFile = false;

        // サービス
        private UIServiceBase _UIService;
        private CommandService _CommandService;
        private DocumentService _DocumentService;
        private DocumentImportService documentImportService;
        private SoundProjectService _ProjectService;
        private BankServiceManager _BankServices;
        private ISoundProjectConvertService _ConvertService;
        private ISoundProjectConvertDependencyService _ConvertDependencyService;
        private RealtimeEditService realtimeEditService = null;
        private ProjectProfileService projectProfileService = null;
        private ParameterValueChangedProxyService parameterValueChangedProxyService = null;
        private CreateComponentService _createComponentService;

        // コマンドバインディング
        private CommandBindingCollection _commandBindings = new CommandBindingCollection();
        private CommandBindingCollection _diabledCommandBindings = new CommandBindingCollection();

        // ショートカット処理
        private int _commandKeyProcessSemaphore = 0;
        private CommandKeyProcessor _commandKeyProcessor;
        private Win32.WindowsHook _windowsHook = new Win32.WindowsHook();

        // ファイルパーサー
        private FileParserStore fileParserStore;

        // ファイル監視
        private FileWatcherService fileWatcherService;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="traits">アプリケーション特性を指定します。</param>
        protected ApplicationBase(IApplicationTraits traits)
        {
            Ensure.Argument.NotNull(traits);

            ApplicationBase.Instance = this;
            if (ApplicationBase.Instance != this)
            {
                return;
            }

            CoreServicesProvider.SetInstance(this);

            this.traits = traits;

            this._appConfiguration = CreateAppConfiguration();
            this._projectConfiguration = CreateProjectConfiguration();
            this._presetListColumnsConfiguration = CreatePresetListColumnsConfiguration();

            this._DocumentService = CreateService(typeof(DocumentService)) as DocumentService;
            this.documentImportService = this.CreateService(typeof(DocumentImportService)) as DocumentImportService;
            this._ProjectService = CreateService(typeof(SoundProjectService)) as SoundProjectService;
            this._BankServices = new BankServiceManager(
                this._DocumentService,
                this.Traits.IntermediateOutputTraits);
            this._createComponentService = CreateService(typeof(CreateComponentService)) as CreateComponentService;

            this._CommandService = CreateService(typeof(CommandService)) as CommandService;
            this._ConvertService = CreateService(typeof(ISoundProjectConvertService)) as ISoundProjectConvertService;
            this._ConvertDependencyService = CreateService(typeof(ISoundProjectConvertDependencyService))
                                        as ISoundProjectConvertDependencyService;
            this.realtimeEditService = CreateService
                (typeof(RealtimeEditService)) as RealtimeEditService;

            this.fileWatcherService = new FileWatcherService();

            this.parameterValueChangedProxyService = CreateService
                (typeof(ParameterValueChangedProxyService)) as ParameterValueChangedProxyService;
            this.projectProfileService = CreateService
                (typeof(ProjectProfileService)) as ProjectProfileService;

            if (this._CommandService == null ||
                this._ConvertService == null ||
                this._ConvertDependencyService == null)
            {
                return;
            }

            //
            this.InitializeCommands();
            this.InitializeCommandBindings();
            this.InitializeDefaultDisabledCommands();

            this._ConvertDependencyService.Attach(_ProjectService);

            this._ProjectService.Opened += OnProjectOpened;
            this._ProjectService.Closed += OnProjectClosed;
            this._ProjectService.ProjectDocumentOpened += OnProjectDocumentOpened;
            this._ProjectService.ProjectDocumentClosed += OnProjectDocumentClosed;
            this._ProjectService.SoundSetDocumentOpened += OnSoundSetDocumentOpened;
            this._ProjectService.SoundSetDocumentClosed += OnSoundSetDocumentClosed;
            this._ProjectService.SoundSetAdded += OnSoundSetAdded;
            this._BankServices.Added += OnBankAdded;
            this._BankServices.Removed += OnBankRemoved;
            this._ConvertService.EndConvert += OnEndConvert;

            this.fileParserStore = new FileParserStore(this.traits.ConversionTraits.IsLittleEndian);
            this.PresetListColumnsService = new PresetListColumnsService(_presetListColumnsConfiguration);
            this.PresetListColumnsService.ConfigPath = Path.Combine(ApplicationConfigurationDirectory, ApplicationConfigurationVersionName);

            this._bookMarkConfiguration = CreateBookmarkConfiguration();
            this.BookmarkService = new BookmarkService(_bookMarkConfiguration);


            this._projectConfiguration.Saving += OnProjectConfigurationSaving;
        }

        /// <summary>
        /// インスタンスを取得します。
        /// </summary>
        public static ApplicationBase Instance
        {
            get { return _instance; }
            protected set
            {
                lock (_lock)
                {
                    if (null != _instance) { return; }
                    _instance = value;
                }
            }
        }

        /// <summary>
        /// アプリケーション特性を取得します。
        /// </summary>
        public IApplicationTraits Traits
        {
            get { return this.traits; }
        }

        /// <summary>
        /// アプリケーション設定を取得します。
        /// </summary>
        public AppConfiguration AppConfiguration
        {
            get { return _appConfiguration; }
        }

        /// <summary>
        /// プロジェクト設定を取得します。
        /// </summary>
        public SoundProjectConfiguration ProjectConfiguration
        {
            get { return _projectConfiguration; }
        }

        /// <summary>
        /// UI サービスを取得します。
        /// </summary>
        public UIServiceBase UIService
        {
            get { return _UIService; }
        }

        /// <summary>
        /// コマンドサービスを取得します。
        /// </summary>
        public CommandService CommandService
        {
            get { return _CommandService; }
        }

        /// <summary>
        /// ドキュメントサービスを取得します。
        /// </summary>
        public DocumentService DocumentService
        {
            get { return _DocumentService; }
        }

        /// <summary>
        /// ドキュメントインポートサービスを取得します。
        /// </summary>
        public DocumentImportService DocumentImportService
        {
            get { return this.documentImportService; }
        }

        /// <summary>
        /// プロジェクトサービスを取得します。
        /// </summary>
        public SoundProjectService ProjectService
        {
            get { return _ProjectService; }
        }

        /// <summary>
        /// バンクサービスの一覧を取得します。
        /// </summary>
        public BankServiceManager BankServices
        {
            get { return _BankServices; }
        }

        /// <summary>
        /// プロジェクトコンバートサービスを取得します。
        /// </summary>
        public ISoundProjectConvertService ConvertService
        {
            get { return _ConvertService; }
        }

        /// <summary>
        /// プロジェクトコンバート依存関係サービスを取得します。
        /// </summary>
        public ISoundProjectConvertDependencyService ConvertDependencyService
        {
            get { return _ConvertDependencyService; }
        }

        /// <summary>
        ///
        /// </summary>
        public RealtimeEditService RealtimeEditService
        {
            get
            {
                return this.realtimeEditService;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public ProjectProfileService ProjectProfileService
        {
            get
            {
                return this.projectProfileService;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public ParameterValueChangedProxyService ParameterValueChangedProxyService
        {
            get
            {
                return this.parameterValueChangedProxyService;
            }
        }

        /// <summary>
        /// コンポーネント作成サービスを取得します。
        /// </summary>
        public CreateComponentService CreateComponentService
        {
            get { return _createComponentService; }
        }

        /// <summary>
        /// アプリケーションが実行できるコマンドの関連付け一覧を取得します。
        /// </summary>
        public CommandBindingCollection CommandBindings
        {
            get { return _commandBindings; }
        }

        /// <summary>
        /// ファイルパーサーストアを取得します。
        /// </summary>
        public FileParserStore FileParserStore
        {
            get { return this.fileParserStore; }
        }

        /// <summary>
        ///
        /// </summary>
        public FileChangedEventHandler FileChanged { get; set; }

        /// <summary>
        /// ヘルプファイルのパスを取得します。
        /// </summary>
        public string HelpURL { get; protected set; }

        /// <summary>
        /// トラブルシューティングヘルプファイルのパスを取得します。
        /// </summary>
        public string TroubleshootingHelpURL { get; protected set; }

        /// <summary>
        /// オプションヘルプファイルのパスを取得します。
        /// </summary>
        public string OptionHelpURL { get; protected set; }

        /// <summary>
        /// プロジェクトヘルプファイルのパスを取得します。
        /// </summary>
        public string ProjectSettingsHelpURL { get; protected set; }

        /// <summary>
        /// ファイル監視サービスを取得します。
        /// </summary>
        public FileWatcherService FileWatcherService
        {
            get { return this.fileWatcherService; }
        }

        /// <summary>
        /// リストカラムのプリセットサービスを取得、設定をします。
        /// </summary>
        public PresetListColumnsService PresetListColumnsService
        {
            get; private set;
        }

        /// <summary>
        /// ブックマークサービスを取得、設定をします。
        /// </summary>
        public BookmarkService BookmarkService
        {
            get; private set;
        }

        /// <summary>
        /// 履歴サービスを取得、設定をします。
        /// </summary>
        public HistoryService HistoryService
        {
            get; private set;
        }

        protected static string BuiltInWavePreprocessExePath
        {
            get
            {
                return Path.Combine(
                    Path.GetDirectoryName(Assembly.GetAssembly(typeof(ApplicationBase)).Location),
                    BuiltInWavePreprocessExeRelativePath);
            }
        }

        /// <summary>
        /// 拡張コマンドの列挙子を返します。
        /// </summary>
        protected virtual IEnumerable<Command> ExtendedCommands
        {
            get
            {
                return SndEditCommands.Commands;
            }
        }

        /// <summary>
        /// アプリケーション設定のファイルパスを取得します。
        /// </summary>
        protected string AppConfigurationFilePath
        {
            get
            {
                return CreateApplicationConfigurationFilePath
                    (ApplicationConfigurationVersionName);
            }
        }

        /// <summary>
        /// アプリケーション設定ファイルのバージョン番号を取得します。
        /// </summary>
        protected string ApplicationConfigurationVersionName
        {
            get
            {
                FileVersionInfo fvi = FileVersionInfo.GetVersionInfo(ApplicationFilePath);
                return fvi.FileVersion;
            }
        }

        /// <summary>
        /// アプリケーション設定ファイルを置くディレクトリ名を取得します。
        /// "AppData/Roaming/Nintendo/???/"
        /// </summary>
        protected string ApplicationConfigurationDirectory
        {
            get
            {
                string companyName = "Nintendo";

                string baseDirectory = Environment.GetFolderPath
                    (Environment.SpecialFolder.ApplicationData);
                string directory = Path.Combine(baseDirectory, companyName);
                directory = Path.Combine(directory, ApplicationConfigurationDirectoryName);
                return directory;
            }
        }

        /// <summary>
        /// アプリケーション設定ファイルを置くディレクトリ名を取得します。
        /// "AppData/Roaming/Nintendo/ここのフォルダ名/"
        /// </summary>
        protected string ApplicationConfigurationDirectoryName
        {
            get
            {
                return Path.GetFileNameWithoutExtension(ApplicationFilePath);
            }
        }

        /// <summary>
        /// アプリケーションのファイルパスを取得します。
        /// </summary>
        protected string ApplicationFilePath
        {
            get
            {
                return this.traits.ApplicationAssembly.Location;
            }
        }

        /// <summary>
        /// プロジェクト設定ファイルパスを取得します。
        /// </summary>
        protected string ProjectConfigurationFilePath
        {
            get
            {
                if (null == ProjectService.ProjectDocument) { return null; }
                return ProjectService.ProjectFilePath + ".settings";
            }
        }

        /// <summary>
        /// コンバート結果のインストール先フォルダパスを取得します。
        /// </summary>
        protected virtual string InstallBinaryFilesDirectoryPath
        {
            get { return null; }
        }

        private XmlSoundProjectDocumentView ProjectViewConfiguration
        {
            get
            {
                if (null == ProjectConfiguration) { return null; }
                if (null == ProjectService.ProjectDocument) { return null; }
                if (!ProjectConfiguration.DocumentViews.
                      ContainsKey(ProjectService.ProjectDocument.Resource.Key)) { return null; }

                return ProjectConfiguration.DocumentViews[ProjectService.ProjectDocument.Resource.Key]
                            as XmlSoundProjectDocumentView;
            }
        }

        /// <summary>
        /// 対象サービスを取得します。
        /// </summary>
        private ComponentService TargetComponentService
        {
            get { return ProjectService; }
        }

        /// <summary>
        /// 対象ドキュメントを取得します。
        /// </summary>
        private SoundDocument TargetDocument
        {
            get { return ProjectService.ProjectDocument; }
        }

        /// <summary>
        /// 対象コンポーネントを取得します。
        /// </summary>
        private Component[] TargetComponents
        {
            get
            {
                if (null == ProjectService.ProjectDocument) { return new Component[0]; }
                if (null == ProjectService.ProjectDocument.Project) { return new Component[0]; }
                return new Component[] { ProjectService.ProjectDocument.Project };
            }
        }

        /// <summary>
        /// アプリケーションがアクティブになると発生します。
        /// </summary>
        public event EventHandler AppActivated;

        /// <summary>
        /// アプリケーションが非アクティブになると発生します。
        /// </summary>
        public event EventHandler AppDeactivated;

        public event CommandKeyProcessEventHandler CommandKeyProcessed;

        public event CommandKeyProcessEventHandler CommandKeyProcessing;

        /// <summary>
        /// アプリケーションを実行します。
        /// </summary>
        /// <param name="args">コマンドライン引数。</param>
        public void Run(string[] args)
        {
            LoadAppConfiguration();
            _projectConfiguration.LoadDefaults(true);
            this.PresetListColumnsService.Load();

            InitializeUIService();

            _commandKeyProcessor = new CommandKeyProcessor(CommandService);
            _commandKeyProcessor.FirstKeyChanged += OnCommandInputFirstKeyChanged;

            _windowsHook.Hook += OnKeyboardHook;
            _windowsHook.SetHook(Win32.WH.WH_KEYBOARD);

            this.RunApplication(args);

            _windowsHook.Unhook();

            _appConfiguration.Save(AppConfigurationFilePath);
        }

        /// <summary>
        /// アプリケーションを終了します。
        /// </summary>
        /// <returns>成功した場合は true、失敗またはキャンセルされた場合は false。</returns>
        public bool Exit()
        {
            if (null != ProjectService.ProjectDocument)
            {
                if (!ExecuteCommand(FileCommands.CloseProject))
                {
                    return false;
                }
            }
            else
            {
                AppConfiguration.Options.Application.General.StartupProject.Value = string.Empty;
            }

            return ExitApplication();
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドの状態。</returns>
        public CommandStatus QueryCommandStatus(Command command)
        {
            if (null == command) { throw new ArgumentNullException("command"); }

            ICommandTarget target = (this as ICommandTarget).FindTarget(command);

            if (null != target && target.QueryStatus(command).IsSupported())
            {
                return target.QueryStatus(command);
            }

            return CommandStatus.Unsupported;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドを実行した場合は true、キャンセルされた場合は false。</returns>
        public bool ExecuteCommand(Command command)
        {
            try
            {
                return ExecuteCommandInternal(command);
            }
            catch (Exception e)
            {
                UIService.ShowMessageBox(e.Message);
                return false;
            }
        }

        /// <summary>
        /// ショートカットキーによるコマンド処理を有効化します。
        /// </summary>
        public void EnableCommandKeyProcess()
        {
            _commandKeyProcessSemaphore = 0;
        }

        /// <summary>
        /// ショートカットキーによるコマンド処理を無効化します。
        /// </summary>
        public void DisableCommandKeyProcess()
        {
            _commandKeyProcessSemaphore = 1;
        }

        public void SetSndEditConfiguration()
        {
            if (ProjectConfiguration.SoundProject.SndEditList != null)
            {
                XmlSndEdit[] items = ProjectConfiguration.SoundProject.SndEditList.SndEdit;
                if (items != null)
                {
                    SetSndEditConfiguration(items);
                }
            }
        }


        public void SetSndEditConfiguration(XmlSndEdit[] items)
        {
            foreach (XmlSndEdit item in items)
            {
                Component component = ProjectService.Sounds
                    .Concat<SoundSetItem>(ProjectService.SoundSetBanks)
                    .FirstOrDefault(s => s.Name == item.Name);

                if (component == null)
                {
                    continue;
                }

                if (string.IsNullOrEmpty(item.SoundSetName) == false &&
                    string.IsNullOrEmpty(item.Type) == false &&
                    (component.GetSoundSet().Name != item.SoundSetName ||
                     component.GetComponentTypeName() != item.Type))
                {
                    continue;
                }

                component.Parameters[ProjectParameterNames.SndEditEnabled].Value = item.Enabled;
                component.Parameters[ProjectParameterNames.SndEdit].Value = true;
            }
        }

        public string GetWavePreprocessExeFullPath()
        {
            if (this.ProjectService == null || this.ProjectService.Project == null)
            {
                return string.Empty;
            }

            return this.ProjectService.Project.DoUseBuiltInWavePreprocessExe
                ? BuiltInWavePreprocessExePath
                : this.ProjectService.Project.GetWavePreprocessExeFullPath(Path.GetDirectoryName(this.ProjectService.ProjectFilePath));
        }

        // <summary>
        // パラメータ初期値設定ファイルのパスを取得します。
        // </summary>
        /// <param name="projectFilePath">プロジェクトファイルのパス</param>
        /// <returns>パラメータ初期値設定ファイルのパス</returns>
        public string GetDefaultParametersFilePath(string projectFilePath)
        {
            Debug.Assert(string.IsNullOrEmpty(projectFilePath) == false);

            return projectFilePath + DefaultParametersFileExtension;
        }

        // <summary>
        // パラメータ初期値設定ファイルのパスを取得します。
        // </summary>
        /// <returns>パラメータ初期値設定ファイルのパス</returns>
        public string GetDefaultParametersFilePath()
        {
            return this.GetDefaultParametersFilePath(this.ProjectService.ProjectFilePath);
        }

        bool IQueryCommandParameter.ContainsParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentService:
                case CommandParameterNames.TargetDocuments:
                case CommandParameterNames.TargetComponents:
                    return true;
            }

            return false;
        }

        object IQueryCommandParameter.GetParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentService:
                    return TargetComponentService;

                case CommandParameterNames.TargetDocuments:
                    if (null == TargetDocument) { return new SoundDocument[0]; }
                    return new SoundDocument[] { TargetDocument };

                case CommandParameterNames.TargetComponents:
                    return TargetComponents;
            }

            throw new KeyNotFoundException();
        }

        /// <summary>
        /// 指定コマンドを実行できるコマンドターゲットを検索します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>他のコマンドターゲット。</returns>
        ICommandTarget ICommandTarget.FindTarget(Command command)
        {
            ICommandTarget target = FindOtherTarget(command);
            if (target == null &&
                _commandBindings.Contains(command.ID) == true)
            {
                target = this;
            }

            if (null == target && _diabledCommandBindings.Contains(command.ID))
            {
                target = this;
            }

            return target;
        }

        /// <summary>
        /// 指定コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        CommandStatus ICommandTarget.QueryStatus(Command command)
        {
            if (_commandBindings.Contains(command.ID))
            {
                return _commandBindings[command.ID].QueryStatus(command);
            }

            if (_diabledCommandBindings.Contains(command.ID))
            {
                return _diabledCommandBindings[command.ID].QueryStatus(command);
            }

            return CommandStatus.Unsupported;
        }

        /// <summary>
        /// 指定コマンドを実行します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドを実行した場合は true、キャンセルされた場合は false。</returns>
        bool ICommandTarget.Execute(Command command)
        {
            if (!_commandBindings.Contains(command.ID)) { return false; }

            try
            {
                return _commandBindings[command.ID].Execute(command);
            }
            catch (FileNotFoundException e)
            {
                UIService.ShowMessageBox(MessageResource.Message_FileNotFound + "\n" +
                                         e.FileName);
                return false;
            }
            catch (FileNotSupportedException e)
            {
                UIService.ShowMessageBox(string.Format("{0}\n{1}", e.Message, e.FilePath));
                return false;
            }
            catch (Exception e)
            {
                UIService.ShowMessageBox(e.Message);
                return false;
            }
        }

        /// <summary>
        /// アプリケーションを実行します。
        /// </summary>
        /// <param name="args">コマンドライン引数。</param>
        protected abstract void RunApplication(string[] args);

        /// <summary>
        /// アプリケーションを終了します。
        /// </summary>
        protected abstract bool ExitApplication();

        /// <summary>
        /// スタートアッププロジェクトまたはコマンドライン指定プロジェクトを読み込みます。
        /// </summary>
        /// <param name="filePath">プロジェクトのファイルパスを指定します。</param>
        protected void StartupProject(string filePath, Func<string, bool> openProjectFileFunc)
        {
            Ensure.Argument.NotNull(filePath);

            string targetProjectFilePath = string.Empty;

            if (filePath.Length > 0)
            {
                targetProjectFilePath = filePath;
            }
            else
            {
                if (!AppConfiguration.Options.Application.General.StartupProject.Enable) { return; }
                if (null == AppConfiguration.Options.Application.General.StartupProject.Value) { return; }

                targetProjectFilePath = AppConfiguration.Options.Application.General.StartupProject.Value;
            }

            if (0 == targetProjectFilePath.Length) { return; }

            openProjectFileFunc(targetProjectFilePath);
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドを実行した場合は true、キャンセルされた場合は false。</returns>
        protected virtual bool ExecuteCommandInternal(Command command)
        {
            if (null == command) { throw new ArgumentNullException("command"); }

            ICommandTarget target = (this as ICommandTarget).FindTarget(command);
            if (null == target) { return false; }

            if (target.QueryStatus(command).IsEnabled())
            {
                return target.Execute(command);
            }

            return false;
        }

        /// <summary>
        /// AppActivated イベントを発行します。
        /// </summary>
        protected void RaiseAppActivated()
        {
            if (null != AppActivated)
            {
                AppActivated(this, new EventArgs());
            }
        }

        /// <summary>
        /// AppDeactivated イベントを発行します。
        /// </summary>
        protected void RaiseAppDeactivated()
        {
            if (null != AppDeactivated)
            {
                AppDeactivated(this, new EventArgs());
            }
        }

        /// <summary>
        /// CommandKeyProcessed イベントを発行します。
        /// </summary>
        /// <param name="e">コマンドキー処理イベントデータ。</param>
        protected void OnCommandKeyProcessed(CommandKeyProcessEventArgs e)
        {
            if (null != CommandKeyProcessed)
            {
                CommandKeyProcessed(this, e);
            }
        }

        /// <summary>
        /// CommandKeyProcessing イベントを発行します。
        /// </summary>
        /// <param name="e">コマンドキー処理イベントデータ。</param>
        protected void OnCommandKeyProcessing(CommandKeyProcessEventArgs e)
        {
            if (null != CommandKeyProcessing)
            {
                CommandKeyProcessing(this, e);
            }
        }

        /// <summary>
        /// アプリケーション設定クラスを作成します。
        /// </summary>
        /// <returns>作成した AppConfiguration。</returns>
        protected abstract AppConfiguration CreateAppConfiguration();

        /// <summary>
        /// プロジェクト設定クラスを作成します。
        /// </summary>
        /// <returns>作成した SoundProjectConfiguration。</returns>
        protected abstract SoundProjectConfiguration CreateProjectConfiguration();

        /// <summary>
        /// カラムプリセット設定クラスを作成します。
        /// </summary>
        /// <returns>作成した PresetListColumnsConfiguration。</returns>
        protected abstract PresetListColumnsConfiguration CreatePresetListColumnsConfiguration();

        /// <summary>
        /// ブックマーク設定クラスを作成します。
        /// </summary>
        /// <returns>作成した BookmarkConfiguration。</returns>
        protected abstract BookmarkConfiguration CreateBookmarkConfiguration();

        /// <summary>
        /// サービスのインスタンスを作成します。
        /// </summary>
        /// <param name="serviceType">作成するサービスの種類。</param>
        /// <returns>サービスのインスタンス。</returns>
        protected virtual object CreateService(Type serviceType)
        {
            if (serviceType == typeof(DocumentService))
            {
                return new DocumentService(CreateDocumentServiceTraits());
            }

            if (serviceType == typeof(DocumentImportService))
            {
                return new DocumentImportService(this.CreateDocumentImportServiceTraits());
            }

            if (serviceType == typeof(SoundProjectService))
            {
                return new SoundProjectService(
                    this.DocumentService,
                    this.Traits.IntermediateOutputTraits);
            }

            if (serviceType == typeof(CommandService))
            {
                return new CommandService();
            }

            if (serviceType == typeof(ParameterValueChangedProxyService))
            {
                return new ParameterValueChangedProxyService(this.ProjectService);
            }

            if (serviceType == typeof(ProjectProfileService))
            {
                return new ProjectProfileService
                    (this.DocumentService,
                      this.BankServices,
                      this.ParameterValueChangedProxyService,
                      this.fileWatcherService,
                      this.Traits.IntermediateOutputTraits);
            }

            if (serviceType == typeof(CreateComponentService))
            {
                return new CreateComponentService(this.Traits.ComponentTraits);
            }

            return null;
        }

        /// <summary>
        /// ドキュメントサービスの特性を生成します。
        /// </summary>
        protected abstract IDocumentServiceTraits CreateDocumentServiceTraits();

        /// <summary>
        /// ドキュメントインポートサービスの特性を生成します。
        /// </summary>
        protected abstract DocumentImportServiceTraits CreateDocumentImportServiceTraits();

        /// <summary>
        /// UIサービスを初期化します。
        /// </summary>
        protected void InitializeUIService()
        {
            _UIService = CreateService(typeof(UIServiceBase)) as UIServiceBase;
            _UIService.Initialize();
        }

        /// <summary>
        /// コマンドの関連付けを初期化します。
        /// </summary>
        protected virtual void InitializeCommandBindings()
        {
            // ファイル関連コマンドハンドラ
            BindCommand(FileCommands.NewProject);
            BindCommand(FileCommands.OpenProject);
            BindCommand(FileCommands.CloseProject);
            BindCommand(FileCommands.SaveAll);
            BindCommand(FileCommands.ExitApplication);
            BindCommand(FileCommands.ImportProject);
            BindCommand(FileCommands.ImportSoundSet);

            // プロジェクト関連コマンドハンドラ
            BindCommand(ProjectCommands.Convert);
            BindCommand(ProjectCommands.Reconvert);
            BindCommand(ProjectCommands.SelectReconvert);
        }

        /// <summary>
        /// デフォルト無効で表示するコマンドを設定します。
        /// </summary>
        protected virtual void InitializeDefaultDisabledCommands()
        {
            // ファイル
            foreach (Command command in FileCommands.Commands)
            {
                DisableCommandByDefault(command);
            }

            // 編集
            foreach (Command command in EditCommands.Commands)
            {
                DisableCommandByDefault(command);
            }

            // 表示
            foreach (Command command in ViewCommands.Commands)
            {
                DisableCommandByDefault(command);
            }

            // プロジェクト
            foreach (Command command in ProjectCommands.Commands)
            {
                DisableCommandByDefault(command);
            }

            // プレビュー
            foreach (Command command in PreviewCommands.Commands.Where(target => !target.ID.StartsWith("_")))
            {
                DisableCommandByDefault(command);
            }

            // ツール
            foreach (Command command in ToolCommands.Commands)
            {
                DisableCommandByDefault(command);
            }

            // ウィンドウ
            foreach (Command command in WindowCommands.Commands)
            {
                DisableCommandByDefault(command);
            }

            // ヘルプ
            foreach (Command command in HelpCommands.Commands)
            {
                DisableCommandByDefault(command);
            }
        }

        /// <summary>
        /// コマンドバインディングを作成します。
        /// </summary>
        /// <param name="serviceType">関連付けるコマンド。</param>
        /// <returns>コマンドバインディングのインスタンス。</returns>
        protected virtual CommandBinding CreateCommandBinding(Command command)
        {
            if (null == command) { throw new ArgumentNullException("command"); }

            // ファイル関連コマンドハンドラ
            if (command == FileCommands.CloseProject)
            {
                return new CommandBinding(this, new CloseProjectHandler());
            }
            if (command == FileCommands.SaveAll)
            {
                return new CommandBinding(this, new SaveAllHandler());
            }

            // プロジェクト関連コマンドハンドラ
            if (command == ProjectCommands.Convert)
            {
                return new CommandBinding(this, command.ID,
                                           QueryStatusExecuteConvert, ExecuteConvert);
            }
            if (command == ProjectCommands.Reconvert)
            {
                return new CommandBinding(this, command.ID,
                                           QueryStatusExecuteReconvert, ExecuteReconvert);
            }
            if (command == ProjectCommands.SelectReconvert)
            {
                return new CommandBinding(this, command.ID,
                                           QueryStatusExecuteSelectReconvert, ExecuteSelectReconvert);
            }

            return null;
        }

        /// <summary>
        /// 指定コマンドを関連付けます。
        /// </summary>
        /// <param name="command">関連付けるコマンド。</param>
        protected void BindCommand(Command command)
        {
            BindCommand(command, null);
        }

        /// <summary>
        /// 指定コマンドを関連付けます。
        /// </summary>
        /// <param name="command">関連付けるコマンド。</param>
        /// <param name="executeHandler">関連付ける Execute ハンドラ。</param>
        protected void BindCommand(Command command, CommandEventHandler executedHandler)
        {
            if (null == command) { throw new ArgumentNullException("command"); }

            CommandBinding binding = CreateCommandBinding(command);
            if (null == binding) { return; }

            _commandBindings.Add(binding);

            if (null != executedHandler)
            {
                binding.CommandExecuted += executedHandler;
            }
        }

        protected void DisableCommandByDefault(Command command)
        {
            if (null == command) { throw new ArgumentNullException("command"); }

            if (_diabledCommandBindings.Contains(command.ID)) { return; }

            _diabledCommandBindings.Add(
                new CommandBinding(this, new DisableCommandHandler(
                                                command.ID,
                                                CommandStatus.SupportedAndVisible)));
        }

        /// <summary>
        /// 指定コマンドを実行できる他のコマンドターゲットを検索します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>他のコマンドターゲット。</returns>
        protected virtual ICommandTarget FindOtherTarget(Command command) { return null; }

        /// <summary>
        /// デフォルトのアプリケーション設定が読み込まれると発生します。
        /// </summary>
        protected virtual void OnAppConfigurationDefaultLoad(EventArgs e)
        {
        }

        /// <summary>
        /// デフォルトのプロジェクト設定が読み込まれると発生します。
        /// </summary>
        protected virtual void OnProjectConfigurationDefaultLoad(EventArgs e)
        {
        }

        /// <summary>
        /// デフォルトのプリセット設定が読み込まれると発生します。
        /// </summary>
        protected virtual void OnPresetConfigurationDefaultLoad(EventArgs e)
        {
        }

        /// <summary>
        /// プロジェクトが開かれると発生します。
        /// </summary>
        /// <param name="e"></param>
        protected virtual void OnProjectOpened(object sender, EventArgs e)
        {
            AppConfiguration.Options.Application.
                General.StartupProject.Value = ProjectService.ProjectDocument.Resource.Key;

            AppConfiguration.MruProjects.PushFront(ProjectService.ProjectDocument.Resource.Key);

            LoadProjectConfiguration(ProjectConfigurationFilePath);

            _ConvertDependencyService.UpdateSoundProjectBinaryInformations();

            this._bookMarkConfiguration = CreateBookmarkConfiguration();
            this.BookmarkService = new BookmarkService(_bookMarkConfiguration);
            this.BookmarkService.ProjectFilePath = this.ProjectService.ProjectFilePath;
            this.BookmarkService.Load();
            this.HistoryService = new HistoryService(ProjectConfiguration.SoundProject.HistoryList);
        }

        /// <summary>
        /// プロジェクトが閉じられると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントデータ。</param>
        protected virtual void OnProjectClosed(object sender, EventArgs e)
        {
            _projectConfiguration.LoadDefaults(true);

            ProjectProfileService.WaveReferenceCountMeasurer.ClearAllData();
            ProjectProfileService.WaveReferenceCountMeasurer.TerminateAllMonitoringFile();
        }

        /// <summary>
        /// サウンドセットが追加されると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected virtual void OnSoundSetAdded(object sender, SoundDocumentEventArgs e)
        {
            InitializeSoundSetViewConfiguration(e.Document.Resource.Key);

            foreach (SoundSetBankBase bank in ProjectService.SoundSetBanks)
            {
                InitializeBankViewConfiguration(bank.FilePath);
            }
        }

        /// <summary>
        /// バンクサービスが追加されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">バンクサービスイベントデータ。</param>
        protected virtual void OnBankAdded(object sender, BankServiceEventArgs e)
        {
            _ConvertDependencyService.UpdateBankBinaryInformations(e.BankService);

            InitializeBankViewConfiguration(e.BankService.BankDocument.Resource.Key);

            this.fileWatcherService.Add(e.BankService.BankDocument.Resource.Key, OnFileChanged);
        }

        /// <summary>
        /// バンクサービスが削除されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">バンクサービスイベントデータ。</param>
        protected virtual void OnBankRemoved(object sender, BankServiceEventArgs e)
        {
            this.fileWatcherService.Remove(e.BankService.BankDocument.Resource.Key, OnFileChanged);
        }

        /// <summary>
        /// コンバートが完了すると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">コンバートイベントデータ。</param>
        protected virtual void OnEndConvert(object sender, SoundProjectConvertEventArgs e)
        {
            if (e.IsPartsConvert)
            {
                return;
            }

            _ConvertDependencyService.UpdateSoundProjectBinaryInformations();

            foreach (BankService bankService in _BankServices.Values)
            {
                _ConvertDependencyService.UpdateBankBinaryInformations(bankService);
            }
        }

        // 暫定処置なので仮想関数にしてしまいます。
        protected virtual CommandStatus GetConvertCommandStatus()
        {
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// アプリケーション設定ファイルのパスを作成します。(バージョン文字列を指定)
        /// </summary>
        protected string CreateApplicationConfigurationFilePath(string versionName)
        {
            string configFolder = Path.Combine(ApplicationConfigurationDirectory, versionName);
            string name = Path.GetFileName(ApplicationFilePath);
            name = Path.ChangeExtension(name, "xml");

            return Path.Combine(configFolder, name);
        }

        /// <summary>
        /// プロジェクト設定が保存される前に発生します。
        /// </summary>
        /// <param name="e"></param>
        protected virtual void OnProjectConfigurationSaving(object sender, EventArgs e)
        {
            XmlSndEditList list = new XmlSndEditList();
            list.SndEdit = this.RealtimeEditService.MonitoringTargets
                .Select(t => new XmlSndEdit(t))
                .Cast<XmlSndEdit>()
                .ToArray();
            ProjectConfiguration.SoundProject.SndEditList = list;
            ProjectConfiguration.SoundProject.HistoryList = HistoryService.GetHistoryList();
        }

        /// <summary>
        /// アプリケーションの設定ファイルを読み込みます。(NW4?_SoundMaker.xml)
        /// </summary>
        private void LoadAppConfiguration()
        {
            // 設定ファイルがディレクトリが存在しない場合には作成します。
            string filePath = AppConfigurationFilePath;
            string directory = Path.GetDirectoryName(filePath);
            if (Directory.Exists(directory) == false)
            {
                try
                {
                    Directory.CreateDirectory(directory);
                }
                catch
                {
                }
            }

            // 設定ファイルが存在しない場合には優先順位に従ってで他のディレクトリからコピーします。
            if (File.Exists(filePath) == false)
            {
                TakeOverApplicationConfigurationFile();
            }

            //
            if (this._appConfiguration.Load(filePath) == false)
            {
                OnAppConfigurationDefaultLoad(EventArgs.Empty);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual bool TakeOverApplicationConfigurationFile()
        {
            string filePath = Path.ChangeExtension(ApplicationFilePath, "xml");

            try
            {
                // アプリケーションと同じディレクトリに存在するのか？
                if (File.Exists(filePath) == true)
                {
                    File.Copy(filePath, AppConfigurationFilePath, true);
                    return true;
                }

                // 現行のオプションファイルからコピー
                Version currentVersion = new Version(ApplicationConfigurationVersionName);
                string oldFilePath = Path.Combine(ApplicationConfigurationDirectory, Path.GetFileName(filePath));
                if (this.CopyOldVersionFile(currentVersion, oldFilePath, AppConfigurationFilePath) == true)
                {
                    return true;
                }
            }
            catch
            {
            }

            return false;
        }

        protected bool CopyOldVersionFile(Version currentVersion, string oldFilePath, string newFilePath)
        {
            string oldConfigFolder = Path.GetDirectoryName(oldFilePath);
            string oldFileName = Path.GetFileName(oldFilePath);

            if (Directory.Exists(oldConfigFolder) == false)
            {
                return false;
            }
            string[] directories = Directory.GetDirectories(oldConfigFolder);

            var versions = directories.Select(d => new Version(Path.GetFileName(d))).Where(v => v != currentVersion).OrderByDescending(a => a);

            foreach (Version version in versions)
            {
                string filePath = Path.Combine(oldConfigFolder, version.ToString(), oldFileName);

                if (File.Exists(filePath) == true)
                {
                    File.Copy(filePath, newFilePath, true);
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public void LoadProjectConfiguration()
        {
            this.LoadProjectConfiguration(ProjectConfigurationFilePath);
        }

        private void LoadProjectConfiguration(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            bool result = _projectConfiguration.Load(filePath);
            if (result && ValidateProjectConfiguration())
            {
                this.isExistSndEditInSettingFile =
                    IsExistSndEditInSettingFile(_projectConfiguration.DocumentViews);
                foreach (XmlDocumentView xmlDocumentView in _projectConfiguration.DocumentViews.Values)
                {
                    MergeSettingAndDefaultListColumns(xmlDocumentView);
                    if (this.realtimeEditService == null &&
                        this.isExistSndEditInSettingFile == false)
                    {
                        DeleteSndEditFromSetting(xmlDocumentView);
                    }
                }

                return;
            }
            else
            {
                this.isExistSndEditInSettingFile = false;
            }

            InitializeProjectViewConfiguration();

            foreach (SoundSetDocument document in ProjectService.SoundSetDocuments)
            {
                InitializeSoundSetViewConfiguration(document.Resource.Key);
            }

            foreach (SoundSetBankBase bank in ProjectService.SoundSetBanks)
            {
                InitializeBankViewConfiguration(bank.FilePath);
            }

            foreach (XmlDocumentView xmlDocumentView in _projectConfiguration.DocumentViews.Values)
            {
                MergeSettingAndDefaultListColumns(xmlDocumentView);
            }

            if (this.realtimeEditService == null &&
                this.isExistSndEditInSettingFile == false)
            {
                foreach (XmlDocumentView xmlDocumentView in _projectConfiguration.DocumentViews.Values)
                {
                    DeleteSndEditFromSetting(xmlDocumentView);
                }
            }

            OnProjectConfigurationDefaultLoad(EventArgs.Empty);
        }

        private bool IsExistSndEditInSettingFile(IDictionary<string, XmlDocumentView> xmlDocumentView)
        {
            foreach (XmlDocumentView view in xmlDocumentView.Values)
            {
                XmlList[] xmlList = GetXmlList(view);
                foreach (XmlList list in xmlList)
                {
                    foreach (XmlListColumn column in list.ListColumns.ListColumn)
                    {
                        if (column.Name == "SndEdit")
                        {
                            return true;
                        }
                    }
                }
            }

            return false;
        }

        private void DeleteSndEditFromSetting(XmlDocumentView setting)
        {
            XmlList[] xmlList = GetXmlList(setting);
            List<XmlList> newList = new List<XmlList>();

            foreach (XmlList list in xmlList)
            {
                if (list.Name != "SndEditList")
                {
                    newList.Add(list);
                    DeleteSndEditListColumns(list.ListColumns);
                }
            }

            SetXmlList(setting, newList.ToArray());
        }

        private void DeleteSndEditListColumns(XmlListColumns settingListColumns)
        {
            List<XmlListColumn> newColumns = new List<XmlListColumn>();

            foreach (XmlListColumn column in settingListColumns.ListColumn)
            {
                if (column.Name != "SndEdit")
                {
                    newColumns.Add(column);
                }
            }

            settingListColumns.ListColumn = newColumns.ToArray();
        }

        private XmlList[] GetXmlList(XmlDocumentView view)
        {
            XmlList[] xmlList = new XmlList[0];

            if (view is XmlSoundProjectDocumentView == true)
            {
                XmlSoundProjectDocumentView dotSetting = view as XmlSoundProjectDocumentView;
                xmlList = dotSetting.ListDefaults.List;
            }
            else if (view is XmlSoundSetDocumentView == true)
            {
                XmlSoundSetDocumentView dotSetting = view as XmlSoundSetDocumentView;
                xmlList = dotSetting.Lists.List;
            }
            else if (view is XmlBankDocumentView == true)
            {
                XmlBankDocumentView dotSetting = view as XmlBankDocumentView;
                xmlList = dotSetting.Lists.List;
            }

            return xmlList;
        }

        private void SetXmlList(XmlDocumentView view, XmlList[] xmlList)
        {
            if (view is XmlSoundProjectDocumentView == true)
            {
                XmlSoundProjectDocumentView dotSetting = view as XmlSoundProjectDocumentView;
                dotSetting.ListDefaults.List = xmlList;
            }
            else if (view is XmlSoundSetDocumentView == true)
            {
                XmlSoundSetDocumentView dotSetting = view as XmlSoundSetDocumentView;
                dotSetting.Lists.List = xmlList;
            }
            else if (view is XmlBankDocumentView == true)
            {
                XmlBankDocumentView dotSetting = view as XmlBankDocumentView;
                dotSetting.Lists.List = xmlList;
            }
        }

        private void MergeSettingAndDefaultListColumns(XmlDocumentView setting)
        {
            XmlList[] xmlList = GetXmlList(setting);

            XmlSoundProjectDocumentView defaultSetting =
                ProjectConfiguration.LoadSoundProjectDocumentViewDefault();
            IDictionary<string, XmlList> defaultSettingDictionary =
                defaultSetting.ListDefaults.ExportDictionary();

            foreach (XmlList list in xmlList)
            {
                if (defaultSettingDictionary.ContainsKey(list.Name) == true)
                {
                    XmlList defaultList = defaultSettingDictionary[list.Name];
                    MergeListColumns(list.ListColumns, defaultList.ListColumns);
                }
            }
        }

        private void MergeListColumns(XmlListColumns settingListColumns, XmlListColumns defaultListColumns)
        {
            // デフォルトにないカラムは除外する。
            List<XmlListColumn> newColumns = new List<XmlListColumn>();
            IDictionary<string, XmlListColumn> defaultListColumnsDictionary =
                defaultListColumns.ExportDictionary();

            foreach (XmlListColumn column in settingListColumns.ListColumn)
            {
                if (defaultListColumnsDictionary.ContainsKey(column.Name) == true ||
                    ListTraits.IsUserParameters(column.Name) == true)
                {
                    bool exists =
                        newColumns.Exists(delegate (XmlListColumn a)
                                {
                                    return a.Name == column.Name;
                                });
                    if (exists == false)
                    {
                        newColumns.Add(column);
                    }
                }
            }


            // 新規カラムの追加
            int index = 0;
            foreach (XmlListColumn column in defaultListColumns.ListColumn)
            {
                bool exists =
                    newColumns.Exists(delegate (XmlListColumn a)
                            {
                                return a.Name == column.Name;
                            });

                if (exists == false)
                {
                    newColumns.Insert(index, column);
                }
                ++index;
            }

            settingListColumns.ListColumn = newColumns.ToArray();
        }

        private bool ValidateProjectConfiguration()
        {
            if (null == ProjectService.ProjectDocument) { return true; }

            if (!_projectConfiguration.DocumentViews.ContainsKey(
                                    ProjectService.ProjectDocument.Resource.Key))
            {
                return false;
            }

            foreach (SoundSetDocument document in ProjectService.SoundSetDocuments)
            {
                if (!_projectConfiguration.DocumentViews.ContainsKey(document.Resource.Key))
                {
                    return false;
                }
            }

            foreach (SoundSetBankBase bank in ProjectService.SoundSetBanks)
            {
                if (!_projectConfiguration.DocumentViews.ContainsKey(bank.FilePath))
                {
                    return false;
                }
            }

            return true;
        }

        private void InitializeProjectViewConfiguration()
        {
            SoundProjectDocument document = ProjectService.ProjectDocument;
            if (null == document) { return; }

            bool loadDefault = true;

            if (_projectConfiguration.DocumentViews.ContainsKey(document.Resource.Key))
            {
                loadDefault = !ProjectConfiguration.DocumentViews[document.Resource.Key].Validate();
            }

            if (loadDefault)
            {
                DefaultProjectViewConfiguration(document.Resource.Key);
            }
        }

        private void DefaultProjectViewConfiguration(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            XmlSoundProjectDocumentView xmlView =
                                        ProjectConfiguration.LoadSoundProjectDocumentViewDefault();

            xmlView.FilePath = filePath;

            ProjectConfiguration.DocumentViews[filePath] = xmlView;
        }

        private void InitializeSoundSetViewConfiguration(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            bool loadDefault = true;

            if (_projectConfiguration.DocumentViews.ContainsKey(filePath))
            {
                loadDefault = !ProjectConfiguration.DocumentViews[filePath].Validate();
            }

            if (loadDefault)
            {
                DefaultSoundSetViewConfiguration(filePath);
            }
        }

        private void DefaultSoundSetViewConfiguration(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            IDictionary<string, XmlList> xmlDefaultLists =
                                        ProjectViewConfiguration.ListDefaults.ExportDictionary();

            XmlSoundSetDocumentView xmlNewConfiguration =
                                        ProjectConfiguration.LoadSoundSetDocumentViewDefault();

            List<XmlList> xmlLists = new List<XmlList>();

            foreach (string listName in XmlSoundSetDocumentView.ListNames)
            {
                xmlLists.Add(xmlDefaultLists[listName].DeepClone());
            }

            xmlNewConfiguration.Lists.List = xmlLists.ToArray();
            xmlNewConfiguration.FilePath = filePath;

            ProjectConfiguration.DocumentViews[filePath] = xmlNewConfiguration;
        }

        private void InitializeBankViewConfiguration(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            bool loadDefault = true;

            if (_projectConfiguration.DocumentViews.ContainsKey(filePath))
            {
                loadDefault = !ProjectConfiguration.DocumentViews[filePath].Validate();
            }

            if (loadDefault)
            {
                DefaultBankViewConfiguration(filePath);
            }
        }

        private void DefaultBankViewConfiguration(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            IDictionary<string, XmlList> xmlDefaultLists =
                                        ProjectViewConfiguration.ListDefaults.ExportDictionary();

            XmlBankDocumentView xmlNewConfiguration =
                                        ProjectConfiguration.LoadBankDocumentViewDefault();

            List<XmlList> xmlLists = new List<XmlList>();

            foreach (string listName in XmlBankDocumentView.ListNames)
            {
                xmlLists.Add(xmlDefaultLists[listName].DeepClone());
            }

            xmlNewConfiguration.Lists.List = xmlLists.ToArray();
            xmlNewConfiguration.FilePath = filePath;

            ProjectConfiguration.DocumentViews[filePath] = xmlNewConfiguration;
        }

        /// <summary>
        /// コマンドを初期化します。
        /// </summary>
        private void InitializeCommands()
        {
            InitializeCommands(FileCommands.Commands);
            InitializeCommands(EditCommands.Commands);
            InitializeCommands(ViewCommands.Commands);
            InitializeCommands(ProjectCommands.Commands);
            InitializeCommands(PreviewCommands.Commands);
            InitializeCommands(ToolCommands.Commands);
            InitializeCommands(WindowCommands.Commands);
            InitializeCommands(HelpCommands.Commands);
            InitializeCommands(TabCommands.Commands);
            InitializeCommands(SoundListCommands.Commands);
            InitializeCommands(InstrumentListCommands.Commands);
            InitializeCommands(PercussionListCommands.Commands);
            InitializeCommands(ProjectTreeCommands.Commands);
            InitializeCommands(this.ExtendedCommands);
        }

        /// <summary>
        /// コマンドを初期化します。
        /// </summary>
        private void InitializeCommands(IEnumerable<Command> commands)
        {
            if (null == commands) { throw new ArgumentNullException("commands"); }

            foreach (Command command in commands)
            {
                CommandService.AddCommand(command);
            }
        }

        /// <summary>
        /// コマンドキーを処理します。
        /// </summary>
        /// <param name="keyData">キーデータ。</param>
        /// <returns>キーメッセージを送信しない場合は true、する場合は false。</returns>
        private bool ProcessCommandKey(int keyData)
        {
            if (_commandKeyProcessSemaphore > 0) { return false; }

            switch (keyData & Mask_KeyCode)
            {
                case 0: /* None */
                case Win32.VK.VK_CONTROL:
                case Win32.VK.VK_SHIFT:
                case Win32.VK.VK_MENU:
                    return false;

                case Win32.VK.VK_LWIN:
                case Win32.VK.VK_RWIN:
                    _commandKeyProcessor.Reset();
                    return false;
            }

            KeyStroke keyStroke = new KeyStroke(keyData & Mask_KeyCode,
                                                 keyData & Mask_Modifiers);

            KeyStroke firstKey = _commandKeyProcessor.FirstKey;
            Command command = _commandKeyProcessor.InputKey(keyStroke);

            if (null != command)
            {

                if (!QueryCommandStatus(command).IsEnabled())
                {
                    return false;
                }

                if (ExecuteCommandInternal(command) == false)
                {
                    return false;
                }

                OnCommandKeyProcessed(
                    new CommandKeyProcessEventArgs(firstKey, keyStroke, command));

                return true;

            }

            // コマンドが見つからなかった。
            if (KeyStroke.NullKeyStroke != firstKey)
            {

                OnCommandKeyProcessed(
                    new CommandKeyProcessEventArgs(firstKey, keyStroke, null));

                return true;

            }
            else
            {

                if (KeyStroke.NullKeyStroke == _commandKeyProcessor.FirstKey) { return false; }

                OnCommandKeyProcessing(
                    new CommandKeyProcessEventArgs(keyStroke, KeyStroke.NullKeyStroke, null));

            }

            return false;
        }

        /// <summary>
        /// コンバートが可能になるまで待機します。
        /// </summary>
        /// <returns>コンバートを開始しても良い場合に trueを返します。</returns>
        private bool WaitUntilCanConvert()
        {
            if (this.ConvertService.IsConverting == true)
            {
                ConvertWaitingDialog dialog = new ConvertWaitingDialog(this.ConvertService);
                if (dialog.ShowDialog() != DialogResult.OK)
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// コンバート可能かどうかを調べます。
        /// </summary>
        /// <param name="command">コンバートコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        private CommandStatus QueryStatusExecuteConvert(Command command)
        {
            return (null != ProjectService.ProjectDocument) ?
                GetConvertCommandStatus() :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// コンバートを実行します。
        /// </summary>
        /// <param name="command">コンバートコマンド。</param>
        /// <returns>実行した場合 true、キャンセルされた、または実行しなかった場合 false。</returns>
        private bool ExecuteConvert(Command command)
        {
            if (WaitUntilCanConvert() == false)
            {
                return false;
            }

            ExecuteCommand(ToolCommands.StopAllSounds);
            if (ExecuteCommand(FileCommands.SaveAll) == false)
            {
                return false;
            }

            string installDirectory = string.Empty;

            if (command.Parameters.ContainsParameter(CommandParameterNames.InstallDirectory))
            {
                installDirectory = command.Parameters.GetParameter(
                                        CommandParameterNames.InstallDirectory) as string;
            }

            this.ConvertService.ParallelConversionCountMax =
                this.AppConfiguration.Options.Application.General.ParallelConversionCountMax;

            // TODO : 有効なプラットフォーム指定する
            this.ConvertService.Convert(
                this.ProjectService,
                SoundProjectConvertSettings.CreateForConvert(string.Empty, installDirectory));
            return true;
        }

        /// <summary>
        /// 再コンバート可能かどうかを調べます。
        /// </summary>
        /// <param name="command">コンバートコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        private CommandStatus QueryStatusExecuteReconvert(Command command)
        {
            return (null != ProjectService.ProjectDocument) ?
                GetConvertCommandStatus() :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// 再コンバートを実行します。
        /// </summary>
        /// <param name="command">コンバートコマンド。</param>
        /// <returns>実行した場合 true、キャンセルされた、または実行しなかった場合 false。</returns>
        private bool ExecuteReconvert(Command command)
        {
            if (WaitUntilCanConvert() == false)
            {
                return false;
            }

            ExecuteCommand(ToolCommands.StopAllSounds);
            if (ExecuteCommand(FileCommands.SaveAll) == false)
            {
                return false;
            }

            string installDirectory = string.Empty;

            if (command.Parameters.ContainsParameter(CommandParameterNames.InstallDirectory))
            {
                installDirectory = command.Parameters.GetParameter(
                                        CommandParameterNames.InstallDirectory) as string;
            }

            this.ConvertService.ParallelConversionCountMax =
                this.AppConfiguration.Options.Application.General.ParallelConversionCountMax;

            // TODO : 有効なプラットフォーム指定する
            this.ConvertService.Convert(
                this.ProjectService,
                SoundProjectConvertSettings.CreateForReconvert(string.Empty, installDirectory));
            return true;
        }

        /// <summary>
        /// 指定アイテムを再コンバート可能かどうかを調べます。
        /// </summary>
        /// <param name="command">コンバートコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        private CommandStatus QueryStatusExecuteSelectReconvert(Command command)
        {
            return (null != ProjectService.ProjectDocument) ?
                GetConvertCommandStatus() :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// 指定アイテムを再コンバートを実行します。
        /// </summary>
        /// <param name="command">コンバートコマンド。</param>
        /// <returns>実行した場合 true、キャンセルされた、または実行しなかった場合 false。</returns>
        private bool ExecuteSelectReconvert(Command command)
        {
            if (WaitUntilCanConvert() == false)
            {
                return false;
            }

            ExecuteCommand(ToolCommands.StopAllSounds);
            if (ExecuteCommand(FileCommands.SaveAll) == false)
            {
                return false;
            }

            string installDirectory = string.Empty;

            if (command.Parameters.ContainsParameter(CommandParameterNames.InstallDirectory))
            {
                installDirectory = command.Parameters.GetParameter(
                                        CommandParameterNames.InstallDirectory) as string;
            }

            this.ConvertService.ParallelConversionCountMax =
                this.AppConfiguration.Options.Application.General.ParallelConversionCountMax;

            SoundProjectConfiguration config = FormsApplication.Instance.ProjectConfiguration;

            SelectReconvertDialog dialog = new SelectReconvertDialog();
            dialog.CheckStreamSound = config.Convert.SelectedReconvert.StreamSound;
            dialog.CheckWaveSoundSet = config.Convert.SelectedReconvert.WaveSoundSet;
            dialog.CheckSequenceSound = config.Convert.SelectedReconvert.SequenceSound;
            dialog.CheckBank = config.Convert.SelectedReconvert.Bank;
            DialogResult result = dialog.ShowDialog();
            if (result == DialogResult.OK)
            {
                config.Convert.SelectedReconvert.StreamSound = dialog.CheckStreamSound;
                config.Convert.SelectedReconvert.WaveSoundSet = dialog.CheckWaveSoundSet;
                config.Convert.SelectedReconvert.SequenceSound = dialog.CheckSequenceSound;
                config.Convert.SelectedReconvert.Bank = dialog.CheckBank;

                SoundProjectReconvertFilter filter = SoundProjectReconvertFilter.None;
                if (dialog.CheckStreamSound == true)
                {
                    filter |= SoundProjectReconvertFilter.StreamSound;
                }
                if (dialog.CheckWaveSoundSet == true)
                {
                    filter |= SoundProjectReconvertFilter.WaveSoundSet;
                }
                if (dialog.CheckSequenceSound == true)
                {
                    filter |= SoundProjectReconvertFilter.SequenceSound;
                }
                if (dialog.CheckBank == true)
                {
                    filter |= SoundProjectReconvertFilter.Bank;
                }

                // TODO : 有効なプラットフォームを指定する
                this.ConvertService.Convert(
                    this.ProjectService,
                    SoundProjectConvertSettings.CreateForReconvert(string.Empty, installDirectory, filter));

                return true;
            }

            return false;
        }

        /// <summary>
        /// コマンド入力の第１キーが変更されると発生します。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnCommandInputFirstKeyChanged(object sender, EventArgs e)
        {
            OnCommandKeyProcessing(new CommandKeyProcessEventArgs(
                        _commandKeyProcessor.FirstKey, KeyStroke.NullKeyStroke, null));
        }

        private void OnKeyboardHook(object sender, Win32.WindowsHookEventArgs e)
        {
            if (Win32.HC.HC_ACTION != e.Code) { return; }

            // KeyUp の場合は処理しない。
            if (0 != ((int)e.LParam & 0x80000000)) { return; }

            // 修飾キー単独の場合は処理しない。
            switch ((int)e.WParam)
            {
                case Win32.VK.VK_CONTROL:
                case Win32.VK.VK_SHIFT:
                case Win32.VK.VK_MENU:
                    return;
            }

            int keyData = (int)e.WParam;

            if (0 != (Win32.User32.GetKeyState(Win32.VK.VK_CONTROL) & 0x80))
            {
                keyData |= Key_Control;
            }
            if (0 != (Win32.User32.GetKeyState(Win32.VK.VK_SHIFT) & 0x80))
            {
                keyData |= Key_Shift;
            }
            if (0 != (Win32.User32.GetKeyState(Win32.VK.VK_MENU) & 0x80))
            {
                keyData |= Key_Alt;
            }

            e.SendKeyMessage = !ProcessCommandKey(keyData);
        }

        ///--------------------------------
        /// <summary>
        /// プロジェクトドキュメントを開いたら発生します。
        /// </summary>
        private void OnProjectDocumentOpened(object sender, SoundDocumentResourceEventArgs e)
        {
            this.fileWatcherService.Add(e.Resource.Key, OnFileChanged);
        }

        ///--------------------------------
        /// <summary>
        /// プロジェクトドキュメントを閉じると発生します。
        /// </summary>
        private void OnProjectDocumentClosed(object sender, SoundDocumentResourceEventArgs e)
        {
            this.fileWatcherService.Remove(e.Resource.Key, OnFileChanged);
        }

        ///--------------------------------
        /// <summary>
        /// サウンドセットドキュメントを開いたら発生します。
        /// </summary>
        private void OnSoundSetDocumentOpened(object sender, SoundDocumentResourceEventArgs e)
        {
            this.fileWatcherService.Add(e.Resource.Key, OnFileChanged);
        }

        ///--------------------------------
        /// <summary>
        /// サウンドセットドキュメントを閉じると発生します。
        /// </summary>
        private void OnSoundSetDocumentClosed(object sender, SoundDocumentResourceEventArgs e)
        {
            this.fileWatcherService.Remove(e.Resource.Key, OnFileChanged);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnFileChanged(string filePath)
        {
            if (this.FileChanged != null)
            {
                this.FileChanged(filePath);
            }
        }
    }
}
