﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Windows.Forms
{
    using System;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.Drawing;
    using System.Windows.Forms;
    using NintendoWare.SoundMaker.Preview;
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Resources;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    public partial class PreviewPlayerParameterPanel : DockingPage, IToolWindowPage
    {
        public const string PageName = "PreviewPlayerParameter";

        #region ** 列挙子

        private enum StartOffsetTimeUnit
        {
            Sec,
            MSec
        }

        #endregion

        #region ** コンボボックスアイテム

        private class BiquadTypeItem
        {
            private string _text = string.Empty;
            private BiquadType _value = BiquadType.None;

            public BiquadTypeItem(string text, BiquadType value)
            {
                Debug.Assert(null != text);

                _text = text;
                _value = value;
            }

            #region ** プロパティ

            public BiquadType Value
            {
                get { return _value; }
            }

            #endregion

            #region ** メソッド

            public override string ToString()
            {
                return (null != _text) ? _text : string.Empty;
            }

            #endregion
        }

        private class BiquadTypeItemCollection : KeyedCollection<BiquadType, BiquadTypeItem>
        {
            protected override BiquadType GetKeyForItem(BiquadTypeItem item)
            {
                return item.Value;
            }
        }

        #endregion

        private class TrackEditor
        {
            private NumericEdit numericEdit;
            private CheckBox muteCheckBox;
            private CheckBox soloCheckBox;
            private CheckBox coupleCheckBox;

            public TrackEditor(NumericEdit numericEdit,
                               CheckBox muteCheckBox,
                               CheckBox soloCheckBox,
                               CheckBox coupleCheckBox)
            {
                this.numericEdit = numericEdit;
                this.muteCheckBox = muteCheckBox;
                this.soloCheckBox = soloCheckBox;
                this.coupleCheckBox = coupleCheckBox;
            }

            public float Volume
            {
                get
                {
                    return (float)(this.numericEdit.Value);
                }
                set
                {
                    this.numericEdit.Value = Convert.ToDecimal(value);
                }
            }

            public bool Mute
            {
                get
                {
                    return this.muteCheckBox.Checked;
                }
                set
                {
                    this.muteCheckBox.Checked = value;
                }
            }

            public bool Solo
            {
                get
                {
                    return this.soloCheckBox.Checked;
                }
                set
                {
                    this.soloCheckBox.Checked = value;
                }
            }

            public bool Couple
            {
                get
                {
                    return this.coupleCheckBox.Checked;
                }
                set
                {
                    this.coupleCheckBox.Checked = value;
                }
            }

            public bool MuteEnabled
            {
                get
                {
                    return this.muteCheckBox.Enabled;
                }
                set
                {
                    this.muteCheckBox.Enabled = value;
                }
            }
        }

        #region ** パラメータ

        private PreviewPlayer _previewPlayer;
        private StartOffsetTimeUnit _startOffsetTimeUnit = StartOffsetTimeUnit.MSec;
        private bool _updateParameterLocked = false;
        private BiquadTypeItemCollection _biquadTypeItems = new BiquadTypeItemCollection();
        private TrackEditor[] _trackEditors = null;
        private bool _trackVolumeChangedLocked = false;

        #endregion

        public PreviewPlayerParameterPanel()
            : base(PageName)
        {
            InitializeComponent();

            InitializeTracks();

            // テキストとアイコンをロードする
            Text = FrameworkResources.MessageResource.ToolWindowName_PreviewPlayerParameter;
            Image = FrameworkResources.ImageResource.
                BitmapIconPreviewPlayerParamWindow.MakeNewTransparent(UIServiceBase.TransparentColor);

            DockVisible = false;

            // 初期サイズを設定する
            FloatingBounds = new Rectangle(-1, -1,
                                            Width + (SystemInformation.FrameBorderSize.Width * 2),
                                            Height + SystemInformation.CaptionHeight + SystemInformation.FrameBorderSize.Height);

            // BiquadType を初期化する
            _biquadTypeItems.Add(new BiquadTypeItem(MessageResource.Label_BiquadType_None, BiquadType.None));
            _biquadTypeItems.Add(new BiquadTypeItem(MessageResource.Label_BiquadType_INHERIT, BiquadType.INHERIT));
            _biquadTypeItems.Add(new BiquadTypeItem(MessageResource.Label_BiquadType_LPF, BiquadType.LPF));
            _biquadTypeItems.Add(new BiquadTypeItem(MessageResource.Label_BiquadType_HPF, BiquadType.HPF));
            _biquadTypeItems.Add(new BiquadTypeItem(MessageResource.Label_BiquadType_BPF512, BiquadType.BPF512));
            _biquadTypeItems.Add(new BiquadTypeItem(MessageResource.Label_BiquadType_BPF1024, BiquadType.BPF1024));
            _biquadTypeItems.Add(new BiquadTypeItem(MessageResource.Label_BiquadType_BPF2048, BiquadType.BPF2048));
#if false
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR0,    BiquadType.USR0 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR1,    BiquadType.USR1 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR2,    BiquadType.USR2 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR3,    BiquadType.USR3 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR4,    BiquadType.USR4 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR5,    BiquadType.USR5 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR6,    BiquadType.USR6 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR7,    BiquadType.USR7 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR8,    BiquadType.USR8 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR9,    BiquadType.USR9 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR10,    BiquadType.USR10 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR11,    BiquadType.USR11 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR12,    BiquadType.USR12 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR13,    BiquadType.USR13 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR14,    BiquadType.USR14 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR15,    BiquadType.USR15 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR16,    BiquadType.USR16 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR17,    BiquadType.USR17 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR18,    BiquadType.USR18 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR19,    BiquadType.USR19 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR20,    BiquadType.USR20 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR21,    BiquadType.USR21 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR22,    BiquadType.USR22 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR23,    BiquadType.USR23 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR24,    BiquadType.USR24 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR25,    BiquadType.USR25 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR26,    BiquadType.USR26 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR27,    BiquadType.USR27 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR28,    BiquadType.USR28 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR29,    BiquadType.USR29 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR30,    BiquadType.USR30 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR31,    BiquadType.USR31 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR32,    BiquadType.USR32 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR33,    BiquadType.USR33 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR34,    BiquadType.USR34 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR35,    BiquadType.USR35 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR36,    BiquadType.USR36 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR37,    BiquadType.USR37 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR38,    BiquadType.USR38 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR39,    BiquadType.USR39 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR40,    BiquadType.USR40 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR41,    BiquadType.USR41 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR42,    BiquadType.USR42 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR43,    BiquadType.USR43 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR44,    BiquadType.USR44 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR45,    BiquadType.USR45 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR46,    BiquadType.USR46 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR47,    BiquadType.USR47 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR48,    BiquadType.USR48 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR49,    BiquadType.USR49 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR50,    BiquadType.USR50 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR51,    BiquadType.USR51 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR52,    BiquadType.USR52 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR53,    BiquadType.USR53 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR54,    BiquadType.USR54 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR55,    BiquadType.USR55 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR56,    BiquadType.USR56 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR57,    BiquadType.USR57 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR58,    BiquadType.USR58 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR59,    BiquadType.USR59 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR60,    BiquadType.USR60 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR61,    BiquadType.USR61 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR62,    BiquadType.USR62 ) );
            _biquadTypeItems.Add(new BiquadTypeItem( MessageResource.Label_BiquadType_USR63,    BiquadType.USR63 ) );
#endif
            foreach (BiquadTypeItem item in _biquadTypeItems)
            {
                _biquadTypeCombo.Items.Add(item);
            }

            _biquadTypeCombo.SelectedIndex = 0;

            // フェードタブを削除します。
            _categoryTab.TabPages.Remove(_pageFade);
        }

        #region ** プロパティ

        /// <summary>
        ///
        /// </summary>
        protected PreviewPlayerManager PreviewPlayerManager
        {
            get { return PreviewManager.Instance.PreviewPlayerManager; }
        }

        #endregion

        #region ** イベントハンドラ

        /// <summary>
        ///
        /// </summary>
        private void OnViewerSynchronousChanged(object sender, EventArgs e)
        {
            SynchronizeAll();
        }

        /// <summary>
        /// プレビュープレイヤーの選択が変更されると発生します。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnPreviewPlayerSelectionChanged(object sender, EventArgs e)
        {
            // シーケンス変数の更新ターゲットを設定する
            SetTargetPreviewPlayer(PreviewPlayerManager.SelectedPreviewPlayer);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnParameterEditValueChanged(object sender, ValueChangedEventArgs e)
        {
            UpdateParameter();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnParameterCheckedChanged(object sender, EventArgs e)
        {
            UpdateParameter();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnBiquadTypeChanged(object sender, EventArgs e)
        {
            UpdateParameter();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnStartOffsetTimeButtonClicked(object sender, EventArgs e)
        {
            if (_startOffsetTimeUnit == StartOffsetTimeUnit.Sec)
            {
                SetStartOffsetTimeUnit(StartOffsetTimeUnit.MSec);
            }
            else
            {
                SetStartOffsetTimeUnit(StartOffsetTimeUnit.Sec);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnResetClicked(object sender, EventArgs e)
        {
            Button target = sender as Button;
            Debug.Assert(null != target, "unexpected error");

            switch (target.Tag as string)
            {
                case "volume":
                    _volumeEdit.Value = Convert.ToDecimal(PreviewPlayer.Parameters.VolumeDefault);
                    break;

                case "pitch":
                    _pitchEdit.Value = Convert.ToDecimal(PreviewPlayer.Parameters.PitchDefault);
                    break;

                case "pan":
                    _panEdit.Value = Convert.ToDecimal(PreviewPlayer.Parameters.PanDefault);
                    break;

                case "surroundPan":
                    _surroundPanEdit.Value = Convert.ToDecimal(PreviewPlayer.Parameters.SurroundPanDefault);
                    break;

                case "mainPan":
                    mainPanEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.MainPanDefault);
                    break;

                case "mainSurroundPan":
                    mainSurroundPanEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.MainSurroundPanDefault);
                    break;

                case "drcPan":
                    _drcPanEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.DrcPanDefault);
                    break;

                case "drcSurroundPan":
                    drcSurroundPanEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.DrcSurroundPanDefault);
                    break;

                case "lpfFrequency":
                    _lpfFrequencyEdit.Value = Convert.ToDecimal(PreviewPlayer.Parameters.LpfFrequencyDefault);
                    break;

                case "biquadType":
                    _biquadTypeCombo.SelectedItem = _biquadTypeItems[PreviewPlayer.Parameters.BiquadTypeDefault];
                    break;

                case "biquadValue":
                    _biquadValueEdit.Value = Convert.ToDecimal(PreviewPlayer.Parameters.BiquadValueDefault);
                    break;

                case "startOffsetTime":
                    _startOffsetTimeEdit.Value = PreviewPlayer.Parameters.StartOffsetTimeDefault;
                    break;

                case "mainOut":
                    _mainOutEnabledCheck.Checked = PreviewPlayer.Parameters.MainOutEnabledDefault;
                    _mainVolumeEdit.Value = Convert.ToDecimal(PreviewPlayer.Parameters.MainOutVolumeDefault);
                    break;

                case "drcOut":
                    drcOutEnabledCheck.Checked = PreviewPlayer.Parameters.DrcOutEnabledDefault;
                    drcVolumeEdit.Value = Convert.ToDecimal(PreviewPlayer.Parameters.DrcOutVolumeDefault);
                    break;

                case "remote1Out":
                    _remote1OutEnabledCheck.Checked = PreviewPlayer.Parameters.Remote0OutEnabledDefault;
                    _remote1VolumeEdit.Value = Convert.ToDecimal(PreviewPlayer.Parameters.Remote0OutVolumeDefault);
                    break;

                case "remote2Out":
                    _remote2OutEnabledCheck.Checked = PreviewPlayer.Parameters.Remote1OutEnabledDefault;
                    _remote2VolumeEdit.Value = Convert.ToDecimal(PreviewPlayer.Parameters.Remote1OutVolumeDefault);
                    break;

                case "remote3Out":
                    _remote3OutEnabledCheck.Checked = PreviewPlayer.Parameters.Remote2OutEnabledDefault;
                    _remote3VolumeEdit.Value = Convert.ToDecimal(PreviewPlayer.Parameters.Remote2OutVolumeDefault);
                    break;

                case "remote4Out":
                    _remote4OutEnabledCheck.Checked = PreviewPlayer.Parameters.Remote3OutEnabledDefault;
                    _remote4VolumeEdit.Value = Convert.ToDecimal(PreviewPlayer.Parameters.Remote3OutVolumeDefault);
                    break;

                case "mainSend":
                    _mainSendEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.MainSendDefault);
                    break;

                case "effectAuxASend":
                    _effectAuxASendEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.EffectAuxASendDefault);
                    break;

                case "effectAuxBSend":
                    _effectAuxBSendEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.EffectAuxBSendDefault);
                    break;

                case "effectAuxCSend":
                    _effectAuxCSendEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.EffectAuxCSendDefault);
                    break;

                case "mainMainSend":
                    mainMainSendEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.MainSendDefault);
                    break;

                case "mainEffectAuxASend":
                    mainEffectAuxASendEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.EffectAuxASendDefault);
                    break;

                case "mainEffectAuxBSend":
                    mainEffectAuxBSendEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.EffectAuxBSendDefault);
                    break;

                case "mainEffectAuxCSend":
                    mainEffectAuxCSendEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.EffectAuxCSendDefault);
                    break;

                case "drcMainSend":
                    drcMainSendEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.MainSendDefault);
                    break;

                case "drcEffectAuxASend":
                    drcEffectAuxASendEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.EffectAuxASendDefault);
                    break;

                case "drcEffectAuxBSend":
                    drcEffectAuxBSendEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.EffectAuxBSendDefault);
                    break;

                case "drcEffectAuxCSend":
                    drcEffectAuxCSendEdit.Value = Convert.ToDecimal
                        (PreviewPlayer.Parameters.EffectAuxCSendDefault);
                    break;

                case "fadeInFrames":
                    _fadeInFramesEdit.Value = PreviewPlayer.Parameters.FadeInFramesDefault;
                    break;

                case "fadeOutFrames":
                    _fadeOutFramesEdit.Value = PreviewPlayer.Parameters.FadeOutFramesDefault;
                    break;

                case "pauseFadeFrames":
                    _pauseFadeFramesEdit.Value = PreviewPlayer.Parameters.PauseFadeFramesDefault;
                    break;
            }
        }

        #endregion

        #region ** イベントハンドラのオーバーライド

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            // イベントハンドラを登録する
            PreviewPlayerManager.PreviewPlayerSelectChanged += OnPreviewPlayerSelectionChanged;
            PreviewPlayerManager.ViewerSynchronousChanged += OnViewerSynchronousChanged;

            // 対象を更新する
            if (null != PreviewPlayerManager.SelectedPreviewPlayer)
            {
                SetTargetPreviewPlayer(PreviewPlayerManager.SelectedPreviewPlayer);
            }
        }

        protected override void OnHandleDestroyed(EventArgs e)
        {
            // イベントハンドラの登録を解除する
            PreviewPlayerManager.PreviewPlayerSelectChanged -= OnPreviewPlayerSelectionChanged;
            PreviewPlayerManager.ViewerSynchronousChanged -= OnViewerSynchronousChanged;

            base.OnHandleDestroyed(e);
        }

        protected override void OnActiveChanged(EventArgs e)
        {
            if (!Active) { return; }

            // 対象を更新する
            if (null != PreviewPlayerManager && null != PreviewPlayerManager.SelectedPreviewPlayer)
            {
                SetTargetPreviewPlayer(PreviewPlayerManager.SelectedPreviewPlayer);
            }

            base.OnActiveChanged(e);
        }

        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);

            SynchronizeAll();
        }

        protected override void OnResize(EventArgs e)
        {
            SuspendLayout();

            base.OnResize(e);

            if (null != _categoryTab)
            {
                _categoryTab.Size = new Size(ClientRectangle.Width - 6, ClientRectangle.Height - 31);
            }

            ResumeLayout();
        }

        #endregion

        #region ** メソッド

        private void InitializeTracks()
        {
            _trackEditors = new TrackEditor[]
                {
                    new TrackEditor(track00_NumericEdit, track00_M_CheckBox, track00_S_CheckBox, track00_C_CheckBox),
                    new TrackEditor(track01_NumericEdit, track01_M_CheckBox, track01_S_CheckBox, track01_C_CheckBox),
                    new TrackEditor(track02_NumericEdit, track02_M_CheckBox, track02_S_CheckBox, track02_C_CheckBox),
                    new TrackEditor(track03_NumericEdit, track03_M_CheckBox, track03_S_CheckBox, track03_C_CheckBox),
                    new TrackEditor(track04_NumericEdit, track04_M_CheckBox, track04_S_CheckBox, track04_C_CheckBox),
                    new TrackEditor(track05_NumericEdit, track05_M_CheckBox, track05_S_CheckBox, track05_C_CheckBox),
                    new TrackEditor(track06_NumericEdit, track06_M_CheckBox, track06_S_CheckBox, track06_C_CheckBox),
                    new TrackEditor(track07_NumericEdit, track07_M_CheckBox, track07_S_CheckBox, track07_C_CheckBox),
                    new TrackEditor(track08_NumericEdit, track08_M_CheckBox, track08_S_CheckBox, track08_C_CheckBox),
                    new TrackEditor(track09_NumericEdit, track09_M_CheckBox, track09_S_CheckBox, track09_C_CheckBox),
                    new TrackEditor(track10_NumericEdit, track10_M_CheckBox, track10_S_CheckBox, track10_C_CheckBox),
                    new TrackEditor(track11_NumericEdit, track11_M_CheckBox, track11_S_CheckBox, track11_C_CheckBox),
                    new TrackEditor(track12_NumericEdit, track12_M_CheckBox, track12_S_CheckBox, track12_C_CheckBox),
                    new TrackEditor(track13_NumericEdit, track13_M_CheckBox, track13_S_CheckBox, track13_C_CheckBox),
                    new TrackEditor(track14_NumericEdit, track14_M_CheckBox, track14_S_CheckBox, track14_C_CheckBox),
                    new TrackEditor(track15_NumericEdit, track15_M_CheckBox, track15_S_CheckBox, track15_C_CheckBox),
                };
        }

        private void SynchronizeAll()
        {
            // 表示中の場合だけ同期する
            if (Visible)
            {
                PreviewPlayerManager.SetAllPreviewPlayerParameters();
            }

            if (null == _viewerNotConnectedError) { return; }

            if (!PreviewPlayerManager.IsSynchronizedViewer)
            {
                _viewerNotConnectedError.SetIconAlignment(this, ErrorIconAlignment.TopLeft);
                _viewerNotConnectedError.SetIconPadding(this, -_viewerNotConnectedError.Icon.Width);
                _viewerNotConnectedError.SetError(this, Resources.MessageResource.Message_ThisControlIsViewerOnly);
            }
            else
            {
                _viewerNotConnectedError.Clear();
            }
        }

        private void SynchronizeThis()
        {
            // 表示中の場合だけ同期する
            if (Visible)
            {
                _previewPlayer.SetParameter();
            }

            if (null == _viewerNotConnectedError) { return; }

            if (!PreviewPlayerManager.IsSynchronizedViewer)
            {
                _viewerNotConnectedError.SetIconAlignment(this, ErrorIconAlignment.TopLeft);
                _viewerNotConnectedError.SetIconPadding(this, -_viewerNotConnectedError.Icon.Width);
                _viewerNotConnectedError.SetError(this, Resources.MessageResource.Message_ThisControlIsViewerOnly);
            }
            else
            {
                _viewerNotConnectedError.Clear();
            }
        }

        private void SetTargetPreviewPlayer(PreviewPlayer target)
        {
            if (target == _previewPlayer) { return; }

            _previewPlayer = null;
            RestoreParameter(target);

            // 対象プレビュープレイヤーを変更する
            _previewPlayer = target;

            if (null == _previewPlayer)
            {
                InvalidateEdit();
                _labelTarget.Text = "";
                return;
            }

            // ラベルを更新する
            _labelTarget.Text = string.Format(Resources.MessageResource.Label_PreviewPlayerParameter, target.Index);
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateParameter()
        {
            if (null == _previewPlayer) { return; }
            if (_updateParameterLocked) { return; }

            _previewPlayer.Paramters.LockEvent();

            //
            _previewPlayer.Paramters.Volume = (float)_volumeEdit.Value;
            _previewPlayer.Paramters.Pitch = (float)_pitchEdit.Value;

            _previewPlayer.Paramters.Pan = (float)_panEdit.Value;
            _previewPlayer.Paramters.SurroundPan = (float)_surroundPanEdit.Value;

            _previewPlayer.Paramters.DrcPan = (float)_drcPanEdit.Value;
            _previewPlayer.Paramters.DrcSurroundPan = (float)drcSurroundPanEdit.Value;

            _previewPlayer.Paramters.MainPan = (float)mainPanEdit.Value;
            _previewPlayer.Paramters.MainSurroundPan = (float)mainSurroundPanEdit.Value;

            _previewPlayer.Paramters.LpfFrequency = (float)_lpfFrequencyEdit.Value;
            _previewPlayer.Paramters.BiquadType = (_biquadTypeCombo.SelectedItem as BiquadTypeItem).Value;
            _previewPlayer.Paramters.BiquadValue = (float)_biquadValueEdit.Value;

            //
            if (_startOffsetTimeUnit == StartOffsetTimeUnit.Sec)
            {
                _previewPlayer.Paramters.StartOffsetTime =
                    decimal.ToUInt32(_startOffsetTimeEdit.Value * 1000);
            }
            else
            {
                _previewPlayer.Paramters.StartOffsetTime =
                    decimal.ToUInt32(_startOffsetTimeEdit.Value);
            }

            //
            _previewPlayer.Paramters.MainOut.Enabled = _mainOutEnabledCheck.Checked;
            _previewPlayer.Paramters.DrcOut.Enabled = drcOutEnabledCheck.Checked;
            _previewPlayer.Paramters.RemoteOut[0].Enabled = _remote1OutEnabledCheck.Checked;
            _previewPlayer.Paramters.RemoteOut[1].Enabled = _remote2OutEnabledCheck.Checked;
            _previewPlayer.Paramters.RemoteOut[2].Enabled = _remote3OutEnabledCheck.Checked;
            _previewPlayer.Paramters.RemoteOut[3].Enabled = _remote4OutEnabledCheck.Checked;

            _previewPlayer.Paramters.MainOut.Volume = (float)_mainVolumeEdit.Value;
            _previewPlayer.Paramters.DrcOut.Volume = (float)drcVolumeEdit.Value;
            _previewPlayer.Paramters.RemoteOut[0].Volume = (float)_remote1VolumeEdit.Value;
            _previewPlayer.Paramters.RemoteOut[1].Volume = (float)_remote2VolumeEdit.Value;
            _previewPlayer.Paramters.RemoteOut[2].Volume = (float)_remote3VolumeEdit.Value;
            _previewPlayer.Paramters.RemoteOut[3].Volume = (float)_remote4VolumeEdit.Value;

            _previewPlayer.Paramters.MainSend = (float)_mainSendEdit.Value;
            _previewPlayer.Paramters.EffectAuxASend = (float)_effectAuxASendEdit.Value;
            _previewPlayer.Paramters.EffectAuxBSend = (float)_effectAuxBSendEdit.Value;
            _previewPlayer.Paramters.EffectAuxCSend = (float)_effectAuxCSendEdit.Value;

            _previewPlayer.Paramters.MainMainSend = (float)mainMainSendEdit.Value;
            _previewPlayer.Paramters.MainEffectAuxASend = (float)mainEffectAuxASendEdit.Value;
            _previewPlayer.Paramters.MainEffectAuxBSend = (float)mainEffectAuxBSendEdit.Value;
            _previewPlayer.Paramters.MainEffectAuxCSend = (float)mainEffectAuxCSendEdit.Value;

            _previewPlayer.Paramters.DrcMainSend = (float)drcMainSendEdit.Value;
            _previewPlayer.Paramters.DrcEffectAuxASend = (float)drcEffectAuxASendEdit.Value;
            _previewPlayer.Paramters.DrcEffectAuxBSend = (float)drcEffectAuxBSendEdit.Value;
            _previewPlayer.Paramters.DrcEffectAuxCSend = (float)drcEffectAuxCSendEdit.Value;

            _previewPlayer.Paramters.FadeInFrames = decimal.ToUInt32(_fadeInFramesEdit.Value);
            _previewPlayer.Paramters.FadeOutFrames = decimal.ToUInt32(_fadeOutFramesEdit.Value);
            _previewPlayer.Paramters.PauseFadeFrames = decimal.ToUInt32(_pauseFadeFramesEdit.Value);

            bool soloCheck = false;
            for (int i = 0; i < PreviewPlayer.Parameters.TracksCountDefault; i++)
            {
                _previewPlayer.Paramters.TracksParameter[i].Volume = _trackEditors[i].Volume;
                _previewPlayer.Paramters.TracksParameter[i].Mute = _trackEditors[i].Mute;
                _previewPlayer.Paramters.TracksParameter[i].Solo = _trackEditors[i].Solo;
                _previewPlayer.Paramters.TracksParameter[i].Couple = _trackEditors[i].Couple;
                soloCheck |= _trackEditors[i].Solo;
            }
            for (int i = 0; i < PreviewPlayer.Parameters.TracksCountDefault; i++)
            {
                if (soloCheck == true)
                {
                    _previewPlayer.Paramters.TracksParameter[i].Enabled = _trackEditors[i].Solo;
                }
                else
                {
                    _previewPlayer.Paramters.TracksParameter[i].Enabled = _trackEditors[i].Mute == false;
                }
            }

            _previewPlayer.Paramters.UnlockEvent();

            SynchronizeThis();
        }

        private void RestoreParameter(PreviewPlayer target)
        {
            if (null == target) { throw new ArgumentNullException("target"); }

            _volumeEdit.Value = Convert.ToDecimal(target.Paramters.Volume);
            _pitchEdit.Value = Convert.ToDecimal(target.Paramters.Pitch);

            _panEdit.Value = Convert.ToDecimal(target.Paramters.Pan);
            _surroundPanEdit.Value = Convert.ToDecimal(target.Paramters.SurroundPan);

            mainPanEdit.Value = Convert.ToDecimal(target.Paramters.MainPan);
            mainSurroundPanEdit.Value = Convert.ToDecimal(target.Paramters.MainSurroundPan);

            _drcPanEdit.Value = Convert.ToDecimal(target.Paramters.DrcPan);
            drcSurroundPanEdit.Value = Convert.ToDecimal(target.Paramters.DrcSurroundPan);

            _lpfFrequencyEdit.Value = Convert.ToDecimal(target.Paramters.LpfFrequency);
            _biquadTypeCombo.SelectedItem = _biquadTypeItems[target.Paramters.BiquadType];
            _biquadValueEdit.Value = Convert.ToDecimal(target.Paramters.BiquadValue);

            if (_startOffsetTimeUnit == StartOffsetTimeUnit.Sec)
            {
                _startOffsetTimeEdit.Value = decimal.Truncate(target.Paramters.StartOffsetTime / 1000);
                target.Paramters.StartOffsetTime = decimal.ToUInt32(_startOffsetTimeEdit.Value * 1000);
            }
            else
            {
                _startOffsetTimeEdit.Value = target.Paramters.StartOffsetTime;
            }

            _mainOutEnabledCheck.Checked = target.Paramters.MainOut.Enabled;
            drcOutEnabledCheck.Checked = target.Paramters.DrcOut.Enabled;
            _remote1OutEnabledCheck.Checked = target.Paramters.RemoteOut[0].Enabled;
            _remote2OutEnabledCheck.Checked = target.Paramters.RemoteOut[1].Enabled;
            _remote3OutEnabledCheck.Checked = target.Paramters.RemoteOut[2].Enabled;
            _remote4OutEnabledCheck.Checked = target.Paramters.RemoteOut[3].Enabled;

            _mainVolumeEdit.Value = Convert.ToDecimal(target.Paramters.MainOut.Volume);
            drcVolumeEdit.Value = Convert.ToDecimal(target.Paramters.DrcOut.Volume);
            _remote1VolumeEdit.Value = Convert.ToDecimal(target.Paramters.RemoteOut[0].Volume);
            _remote2VolumeEdit.Value = Convert.ToDecimal(target.Paramters.RemoteOut[1].Volume);
            _remote3VolumeEdit.Value = Convert.ToDecimal(target.Paramters.RemoteOut[2].Volume);
            _remote4VolumeEdit.Value = Convert.ToDecimal(target.Paramters.RemoteOut[3].Volume);

            _mainSendEdit.Value = Convert.ToDecimal(target.Paramters.MainSend);
            _effectAuxASendEdit.Value = Convert.ToDecimal(target.Paramters.EffectAuxASend);
            _effectAuxBSendEdit.Value = Convert.ToDecimal(target.Paramters.EffectAuxBSend);
            _effectAuxCSendEdit.Value = Convert.ToDecimal(target.Paramters.EffectAuxCSend);

            mainMainSendEdit.Value = Convert.ToDecimal(target.Paramters.MainMainSend);
            mainEffectAuxASendEdit.Value = Convert.ToDecimal(target.Paramters.MainEffectAuxASend);
            mainEffectAuxBSendEdit.Value = Convert.ToDecimal(target.Paramters.MainEffectAuxBSend);
            mainEffectAuxCSendEdit.Value = Convert.ToDecimal(target.Paramters.MainEffectAuxCSend);

            drcMainSendEdit.Value = Convert.ToDecimal(target.Paramters.DrcMainSend);
            drcEffectAuxASendEdit.Value = Convert.ToDecimal(target.Paramters.DrcEffectAuxASend);
            drcEffectAuxBSendEdit.Value = Convert.ToDecimal(target.Paramters.DrcEffectAuxBSend);
            drcEffectAuxCSendEdit.Value = Convert.ToDecimal(target.Paramters.DrcEffectAuxCSend);

            _fadeInFramesEdit.Value = Convert.ToDecimal(target.Paramters.FadeInFrames);
            _fadeOutFramesEdit.Value = Convert.ToDecimal(target.Paramters.FadeOutFrames);
            _pauseFadeFramesEdit.Value = Convert.ToDecimal(target.Paramters.PauseFadeFrames);

            for (int i = 0; i < PreviewPlayer.Parameters.TracksCountDefault; i++)
            {
                _trackEditors[i].Volume = target.Paramters.TracksParameter[i].Volume;
                _trackEditors[i].Mute = target.Paramters.TracksParameter[i].Mute;
                _trackEditors[i].Solo = target.Paramters.TracksParameter[i].Solo;
                _trackEditors[i].Couple = target.Paramters.TracksParameter[i].Couple;
            }

            if (!Enabled)
            {
                Enabled = true;
            }
        }

        private void InvalidateEdit()
        {
            Enabled = false;
        }

        private void SetStartOffsetTimeUnit(StartOffsetTimeUnit unit)
        {
            if (_startOffsetTimeUnit == unit) { return; }


            _startOffsetTimeUnit = unit;

            if (unit == StartOffsetTimeUnit.Sec)
            {

                decimal newValue = decimal.Truncate(_previewPlayer.Paramters.StartOffsetTime / 1000);

                // 単位変換にて実質の値が変わらない場合は、イベントを無視する
                _updateParameterLocked = (newValue * 1000 == _previewPlayer.Paramters.StartOffsetTime);

                // 小数点以下は切り捨て
                _startOffsetTimeEdit.Value = newValue;

                _startOffsetTimeEdit.Maximum /= 1000;
                _startOffsetTimeEdit.TickFrequency /= 1000;
                _startOffsetTimeEdit.LargeChange /= 100;
                _startOffsetTimeEdit.SmallChange /= 100;

                _startOffsetTimeUnitButton.Text = "sec";

            }
            else
            {

                // 単位変換にて実質の値が変わらない場合は、イベントを無視する
                _updateParameterLocked = (_startOffsetTimeEdit.Value * 1000 == _previewPlayer.Paramters.StartOffsetTime);

                _startOffsetTimeEdit.TickFrequency *= 1000;
                _startOffsetTimeEdit.LargeChange *= 100;
                _startOffsetTimeEdit.SmallChange *= 100;
                _startOffsetTimeEdit.Maximum *= 1000;
                _startOffsetTimeEdit.Value = _previewPlayer.Paramters.StartOffsetTime;

                _startOffsetTimeUnitButton.Text = "msec";

            }

            _updateParameterLocked = false;
        }


        private void UpdateTrackTab()
        {
            bool soloCheck = false;

            foreach (TrackEditor editor in _trackEditors)
            {
                soloCheck |= editor.Solo;
            }
            foreach (TrackEditor editor in _trackEditors)
            {
                editor.MuteEnabled = (soloCheck == false);
            }
        }

        #endregion

        private void OnTrackResetClicked(object sender, EventArgs e)
        {
            if (sender is Button)
            {
                Button resetButton = sender as Button;
                if (resetButton.Tag is string)
                {
                    int index = int.Parse(resetButton.Tag as string);
                    if (0 <= index && index < PreviewPlayer.Parameters.TracksCountDefault)
                    {
                        _trackEditors[index].Volume =
                            PreviewPlayer.Parameters.TrackVolumeDefault;
                    }
                }
            }
        }

        private void OnTrack_M_ResetClicked(object sender, EventArgs e)
        {
            foreach (TrackEditor editor in _trackEditors)
            {
                editor.Mute = false;
            }
        }

        private void OnTrack_S_ResetClicked(object sender, EventArgs e)
        {
            foreach (TrackEditor editor in _trackEditors)
            {
                editor.Solo = false;
            }
            UpdateTrackTab();
        }

        private void OnTrack_C_ResetClicked(object sender, EventArgs e)
        {
            foreach (TrackEditor editor in _trackEditors)
            {
                editor.Couple = false;
            }
        }

        private void OnTrack_M_SetClicked(object sender, EventArgs e)
        {
            foreach (TrackEditor editor in _trackEditors)
            {
                editor.Mute = !(editor.Mute);
            }
        }

        private void OnTrack_S_SetClicked(object sender, EventArgs e)
        {
            foreach (TrackEditor editor in _trackEditors)
            {
                editor.Solo = !(editor.Solo);
            }
            UpdateTrackTab();
        }

        private void OnTrack_C_SetClicked(object sender, EventArgs e)
        {
            foreach (TrackEditor editor in _trackEditors)
            {
                editor.Couple = !(editor.Couple);
            }
        }

        private void OnAllResetClicked(object sender, EventArgs e)
        {
            foreach (TrackEditor editor in _trackEditors)
            {
                editor.Volume = PreviewPlayer.Parameters.TrackVolumeDefault;
            }
        }

        private void OnTrack_S_CheckBoxClicked(object sender, EventArgs e)
        {
            UpdateTrackTab();
        }

        private void OnTrackVolumeChanged(object sender, ValueChangedEventArgs e)
        {
            if (_trackVolumeChangedLocked == true)
            {
                return;
            }

            try
            {
                _trackVolumeChangedLocked = true;

                if (sender is NumericEdit)
                {
                    NumericEdit numericEdit = sender as NumericEdit;
                    if (numericEdit.Tag is string)
                    {
                        int index = int.Parse(numericEdit.Tag as string);
                        if (0 <= index && index < PreviewPlayer.Parameters.TracksCountDefault)
                        {
                            if (_trackEditors[index].Couple == true)
                            {
                                foreach (TrackEditor editor in _trackEditors)
                                {
                                    if (editor.Couple == true)
                                    {
                                        editor.Volume = _trackEditors[index].Volume;
                                    }
                                }
                            }
                        }
                    }

                    UpdateParameter();
                }
            }
            finally
            {
                _trackVolumeChangedLocked = false;
            }
        }
    }
}
