﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Windows.Forms
{
    using System;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.Drawing;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Preview;
    using NintendoWare.ToolDevelopmentKit;
    using ComponentModel = System.ComponentModel;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;
    using Win32 = NintendoWare.SoundFoundation.Core.Win32;

    public partial class PreviewBankPanel :
        DockingPage, IToolWindowPage, ICommandTarget
    {
        public const string PageName = "PreviewBank";
        private const string ClipboardDataName = "NwSoundMaker";

        private CommandBindingCollection commandBindings = new CommandBindingCollection();

        private PreviewBankControlCollection _previewBanks = new PreviewBankControlCollection();
        private PreviewBankControl _selectedControl = null;

        public PreviewBankPanel()
            : base(PageName)
        {
            InitializeComponent();

            Text = FrameworkResources.MessageResource.ToolWindowName_PreviewBank;
            Image = FrameworkResources.ImageResource.
                    BitmapIconPreviewBankWindow.MakeNewTransparent(UIServiceBase.TransparentColor);

            // 初期サイズを設定する
            FloatingBounds = new Rectangle(-1, -1,
                                            Width + (SystemInformation.FrameBorderSize.Width * 2),
                                            Height + SystemInformation.CaptionHeight + SystemInformation.FrameBorderSize.Height);

            Initialize();
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 選択中のアイテムを取得または設定します。
        /// </summary>
        [ComponentModel.Category("Behavior")]
        [ComponentModel.Browsable(false)]
        [ComponentModel.ReadOnly(true)]
        public PreviewBankControl SelectedItem
        {
            get { return _selectedControl; }
        }

        /// <summary>
        ///
        /// </summary>
        [ComponentModel.Category("Behavior")]
        [ComponentModel.ReadOnly(true)]
        public uint ItemCount
        {
            get { return (uint)PreviewBankManager.PreviewBanks.Count; }
        }

        /// <summary>
        ///
        /// </summary>
        public uint SelectedPreviewBankIndex
        {
            get
            {
                return PreviewBankManager.SelectedPreviewBankIndex;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private PreviewBankManager PreviewBankManager
        {
            get { return PreviewManager.Instance.PreviewBankManager; }
        }

        private PreviewBank SelectedPreviewBank
        {
            get { return PreviewBankManager.SelectedPreviewBank; }
        }

        private NTabControl OwnerTabControl
        {
            get { return Parent as NTabControl; }
        }

        private SoundProjectService ProjectService
        {
            get { return FormsApplicationCommon.Instance.ProjectService; }
        }

        //-----------------------------------------------------------------

        public void Initialize()
        {
            Debug.Assert(null != PreviewBankManager, "unexpected Error");
            if (PreviewBankManager.PreviewBanks.Count == 0) { throw new Exception(); }

            PreviewBankManager.PreviewBankSelectChanged += OnPreviewBankSelectChanged;
            FormsApplication.Instance.ProjectService.Opened += (sender, e) => this.UpdateItems();
            FormsApplication.Instance.ProjectService.ComponentsAdded += (sender, e) => this.UpdateItems();
            FormsApplication.Instance.ProjectService.ComponentsRemoved += (sender, e) => this.UpdateItems();

            SuspendLayout();

            // 新しいコントロールコレクションを作成する
            PreviewBankControlCollection oldCollection = DetachContainer();
            PreviewBankControlCollection newCollection = new PreviewBankControlCollection();

            int controlTop = 0;

            for (int index = 0; index < PreviewBankManager.PreviewBanks.Count; index++)
            {

                PreviewBankControl newPreviewBank;

                if (null != oldCollection && index < oldCollection.Count)
                {
                    newPreviewBank = oldCollection[index];
                }
                else
                {

                    newPreviewBank = new PreviewBankControl(PreviewBankManager.PreviewBanks[index]);

                    newPreviewBank.Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right;
                    newPreviewBank.AllowDrop = true;
                    newPreviewBank.Name = "PreviewBank" + index.ToString();
                    newPreviewBank.Location = new System.Drawing.Point(0, controlTop);
                    newPreviewBank.Size = new System.Drawing.Size(ClientRectangle.Width, newPreviewBank.Height);
                    newPreviewBank.TabStop = false;
                    newPreviewBank.TabIndex = index;

                    newPreviewBank.SelectedChanged += OnPreviewBankSelectedChanged;
                    newPreviewBank.DialogKeyProcessed += OnDialogKeyProcessed;

                }

                Controls.Add(newPreviewBank);
                newCollection.Add(newPreviewBank);

                controlTop += newPreviewBank.Height;

            }

            AttachContainer(newCollection);

            this.ResumeLayout();

            this.InitializeCommandBindings();
        }

        public PreviewBankControl GetItem(uint index)
        {
            if (index >= this._previewBanks.Count)
            {
                return null;
            }

            return this._previewBanks[(int)index];
        }

        public void SelectItem(uint targetIndex)
        {
            if (null != _selectedControl)
            {
                _selectedControl.Selected = false;
            }

            if (targetIndex < _previewBanks.Count)
            {
                _selectedControl = _previewBanks[(int)targetIndex];
                _selectedControl.Selected = true;
            }
        }

        public void UpdateItems()
        {
            if (!FormsApplication.Instance.ProjectService.IsOpened)
            {
                return;
            }

            foreach (PreviewBankControl control in _previewBanks)
            {
                control.UpdateItem();
            }
        }

        public void Attach(string itemName)
        {
            this.Attach(
                this.GetValidItem((itemName == null) ? string.Empty : itemName)
                );
        }

        public void Attach(SequenceSoundBase item)
        {
            if (item == null)
            {
                this.Clear();
                return;
            }

            int index = 0;

            foreach (SoundSetBankBase soundSetBank in item.SoundSetBanks)
            {
                if (index >= this._previewBanks.Count)
                {
                    break;
                }

                this._previewBanks[index].Item = soundSetBank;
                index++;
            }

            for (; index < this._previewBanks.Count; index++)
            {
                this._previewBanks[index].Item = null;
            }
        }

        /// <summary>
        /// 指定コマンドを処理するコマンドターゲットを検索します。
        /// </summary>
        /// <param name="command">処理するコマンド。</param>
        /// <returns>コマンドターゲット。</returns>
        ICommandTarget ICommandTarget.FindTarget(Command command)
        {
            if (!this.commandBindings.Contains(command.ID))
            {
                return null;
            }

            return this;
        }

        /// <summary>
        /// 指定コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        CommandStatus ICommandTarget.QueryStatus(Command command)
        {
            if (!this.commandBindings.Contains(command.ID))
            {
                return CommandStatus.Unsupported;
            }

            return this.commandBindings[command.ID].QueryStatus(command);
        }

        /// <summary>
        /// 指定コマンドを実行します。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        bool ICommandTarget.Execute(Command command)
        {
            if (!this.commandBindings.Contains(command.ID))
            {
                return false;
            }

            return this.commandBindings[command.ID].Execute(command);
        }

        //-----------------------------------------------------------------

        protected bool OnDialogKeyProcessed(object sender, KeyEventArgs e)
        {
            switch (e.KeyData)
            {
                case Keys.Up:
                    SelectPrevItem();
                    return true;

                case Keys.Down:
                    SelectNextItem();
                    return true;

                case Keys.Home:
                    SelectFirstItem();
                    return true;

                case Keys.End:
                    SelectLastItem();
                    return true;

                case Keys.Tab:
                    ProcessDialogKey(Keys.Tab);
                    return true;
            }

            return false;
        }

        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case Win32.WM.WM_SETFOCUS:
                    if (null != SelectedItem)
                    {
                        SelectedItem.Focus();
                    }
                    return;
            }

            base.WndProc(ref m);
        }

        protected override void OnWindowActiveChanged(EventArgs e)
        {
            base.OnWindowActiveChanged(e);

            if (this.SelectedItem != null)
            {
                this.SelectedItem.SetActive(this.WindowActive);
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// コマンドの関連付けを初期化します。
        /// </summary>
        private void InitializeCommandBindings()
        {
            this.commandBindings.Add(
                new CommandBinding(this, EditCommands.Delete.ID, QueryStatusDelete, ExecuteDelete));
        }

        private void AttachContainer(PreviewBankControlCollection newContainer)
        {
            Assertion.Argument.NotNull(newContainer);

            _previewBanks = newContainer;

            SelectItem(PreviewBankManager.SelectedPreviewBank.Index);
        }

        private PreviewBankControlCollection DetachContainer()
        {
            Assertion.Operation.True(_previewBanks != null);

            foreach (PreviewBankControl item in _previewBanks)
            {
                if (null == item) continue;
                Controls.Remove(item);
            }

            PreviewBankControlCollection oldContainer = _previewBanks;
            _previewBanks = new PreviewBankControlCollection();

            return oldContainer;
        }

        private SequenceSoundBase GetValidItem(string itemName)
        {
            Assertion.Argument.NotNull(itemName);

            if (!this.ProjectService.ComponentDictionary.Contains(itemName))
            {
                return null;
            }

            foreach (Component component in this.ProjectService.ComponentDictionary[itemName])
            {
                if (component is SequenceSoundBase)
                {
                    return component as SequenceSoundBase;
                }
            }

            return null;
        }

        private void Clear()
        {
            foreach (PreviewBankControl control in _previewBanks)
            {
                control.Item = null;
            }
        }

        private void SelectPrevItem()
        {
            if (ItemCount == 0) { return; }
            if (null == SelectedItem) { return; }
            if (SelectedItem.Index == 0) { return; }

            SelectItem(SelectedItem.Index - 1);
        }

        private void SelectNextItem()
        {
            if (ItemCount == 0) { return; }
            if (null == SelectedItem) { return; }
            if (SelectedItem.Index + 1 >= ItemCount) { return; }

            SelectItem(SelectedItem.Index + 1);
        }

        private void SelectFirstItem()
        {
            if (ItemCount == 0) { return; }
            SelectItem(0);
        }

        private void SelectLastItem()
        {
            if (ItemCount == 0) { return; }
            SelectItem(ItemCount - 1);
        }

        private void UpdateColors()
        {
            foreach (PreviewBankControl control in _previewBanks)
            {
                control.UpdateColors(this.WindowActive);
            }
        }

        private CommandStatus QueryStatusDelete(Command command)
        {
            if (null == this.SelectedItem) { return CommandStatus.SupportedAndVisible; }
            return (this.SelectedItem.Item != null) ?
                   CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        private bool ExecuteDelete(Command command)
        {
            this.SelectedItem.Item = null;
            return true;
        }

        //-----------------------------------------------------------------

        private void OnPreviewBankSelectChanged(object sender, EventArgs e)
        {
            SelectItem(PreviewBankManager.SelectedPreviewBankIndex);
        }

        private void OnPreviewBankFocusEnter(object sender, EventArgs e)
        {
            PreviewBankControl target = sender as PreviewBankControl;
            Assertion.Operation.True(null != target);

            // プレビュープレイヤーの選択を変更する
            PreviewBankManager.SelectedPreviewBankIndex = target.PreviewBank.Index;
        }

        private void OnPreviewBankSelectedChanged(object sender, EventArgs e)
        {
            PreviewBankControl target = sender as PreviewBankControl;
            Assertion.Operation.True(null != target);

            if (!target.Selected) { return; }

            // プレビュープレイヤーの選択を変更する
            PreviewBankManager.SelectedPreviewBankIndex = target.PreviewBank.Index;
        }
    }

    #region ** コレクションクラス

    public class PreviewBankControlCollection : Collection<PreviewBankControl> { }

    #endregion
}
