﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;

namespace NintendoWare.SoundMaker.Windows.Forms
{
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Projects;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMakerPlugin;

    public partial class BankPanelCommon : BankPanel
    {
        private IParameterValue[] encodingValues = ListCtrlCommon.EncodingValues;

        private NintendoWare.SoundMaker.Framework.Windows.Forms.ListPartTextDrawer textDrawer = null;
        private ListPartCommonRowHeaderDrawer rowHeaderDrawer = null;

        private readonly CommonListFileWatcher _listFileWatcher;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public BankPanelCommon(string filePath)
            : base(filePath)
        {
            this.textDrawer = new NintendoWare.SoundMaker.Framework.Windows.Forms.ListPartTextDrawer();
            this.rowHeaderDrawer = new ListPartCommonRowHeaderDrawer();
            this.ListDecorationEvaluator.AddDoShowErrorIconDelegate(ProjectParameterNames.WaveEncoding, this.DoShowErrorIconWaveEncoding);
            this.ListDecorationEvaluator.AddDoShowErrorIconDelegate(ProjectParameterNames.SampleRate, this.DoShowErrorIconSampleRate);

            CurrentListCtrl.PartDrawers.Replace(this.textDrawer);
            PercussionListPanel.PercussionList.PartDrawers.Replace(this.textDrawer);

            CurrentListCtrl.PartDrawers.Replace(this.rowHeaderDrawer);
            PercussionListPanel.PercussionList.PartDrawers.Replace(this.rowHeaderDrawer);

            CurrentListCtrl.ReplaceArgumentDelegates.Add(ProjectParameterNames.WaveEncoding, this.ReplaceArgumentWaveEncoding);
            PercussionListPanel.PercussionList.ReplaceArgumentDelegates.Add(ProjectParameterNames.WaveEncoding, this.ReplaceArgumentWaveEncoding);

            ListCtrlCommon.AddListColumnStyle(CurrentListCtrl);
            ListCtrlCommon.AddListColumnStyle(PercussionListPanel.PercussionList);

            InitializeComponent();

            SampleMapPanel.IsValidateWaveEncodingValueDelegate = this.IsValidateWaveEncodingValue;

            _listFileWatcher = new CommonListFileWatcher(this, this.ListControls);
            _listFileWatcher.FileChanged += (sender, args) => this.RedrawPanel();
        }

        /// <summary>
        ///
        /// </summary>
        private FormsApplicationCommon Application
        {
            get { return FormsApplicationCommon.Instance; }
        }

        private int SelectedKey
        {
            get
            {
                if (InstrumentList.ContainsFocus)
                {
                    if (0 == InstrumentList.SelectedItems.Length) { return -1; }
                    return SelectedKeyboardKey;
                }

                if (SampleMapPanel.ContainsFocus)
                {

                    VelocityRegion[] velocityRegions = SampleMapPanel.SelectedVelocityRegions;
                    if (1 != velocityRegions.Length) { return -1; }

                    return velocityRegions[0].OriginalKey;

                }

                if (PercussionListPanel.ContainsFocus)
                {

                    if (null == PercussionListPanel.ActiveListCtrl) { return -1; }

                    IListItem[] items = PercussionListPanel.ActiveListCtrl.SelectedItems;
                    if (1 != items.Length) { return -1; }

                    return (items[0] as PercussionListItem).Index;

                }

                return -1;
            }
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            _listFileWatcher.Start();
        }

        public override void OnClosed()
        {
            base.OnClosed();

            _listFileWatcher.Stop();
        }

        /// <summary>
        ///
        /// </summary>
        public override void Deactivated()
        {
            base.Deactivated();

            Stop();
        }

        /// <summary>
        ///
        /// </summary>
        protected override void InitializeCommandBindings()
        {
            base.InitializeCommandBindings();
        }

        /// <summary>
        ///
        /// </summary>
        protected override CommonListAdapter CreateListAdapter()
        {
            CommonListAdapter listAdapter = base.CreateListAdapter();
            return listAdapter;
        }

        /// <summary>
        ///
        /// </summary>
        protected override void ProgramChange(BankDocument bankDocument, Instrument instrument)
        {
            Application.ProgramChange(bankDocument, instrument);
        }

        protected override void NoteOn(int programNo, int key, int velocity)
        {
            Application.NoteOn(programNo, key, velocity);
        }

        protected override void NoteOff(int key)
        {
            Application.NoteOff(key);
        }

        protected override void ProgramChange(Instrument instrument)
        {
            Application.ProgramChange(BankDocument, instrument);
        }

        protected override void Play(int key, int velocity)
        {
            Application.NotePlayer.Play(key, velocity);
        }

        protected override void Stop()
        {
            Application.NotePlayer.Stop();
        }

        /// <summary>
        /// 圧縮形式でエラーアイコンの表示非表示判定メソッドです。
        /// </summary>
        private string DoShowErrorIconWaveEncoding(Component component)
        {
            if (component is Instrument == true) // VelocityRegion 全て調べます。
            {
                foreach (KeyRegion key in component.Children)
                {
                    foreach (VelocityRegion vel in key.Children)
                    {
                        if (this.Application.CurrentSoundMakerPlugin.IsValidateWaveEncodingValue(vel.Encoding) == false)
                        {
                            // 波形のエンコード形式が不正です。
                            return string.Format(Resources.MessageResource.ErrorMessage_InvalidWaveEncoding, vel.Encoding);
                        }
                    }
                }
            }
            else if (component is VelocityRegion == true)
            {
                if (this.Application.CurrentSoundMakerPlugin.IsValidateWaveEncodingValue((component as VelocityRegion).Encoding) == false)
                {
                    // 波形のエンコード形式が不正です。
                    return string.Format(Resources.MessageResource.ErrorMessage_InvalidWaveEncoding, (component as VelocityRegion).Encoding);
                }
            }

            return null;
        }

        /// <summary>
        /// サンプルレートのエラーアイコンの表示非表示判定メソッドです。
        /// </summary>
        private string DoShowErrorIconSampleRate(Component component)
        {
            if (component is Instrument == true &&
                (bool)component.Parameters[ProjectParameterNames.IsResampleEnabled].Value == true)
            {
                // VelocityRegion を抽出します。
                var velocityRegions = component.Children
                    .Cast<KeyRegion>()
                    .SelectMany(k => k.Children.Cast<VelocityRegion>());

                // 波形ファイルの情報がなければ更新します。
                velocityRegions
                    .Where(v => v.WaveFile == null)
                    .ToList()
                    .ForEach(v => v.UpdateWaveFile());

                // １つでもダウンサンプルではなくアップサンプルになっていたらエラーにします。
                if (velocityRegions
                    .Where(v => v.WaveFile != null)
                    .Any(v => v.WaveFile.SampleRate < (component as Instrument).SampleRate))
                {
                    // アップサンプルはサポートされません。
                    return Resources.MessageResource.ErrorMessage_UpsampleNotSupported;
                }
            }

            return null;
        }

        private IInplaceEditorArgument ReplaceArgumentWaveEncoding(ListColumnStyle style,
                                                                   IInplaceEditorArgument origArgs,
                                                                   ComponentListItem item)
        {
            ISoundMakerPlugin plugin = SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin;

            return new DropDownListInplaceEditor.Argument(encodingValues.Where(e => plugin.IsValidateWaveEncodingValue((WaveEncoding)e.Value)).ToArray());
        }

        private bool IsValidateWaveEncodingValue(WaveEncoding waveEncoding)
        {
            ISoundMakerPlugin plugin = SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin;

            return plugin.IsValidateWaveEncodingValue(waveEncoding);
        }
    }
}
