﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Windows.Forms;

namespace NintendoWare.SoundMaker.Windows.Forms
{
    public partial class SeqVariableSetControl : NUserControl
    {
        #region ** 定数

        private const uint _defaultVarCount = 16;
        private const uint _maxVarCount = 128;

        #endregion

        private uint _varCount = _defaultVarCount;
        private SeqVariableControlCollection _varControls = null;

        public SeqVariableSetControl()
        {
            InitializeComponent();

            Initialize();
        }

        #region ** プロパティ

        /// <summary>
        /// シーケンス変数の数を取得または設定します。
        /// </summary>
        [Category("Behavior")]
        [Description("シーケンス変数の数を取得または設定します。")]
        [DefaultValue(typeof(uint), "16")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public uint VariableCount
        {
            get { return _varCount; }
            set
            {
                if (value == _varCount) { return; }
                if (value == 0) { throw new ArgumentOutOfRangeException(); }
                if (value > _maxVarCount) { throw new ArgumentOutOfRangeException(); }

                Initialize(value);
            }
        }

        [ReadOnly(true)]
        public SeqVariableControlCollection VariableControls
        {
            get { return _varControls; }
        }

        #endregion

        #region ** イベント

        public event SeqVariableEventHandler CheckedChanged;
        public event SeqVariableValueChangedEventHandler EditValueChanged;

        #endregion

        #region ** イベントハンドラ

        private void OnEditCheckedChanged(object sender, EventArgs e)
        {
            if (null != CheckedChanged)
            {
                CheckedChanged(this, new SeqVariableEventArgs(sender as SeqVariableControl));
            }
        }

        private void OnEditValueChanged(object sender, ValueChangedEventArgs e)
        {
            if (null != EditValueChanged)
            {
                EditValueChanged(this, new SeqVariableValueChangedEventArgs(sender as SeqVariableControl, e));
            }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// シーケンス変数の数を変更して、コントロールを初期化します。
        /// </summary>
        /// <param name="variableCount">シーケンス変数の数</param>
        private void Initialize(uint variableCount)
        {
            _varCount = variableCount;
            Initialize();
        }

        /// <summary>
        /// コントロールを初期化します。
        /// </summary>
        private void Initialize()
        {
            if (null != _varControls && _varCount == _varControls.Count) { return; }

            Debug.Assert(0 < _varCount);
            Debug.Assert(_maxVarCount >= _varCount);

            SuspendLayout();

            // 作成済コントロールを全て解放する
            Uninitialize();


            // コントロールを作成する
            _varControls = new SeqVariableControlCollection();

            int controlPosY = 0;

            for (uint i = 0; i < _varCount; i++)
            {

                SeqVariableControl newControl = new SeqVariableControl();

                newControl.Index = i;
                newControl.Location = new Point(0, controlPosY);
                newControl.Size = new Size(Width, newControl.Height);
                newControl.Anchor = AnchorStyles.Left | AnchorStyles.Top | AnchorStyles.Right;
                newControl.TabIndex = (int)i;
                newControl.EditValue = -1;

                newControl.CheckedChanged += OnEditCheckedChanged;
                newControl.ValueChanged += OnEditValueChanged;

                Controls.Add(newControl);
                _varControls.Add(newControl);

                controlPosY += newControl.Height;

            }

            ResumeLayout(false);
        }

        private void Uninitialize()
        {
            if (null == _varControls) { return; }

            foreach (SeqVariableControl control in _varControls)
            {
                Controls.Remove(control);
                control.Dispose();
            }

            _varControls = null;
        }

        #endregion
    }

    #region ** イベント

    public delegate void SeqVariableEventHandler(object sender, SeqVariableEventArgs e);
    public delegate void SeqVariableValueChangedEventHandler(object sender, SeqVariableValueChangedEventArgs e);

    public class SeqVariableEventArgs : EventArgs
    {
        private SeqVariableControl _target;

        public SeqVariableEventArgs(SeqVariableControl target)
        {
            Debug.Assert(null != target);
            _target = target;
        }

        #region ** プロパティ

        public SeqVariableControl Control
        {
            get { return _target; }
        }

        #endregion
    }

    public class SeqVariableValueChangedEventArgs : ValueChangedEventArgs
    {
        private SeqVariableControl _target;

        public SeqVariableValueChangedEventArgs(SeqVariableControl target, ValueChangedEventArgs e)
            : base(e.Commited, e.Synchronized)
        {
            Debug.Assert(null != target);
            _target = target;
        }

        #region ** プロパティ

        public SeqVariableControl Control
        {
            get { return _target; }
        }

        #endregion
    }

    #endregion

    #region ** コレクションクラス

    public class SeqVariableControlCollection : Collection<SeqVariableControl> { }

    #endregion
}
