﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using NintendoWare.SoundFoundation.Windows.Forms;
using NintendoWare.SoundMaker.Preview;
using NW4R.ProtocolSound;
using Win32 = NintendoWare.SoundFoundation.Core.Win32;

namespace NintendoWare.SoundMaker.Windows.Forms
{
    public partial class EffectParameterPane : NUserControl
    {
        #region ** 内部クラス

        private class EffectItem
        {
            private string _name;
            private ToolSoundEffectType _effectType;
            private EffectParameterControl _parameterControl;

            public EffectItem(string name, ToolSoundEffectType effectType, EffectParameterControl control)
            {
                if (null == name) { throw new ArgumentNullException("name"); }

                _name = name;
                _effectType = effectType;
                _parameterControl = control;
            }

            #region ** プロパティ

            public string Name
            {
                get { return _name; }
            }

            public ToolSoundEffectType EffectType
            {
                get { return _effectType; }
            }

            public EffectParameterControl Control
            {
                get { return _parameterControl; }
            }

            #endregion

            #region ** メソッドのオーバーライド

            public override string ToString()
            {
                return Name;
            }

            #endregion
        }

        #endregion

        #region ** パラメータ

        private SortedList<ToolSoundEffectType, EffectItem> _effectItems = new SortedList<ToolSoundEffectType, EffectItem>();
        private EffectItem _selectedItem = null;
        private EffectParameterSet _parameters = null;

        // 状態
        private bool _systemPresetMenuVisibled = false;

        #endregion


        /// <summary>
        /// コンストラクタ
        /// </summary>
        public EffectParameterPane()
        {
            InitializeComponent();

            // パラメータコントロールを追加する
            AddEffectItem(new EffectItem(Resources.MessageResource.Label_EffectType_NoEffect,
                           ToolSoundEffectType.EffectType_NoEffect, new EffectNoEffectPrameterControl()));
            AddEffectItem(new EffectItem(Resources.MessageResource.Label_EffectType_ReverbHi,
                           ToolSoundEffectType.EffectType_ReverbHi, new EffectReverbHiPrameterControl()));
            AddEffectItem(new EffectItem(Resources.MessageResource.Label_EffectType_ReverbStd,
                           ToolSoundEffectType.EffectType_ReverbStd, new EffectReverbStdPrameterControl()));
            AddEffectItem(new EffectItem(Resources.MessageResource.Label_EffectType_Chorus,
                           ToolSoundEffectType.EffectType_Chorus, new EffectChorusPrameterControl()));
            AddEffectItem(new EffectItem(Resources.MessageResource.Label_EffectType_Delay,
                           ToolSoundEffectType.EffectType_Delay, new EffectDelayPrameterControl()));
            AddEffectItem(new EffectItem(Resources.MessageResource.Label_EffectType_MultiChannelReverb,
                           ToolSoundEffectType.EffectType_MultiChannelReverb, new EffectMultiChannelReverbParameterControl()));
            AddEffectItem(new EffectItem(Resources.MessageResource.Label_EffectType_MultiChannelChorus,
                           ToolSoundEffectType.EffectType_MultiChannelChorus, new EffectMultiChannelChorusParameterControl()));
            AddEffectItem(new EffectItem(Resources.MessageResource.Label_EffectType_MultiChannelDelay,
                           ToolSoundEffectType.EffectType_MultiChannelDelay, new EffectMultiChannelDelayParameterControl()));
        }

        #region ** プロパティ

        /// <summary>
        /// エフェクトパラメータを取得または設定します。
        /// </summary>
        [Browsable(false)]
        [ReadOnly(true)]
        public EffectParameterSet Parameters
        {
            get { return _parameters; }
            set { Attach(value); }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// エフェクトパラメータを関連付けます。
        /// </summary>
        /// <param name="newParameters">エフェクトパラメータ</param>
        private void Attach(EffectParameterSet newParameters)
        {
            Debug.Assert(null != newParameters);
            if (newParameters == _parameters) { return; }


            // イベントハンドラを差し替えて、パラメータを更新する
            if (null != _parameters)
            {
                _parameters.EffectParameterChanged -= OnEffectParameterChanged;
            }
            newParameters.EffectParameterChanged += OnEffectParameterChanged;

            _parameters = newParameters;


            EffectItem targetItem = _effectItems[newParameters.Selected.EffectType];
            Debug.Assert(null != targetItem);

            // 対象項目を選択する
            _effectTypeCombo.SelectedItem = targetItem;

            // アイテムに関連付けられたパラメータを更新する
            targetItem.Control.Parameter = newParameters.Select(targetItem.EffectType);
        }

        private void AddEffectItem(EffectItem newItem)
        {
            // エフェクトアイテムリストに追加する
            _effectItems.Add(newItem.EffectType, newItem);

            // コンボボックスにアイテムを追加
            _effectTypeCombo.Items.Add(newItem);

            // パラメータパネルにコントロールを追加する
            _parameterPanel.Controls.Add(newItem.Control);
            newItem.Control.Dock = DockStyle.Fill;

            newItem.Control.Hide();
        }

        private void SelectItem(EffectItem targetItem)
        {
            if (targetItem == _selectedItem) { return; }

            // パラメータコントロールの表示を切り替える
            SuspendLayout();

            // パラメータパネルの表示を切り替える
            targetItem.Control.Show();

            if (null != _selectedItem)
            {
                _selectedItem.Control.Hide();
            }

            _selectedItem = targetItem;

            ResumeLayout(false);
        }

        #endregion

        #region ** イベントハンドラ

        private void OnEffectParameterChanged(object sender, EventArgs e)
        {
            EffectParameter target = sender as EffectParameter;
            Debug.Assert(null != target);

            if (null != _selectedItem && target.EffectType == _selectedItem.EffectType) { return; }

            // エフェクトの種類を更新する
            // パラメータコントロールの表示は、イベントハンドラ内で切り替わる
            _effectTypeCombo.SelectedItem = _effectItems[target.EffectType];
        }

        /// <summary>
        /// エフェクトの種類が変更されたときに実行されます。
        /// </summary>
        /// <param name="sender">送信元オブジェクト</param>
        /// <param name="e">イベントパラメータ</param>
        private void OnEffectTypeChanged(object sender, EventArgs e)
        {
            ComboBox target = sender as ComboBox;
            Debug.Assert(null != target);

            if (null == target.SelectedItem) { return; }

            EffectItem targetItem = target.SelectedItem as EffectItem;
            Debug.Assert(null != targetItem);


            // アイテムに関連付けられたパラメータを更新する
            Debug.Assert(null != Parameters);
            targetItem.Control.Parameter = Parameters.Select(targetItem.EffectType);

            // パラメータコントロールの表示を切り替える
            SelectItem(targetItem);
        }

        private void OnSystemPresetButtonClicked(object sender, EventArgs e)
        {
            // メニューを表示中の場合は処理しない
            if (_systemPresetMenuVisibled) { return; }

            Button target = sender as Button;
            Debug.Assert(null != target);

            _systemPresetMenu.Show(target, new Point(0, target.Height));
            _systemPresetMenuVisibled = true;
        }

        private void OnSystemPresetButtonMouseActivating(object sender, EventArgs e)
        {
            // メニューの表示状態を更新する
            _systemPresetMenuVisibled = _systemPresetMenu.Visible;
        }

        private void OnSystemPresetMenuItemClicked(object sender, ToolStripItemClickedEventArgs e)
        {
            string key = e.ClickedItem.Tag as string;

            // プリセットからエフェクトの種類とパラメータをロードする
            ToolSoundEffectType type = EffectSystemPresetLoader.Instance.GetType(key);
            Parameters.Select(type);

            EffectSystemPresetLoader.Instance.Load(key, Parameters.Selected);
        }

        #endregion

        #region ** 専用コントロール

        /// <summary>
        /// MouseActivating イベントを処理するためのボタンクラス
        /// </summary>
        private class MenuPresetButton : Button
        {
            #region ** イベント

            public event EventHandler MouseActivating;

            #endregion

            #region ** イベントハンドラ

            public virtual void OnMouseActivating(EventArgs e)
            {
                if (null == MouseActivating) { return; }
                MouseActivating(this, e);
            }

            #endregion

            #region ** メソッドのオーバーライド

            protected override void WndProc(ref Message m)
            {
                switch (m.Msg)
                {
                    case Win32.WM.WM_MOUSEACTIVATE:
                        OnMouseActivating(new EventArgs());
                        base.WndProc(ref m);
                        return;
                }

                base.WndProc(ref m);
            }

            #endregion
        }

        #endregion
    }
}
