﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Windows.Forms
{
    /// <summary>
    /// プリセットを操作するためのコントロールです。
    /// </summary>
    public partial class PresetControl : UserControl
    {
        private string _defaultItemName = string.Empty;
        private int _itemMax = int.MaxValue;

        public PresetControl()
        {
            InitializeComponent();
        }

        /// <summary>
        /// プリセットアイテムがロードする際に発生します。
        /// </summary>
        public event EventHandler<PresetItemEventArgs> LoadItem;

        /// <summary>
        /// プリセットアイテムを保存する際に発生します。
        /// </summary>
        public event EventHandler<PresetItemEventArgs> SaveItem;

        /// <summary>
        /// プリセットアイテムを削除する際に発生します。
        /// </summary>
        public event EventHandler<PresetItemEventArgs> DeleteItem;

        /// <summary>
        /// デフォルトのプリセット名を取得または設定します。
        /// </summary>
        public string DefaultPresetName
        {
            get { return _defaultItemName; }
            set
            {
                if (null == value) { throw new ArgumentNullException("value"); }
                _defaultItemName = value;
            }
        }

        /// <summary>
        /// アイテム数の上限を取得または設定します。
        /// </summary>
        public int ItemMax
        {
            get { return _itemMax; }
            set
            {
                _itemMax = (0 > value) ? int.MaxValue : value;
            }
        }

        /// <summary>
        /// 選択されたアイテムの名前を取得します。
        /// </summary>
        public string SelectedItemName
        {
            get { return _itemComboBox.Text; }
        }

        /// <summary>
        /// プリセット数が上限に達しているか確認します。
        /// </summary>
        private bool IsItemFull
        {
            get { return (_itemComboBox.Items.Count == _itemMax); }
        }

        /// <summary>
        /// アイテムを初期化します。
        /// </summary>
        public void InitializeItems(object[] items)
        {
            if (null == items) { throw new ArgumentNullException("items"); }

            _itemComboBox.Items.Clear();
            _itemComboBox.Items.AddRange(items);

            UpdateToolState();
        }

        /// <summary>
        /// アイテム名をリセットします。
        /// </summary>
        public void ResetItemText()
        {
            _itemComboBox.Text = CreateNewItemName();
            UpdateToolState();
        }

        /// <summary>
        /// アイテム名をクリアします。
        /// </summary>
        public void ClearItemText()
        {
            _itemComboBox.Text = string.Empty;
            _itemComboBox.SelectedIndex = -1;

            UpdateToolState();
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);
            ResetItemText();
        }

        /// <summary>
        /// プリセットアイテムがロードされると発生します。
        /// </summary>
        /// <param name="e"></param>
        protected void OnLoadItem(PresetItemEventArgs e)
        {
            if (null != LoadItem)
            {
                LoadItem(this, e);
            }
        }

        /// <summary>
        /// プリセットアイテムを保存する際に発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected void OnSaveItem(PresetItemEventArgs e)
        {
            if (null != SaveItem)
            {
                SaveItem(this, e);
            }
        }

        /// <summary>
        /// プリセットアイテムを削除する際に発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected void OnDeleteItem(PresetItemEventArgs e)
        {
            if (null != DeleteItem)
            {
                DeleteItem(this, e);
            }
        }

        private bool ContainsName(string itemName)
        {
            if (null == itemName) { throw new ArgumentNullException("itemName"); }
            return (-1 != _itemComboBox.FindStringExact(itemName));
        }

        /// <summary>
        /// 新しいアイテム名を作成します。
        /// </summary>
        private string CreateNewItemName()
        {
            if (0 == DefaultPresetName.Length) { return string.Empty; }

            string name = DefaultPresetName;
            if (!ContainsName(name)) { return name; }

            int count = 1;

            while (true)
            {

                name = DefaultPresetName + count.ToString();
                if (!ContainsName(name)) { break; }

                count++;

            }

            return name;
        }

        private void UpdateToolState()
        {
            _deleteButton.Enabled = ContainsName(_itemComboBox.Text);

            if (IsItemFull)
            {

                _itemComboBox.DropDownStyle = ComboBoxStyle.DropDownList;
                _saveButton.Enabled = false;

            }
            else
            {

                _itemComboBox.DropDownStyle = ComboBoxStyle.DropDown;
                _saveButton.Enabled = (_itemComboBox.Text.Length > 0);

            }
        }


        private void OnItemComboTextChanged(object sender, EventArgs e)
        {
            UpdateToolState();
        }

        private void OnItemComboSelectionChanged(object sender, EventArgs e)
        {
            if (null == _itemComboBox.SelectedItem) { return; }

            OnLoadItem(new PresetItemEventArgs(SelectedItemName));

            UpdateToolState();
        }

        private void OnSaveClick(object sender, EventArgs e)
        {
            if (0 == _itemComboBox.Text.Length) { return; }
            OnSaveItem(new PresetItemEventArgs(SelectedItemName));
        }

        private void OnDeleteClick(object sender, EventArgs e)
        {
            if (0 == _itemComboBox.Text.Length) { return; }
            OnDeleteItem(new PresetItemEventArgs(SelectedItemName));
        }
    }
}
