﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Runtime.Serialization.Formatters.Binary;

namespace NintendoWare.SoundMaker.Windows.Forms
{
    using Nintendo.AudioToolkit.DataModels;
    using Nintendo.AudioToolkit.Presenters;
    using Nintendo.AudioToolkit.DomainModels;
    using Nintendo.AudioToolkit.Operations;
    using Nintendo.AudioToolkit.Views;
    using Nintendo.AudioToolkit.Windows.Controls;
    using Nintendo.Foundation.Audio;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Resources;

    public partial class WaveSoundEditWindow : Form
    {
        private SoundProjectServiceCommon soundProjectService = null;
        private WaveSound waveSound = null;

        private OperationHistory operationHistory = null;
        private ClipboardServiceWSD clipboardServiceWSD = null;

        private WaveSoundPresenter waveSoundPresenter = null;
        private WaveSoundPropertyPresenter waveSoundPropertyPresenter = null;

        public WaveSoundEditWindow(SoundProjectServiceCommon soundProjectService)
        {
            InitializeComponent();

            this.soundProjectService = soundProjectService;

            this.clipboardServiceWSD = new ClipboardServiceWSD();
            this.clipboardServiceWSD.AddConverter(new ClipConverter());
            this.clipboardServiceWSD.AddConverter(new TrackConverter());

            this.waveSoundPresenter = new WaveSoundPresenter();
            this.waveSoundPresenter.Initialize();
            this.waveSoundPresenter.BeginTransaction += (s, e) => this.operationHistory.BeginTransaction();
            this.waveSoundPresenter.EndTransaction += (s, e) => this.operationHistory.EndTransaction();
            this.waveSoundPresenter.CancelTransaction += (s, e) => this.operationHistory.CancelTransaction();
            this.waveSoundPresenter.DropFiles += OnDropFiles;
            this.waveSoundPresenter.RequestDuplicate += OnRequestDuplicate;
            this.waveSoundPresenter.SelectionChanged += OnSelectionChanged;
            this.elementHostLeft.Child = this.waveSoundPresenter.View;

            this.waveSoundPropertyPresenter = new WaveSoundPropertyPresenter();
            this.waveSoundPropertyPresenter.Initialize();
            this.elementHostRight.Child = this.waveSoundPropertyPresenter.View;

            this.waveSoundPresenter.SetModel(null);
            this.waveSoundPropertyPresenter.SetModel(null);
            this.waveSound = null;

            this.UpdateMenu();
        }

        public WaveSound WaveSound
        {
            get
            {
                return this.waveSound;
            }
            set
            {
                if (value == null)
                {
                    this.waveSound = null;
                    this.waveSoundPresenter.SetModel(null);
                    this.waveSoundPropertyPresenter.SetModel(null);
                    this.SetOperationHistory(null);
                }
                else
                {
                    this.waveSound = value;
                    this.waveSoundPresenter.SetModel(value.WaveSoundResource);
                    SoundSet soundSet = this.GetSoundSet(value);
                    OperationHistory operationHistory = this.soundProjectService.SoundSetDocuments.Where(d => d.SoundSet == soundSet).Select(d => d.OperationHistory).FirstOrDefault();
                    this.SetOperationHistory(operationHistory);
                }

                this.UpdateTitle();
            }
        }

        private SoundSet GetSoundSet(Component component)
        {
            if (component == null || component is SoundSet)
            {
                return component as SoundSet;
            }
            else
            {
                return GetSoundSet(component.Parent);
            }
        }

        private void SetOperationHistory(OperationHistory history)
        {
            if (this.operationHistory != null)
            {
                this.operationHistory.ItemsChanged -= this.UpdateMenu;
            }

            this.operationHistory = history;

            if (this.operationHistory != null)
            {
                this.operationHistory.ItemsChanged += this.UpdateMenu;
            }
        }

        private void UpdateTitle()
        {
            string name = string.Empty;

            if (this.waveSound != null)
            {
                name = this.waveSound.Name;
            }

            this.Text = name;
            if (string.IsNullOrEmpty(name) == false)
            {
                this.Text += " - ";
            }
            this.Text += MessageResource.Label_WaveSoundEditWindowTitle;
        }

        private void UpdateMenu(object sender, EventArgs e)
        {
            this.UpdateMenu();
        }

        private void UpdateMenu()
        {
            this.menuItem_Undo.Enabled = false;
            this.menuItem_Redo.Enabled = false;
            this.menuItem_Cut.Enabled = false;
            this.menuItem_Copy.Enabled = false;
            this.menuItem_Paste.Enabled = false;
            this.menuItem_Delete.Enabled = false;
            this.menuItem_SelectAll.Enabled = false;

            this.cmenuItem_Cut.Enabled = false;
            this.cmenuItem_Copy.Enabled = false;
            this.cmenuItem_Paste.Enabled = false;
            this.cmenuItem_Delete.Enabled = false;

            if (this.waveSound == null || this.waveSound.WaveSoundResource == null)
            {
                return;
            }

            if (this.operationHistory != null)
            {
                this.menuItem_Undo.Enabled = this.operationHistory.CanUndo();
                this.menuItem_Redo.Enabled = this.operationHistory.CanRedo();

                if (0 < this.waveSoundPresenter.SelectedClips.Count())
                {
                    this.menuItem_Cut.Enabled = true;
                    this.menuItem_Delete.Enabled = true;
                    this.menuItem_Copy.Enabled = true;
                    this.cmenuItem_Cut.Enabled = true;
                    this.cmenuItem_Delete.Enabled = true;
                    this.cmenuItem_Copy.Enabled = true;
                }

                this.menuItem_Paste.Enabled = true;
                this.menuItem_SelectAll.Enabled = true;
                this.cmenuItem_Paste.Enabled = true;
            }
        }

        private void OnDropFiles(object sender, WaveSoundPresenter.DropFileEventArgs e)
        {
            this.AddClip(e.TrackNo, e.Position.X, (int)e.Position.Y, e.FilePaths[0]);
        }

        private void OnRequestDuplicate(object sender, WaveSoundPresenter.RequestDuplicateEventArgs e)
        {
            if (e.Instance is WaveSoundClipResource)
            {
                var source = e.Instance as WaveSoundClipResource;
                var target = new WaveSoundClipResourceData();
                target.IsEnabled = source.IsEnabled;
                target.Path = source.Path;
                target.Position = source.Position;
                target.StartOffset = source.StartOffset;
                target.Duration = source.Duration;
                target.Volume = source.Volume;
                target.Pitch = source.Pitch;
                target.Pan = source.Pan;
                e.NewInstance = new WaveSoundClipResource(target);
            }
        }

        private void OnSelectionChanged(object sender, EventArgs e)
        {
            if (this.waveSoundPresenter.SelectedClips.Count() == 1)
            {
                var clip = waveSoundPresenter.SelectedClips.First();
                this.waveSoundPropertyPresenter.SetModel(clip);
            }
            else
            {
                this.waveSoundPropertyPresenter.SetModel(null);
            }

            this.UpdateMenu();
        }

        /// <summary>
        /// 元に戻す
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_Undo(object sender, EventArgs e)
        {
            if (this.operationHistory != null)
            {
                this.operationHistory.Undo();
                this.UpdateMenu();
            }
        }

        /// <summary>
        /// やり直し
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_Redo(object sender, EventArgs e)
        {
            if (this.operationHistory != null)
            {
                this.operationHistory.Redo();
                this.UpdateMenu();
            }
        }

        /// <summary>
        /// 切り取り
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_Cut(object sender, EventArgs e)
        {
            this.OnClick_Copy(sender, e);
            this.OnClick_Delete(sender, e);
        }

        /// <summary>
        /// コピー
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_Copy(object sender, EventArgs e)
        {
            // トラックとクリップの両方が選択されている時にはクリップのコピーを優先しています。
            if (this.waveSoundPresenter.SelectedClips.Count() > 0)
            {
                this.clipboardServiceWSD.To(this.waveSoundPresenter.SelectedClips);
            }
            else if (this.waveSoundPresenter.SelectedTracks.Count() > 0)
            {
                this.clipboardServiceWSD.To(this.waveSoundPresenter.SelectedTracks);
            }
        }

        /// <summary>
        /// 貼り付け
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_Paste(object sender, EventArgs e)
        {
            if (this.clipboardServiceWSD.Contains == false)
            {
                return;
            }

            foreach (var model in this.clipboardServiceWSD.From())
            {
                if (model is WaveSoundClipResource)
                {
                    this.waveSoundPresenter.AddClipToSelectedTrack(model as WaveSoundClipResource);
                }
                else if (model is WaveSoundTrackResource)
                {
                    this.waveSoundPresenter.AddTrack(model as WaveSoundTrackResource);
                }
            }
        }

        /// <summary>
        /// 削除
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_Delete(object sender, EventArgs e)
        {
            if (this.operationHistory != null)
            {
                this.operationHistory.BeginTransaction();
                this.waveSoundPresenter.RemoveSelectedClips();
                this.operationHistory.EndTransaction();
            }
        }

        /// <summary>
        /// すべて選択
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_SelectAll(object sender, EventArgs e)
        {
            this.waveSoundPresenter.SelectClipAll();
        }

        /// <summary>
        /// トラックの追加
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_AddTrack(object sender, EventArgs e)
        {
            var track = new WaveSoundTrackResourceData();
            var trackDM = new WaveSoundTrackResource(track);
            this.waveSoundPresenter.AddTrack(trackDM);
        }

        /// <summary>
        /// 選択したトラックを削除
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnClick_DeleteTrack(object sender, EventArgs e)
        {
            this.waveSoundPresenter.RemoveSelectedTracks();
        }

        protected override void OnClosing(CancelEventArgs e)
        {
            e.Cancel = true;
            this.Hide();
            base.OnClosing(e);
        }

        public class ClipConverter : ClipboardServiceWSD.IConverter<WaveSoundClipResource, WaveSoundClipResourceDataExchange>
        {
            public override WaveSoundClipResourceDataExchange To(WaveSoundClipResource source)
            {
                var target = new WaveSoundClipResourceDataExchange();
                target.IsEnabled = source.IsEnabled;
                target.Path = source.Path;
                target.Position = source.Position;
                target.StartOffset = source.StartOffset;
                target.Duration = source.Duration;
                target.Volume = source.Volume;
                target.Pitch = source.Pitch;
                target.Pan = source.Pan;
                return target;
            }

            public override WaveSoundClipResource From(WaveSoundClipResourceDataExchange source)
            {
                var target = new WaveSoundClipResourceData();
                target.IsEnabled = source.IsEnabled;
                target.Path = source.Path;
                target.Position = source.Position;
                target.StartOffset = source.StartOffset;
                target.Duration = source.Duration;
                target.Volume = source.Volume;
                target.Pitch = source.Pitch;
                target.Pan = source.Pan;
                return new WaveSoundClipResource(target);
            }
        }

        public class TrackConverter : ClipboardServiceWSD.IConverter<WaveSoundTrackResource, WaveSoundTrackResourceDataExchange>
        {
            public override WaveSoundTrackResourceDataExchange To(WaveSoundTrackResource source)
            {
                var target = new WaveSoundTrackResourceDataExchange();

                foreach (var clip in source.Clips)
                {
                    target.ClipResources.Add(new ClipConverter().To(clip) as WaveSoundClipResourceDataExchange);
                }

                return target;
            }

            public override WaveSoundTrackResource From(WaveSoundTrackResourceDataExchange source)
            {
                var target = new WaveSoundTrackResourceData();

                foreach (var clip in source.ClipResources)
                {
                    target.ClipResources.Add((new ClipConverter().From(clip) as WaveSoundClipResource).Data);
                }

                return new WaveSoundTrackResource(target);
            }
        }

        [Serializable]
        public class WaveSoundClipResourceDataExchange
        {
            public bool IsEnabled { get; set; }
            public int Position { get; set; }
            public int StartOffset { get; set; }
            public int Duration { get; set; }
            public string Path { get; set; }
            public float Volume { get; set; }
            public float Pitch { get; set; }
            public float Pan { get; set; }
        }

        [Serializable]
        public class WaveSoundTrackResourceDataExchange
        {
            public IList<WaveSoundClipResourceDataExchange> ClipResources = new List<WaveSoundClipResourceDataExchange>();
        }

        private void AddClip(int trackNo, double position, int row, string filePath)
        {
            if (string.IsNullOrEmpty(filePath) == true)
            {
                return;
            }

            WaveSoundClipResource waveSoundClipResource = CreateWaveSoundClipResource(trackNo, position, row, filePath);
            if (waveSoundClipResource == null)
            {
                return;
            }

            this.waveSoundPresenter.AddClip(this.WaveSound.WaveSoundResource.Tracks[trackNo], waveSoundClipResource);

            this.waveSoundPresenter.UnselectClipAll();
            this.waveSoundPresenter.SelectClip(waveSoundClipResource);
        }


        private void OnClick_AddClip(object sender, EventArgs e)
        {
            string filePath = WaveFileQuester.QueryLoad(this, null);
            int trackNo = 0;
            double position = 0.0;
            int row = 0;

            if (this.waveSoundPresenter.GetCenterLogicalPoint(ref trackNo, ref position, ref row) == true)
            {
                this.AddClip(trackNo, position, row, filePath);
            }
        }

        private void OnClick_AddClipContext(object sender, EventArgs e)
        {
            string filePath = WaveFileQuester.QueryLoad(this, null);
            var screenPoint = this.contextMenuStrip.Location;
            int trackNo = 0;
            double position = 0.0;
            int row = 0;

            if (this.waveSoundPresenter.GetLogicalPoint(ref trackNo, ref position, ref row, screenPoint.X, screenPoint.Y) == true)
            {
                this.AddClip(trackNo, position, row, filePath);
            }
        }

        private WaveSoundClipResource CreateWaveSoundClipResource(int trackNo, double position, int row, string filePath)
        {
            WaveFileInfo fileInfo = null;
            try
            {
                fileInfo = WaveFileReader.ReadWaveFileInfo(filePath);
            }
            catch
            {
            }

            if (fileInfo == null)
            {
                MessageBox.Show(MessageResource.ErrorMessage_NotWaveFile);
                return null;
            }

            var clip = new WaveSoundClipResourceData();
            clip.Position = (int)position;
            clip.Row = row;
            clip.Duration = (int)fileInfo.Duration.TotalMilliseconds;
            clip.Path = filePath;
            return new WaveSoundClipResource(clip);
        }

        private void OnClickSaveAll(object sender, EventArgs e)
        {
            SaveHelper.SaveAll();
        }
    }


    /// <summary>
    /// クリップボードサービス
    /// </summary>
    public class ClipboardServiceWSD
    {
        private const string DataIdentifier = "WaveSouneEditor";

        private Dictionary<Type, IConverter> toConverters = new Dictionary<Type, IConverter>();
        private Dictionary<Type, IConverter> fromConverters = new Dictionary<Type, IConverter>();

        public bool Contains
        {
            get
            {
                return Clipboard.ContainsData(DataIdentifier);
            }
        }

        public void AddConverter<T1, T2>(IConverter<T1, T2> converter) where T1 : class where T2 : class
        {
            this.toConverters.Add(typeof(T1), converter);
            this.fromConverters.Add(typeof(T2), converter);
        }

        public void To(IEnumerable<object> models)
        {
            var datas = new List<object>();

            foreach (var model in models)
            {
                var converter = this.toConverters[model.GetType()];
                var cdata = converter.To(model);

                using (MemoryStream stream = new MemoryStream())
                {
                    var formatter = new BinaryFormatter();
                    formatter.Serialize(stream, cdata);
                    datas.Add(stream.GetBuffer());
                }
            }

            Clipboard.SetData(DataIdentifier, datas);
        }

        public IEnumerable<object> From()
        {
            var models = new List<object>();

            var datas = Clipboard.GetData(DataIdentifier) as List<object>;
            foreach (var data in datas)
            {
                using (MemoryStream stream = new MemoryStream(data as byte[]))
                {
                    var formatter = new BinaryFormatter();
                    var cdata = formatter.Deserialize(stream);
                    var converter = this.fromConverters[cdata.GetType()];
                    var model = converter.From(cdata);

                    models.Add(model);
                }
            }
            return models;
        }

        public interface IConverter
        {
            Type ToType { get; }
            Type FromType { get; }

            object To(object o);
            object From(object o);
        }

        public abstract class IConverter<T1, T2> : IConverter where T1 : class where T2 : class
        {
            Type IConverter.ToType
            {
                get { return typeof(T1); }
            }

            Type IConverter.FromType
            {
                get { return typeof(T2); }
            }

            object IConverter.To(object o)
            {
                return this.To(o as T1);
            }

            object IConverter.From(object o)
            {
                return this.From(o as T2);
            }

            public abstract T2 To(T1 source);
            public abstract T1 From(T2 target);
        }
    }
}
