﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Windows.Forms.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text.RegularExpressions;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Configurations;
    using NintendoWare.SoundMaker.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Preview;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    /// テキストシーケンスを波形化 コマンドを処理します。
    /// </summary>
    public class WaveConvertHandler : CommandHandler
    {
        private Action<string> setStatusBarText;

        public class WaveOutInfo
        {
            public string SequenceFilePath { get; set; }
            public string TempWaveFilePath { get; set; }
            public int ChannelCount { get; set; }
            public int Volume { get; set; }
            public string PostCommand { get; set; }
            public string PostCommandArgs { get; set; }
            public string DummyBankName { get; set; }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="queryParameter">パラメータの問い合わせ先。</param>
        /// <param name="setStatusBarText">ステータスバーのテキスト更新メソッド。</param>
        /// <param name="isPreview">波形化後にプレビューするかどうか。</param>
        public WaveConvertHandler(IQueryCommandParameter queryParameter, Action<string> setStatusBarText, bool isPreview)
            : base(queryParameter)
        {
            this.IsPreview = isPreview;
            this.setStatusBarText = setStatusBarText;
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get
            {
                return this.IsPreview ?
                    PreviewCommands.WaveConvertAndPreview.ID : PreviewCommands.WaveConvert.ID;
            }
        }

        private bool IsPreview { get; set; }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            if (null == GetTargetProjectService(parameters)) { return CommandStatus.SupportedAndVisible; }

            var targetSounds = GetTargetSounds(parameters);

            if (0 == targetSounds.Length)
            {
                return CommandStatus.SupportedAndVisible;
            }

            // WaveSoundBase 以外が混ざっていたら実行しません。
            if (targetSounds.Any(target => !(target is WaveSoundBase)))
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            Sound[] targets = GetTargetSounds(parameters);
            if (0 == targets.Length) { return false; }

            foreach (var target in targets.OfType<WaveSoundBase>())
            {
                var waveOutInfo = this.GetWaveOutInfo(target);

                if (waveOutInfo == null)
                {
                    continue;
                }

                string bankName =
                    string.IsNullOrEmpty(waveOutInfo.DummyBankName) ? "__DUMMY_BANK__" : waveOutInfo.DummyBankName;

                var bank = this.GetTargetProjectService(parameters).
                    ComponentDictionary[bankName].
                    OfType<SoundSetBankBase>().
                    FirstOrDefault();

                if (bank == null)
                {
                    FormsApplicationCommon.Instance.UIService.ShowMessageBox(
                        string.Format("\"{0}\" not found.", bankName)
                        );
                    return false;
                }

                try
                {
                    var waveFileDirectoryPath = Path.GetDirectoryName(target.FilePath);

                    var seq = new SequenceSoundBase()
                    {
                        FilePath = Path.Combine(waveFileDirectoryPath, waveOutInfo.SequenceFilePath).Replace('/', '\\'),
                        FileType = SequenceSoundFileType.Text,
                        Volume = waveOutInfo.Volume,
                        StartPosition = target.Name,
                    };

                    seq.SoundSetBankReferences.Add(new ComponentReference() { TargetName = bank.Name });
                    seq.SoundSetBanks.Add(bank);

                    OutputWaveFile(
                        seq,
                        30,     // HACK : とりあえず固定で 30 秒
                        string.IsNullOrEmpty(waveOutInfo.TempWaveFilePath) ? target.FilePath : waveOutInfo.TempWaveFilePath,
                        waveOutInfo.ChannelCount == 1);

                    if (!string.IsNullOrEmpty(waveOutInfo.PostCommand))
                    {
                        var process = Process.Start(
                            new ProcessStartInfo(waveOutInfo.PostCommand, waveOutInfo.PostCommandArgs)
                            {
                                WorkingDirectory = waveFileDirectoryPath,
                            }
                            );

                        process.WaitForExit();

                        if (process.ExitCode != 0)
                        {
                            throw new Exception(string.Format(
                                FrameworkResources.MessageResource.Message_CommandReturnErrorCode,
                                process.ExitCode
                                ));
                        }
                    }

                    if (this.setStatusBarText != null)
                    {
                        this.setStatusBarText(string.Format(
                            FrameworkResources.MessageResource.Message_CompleteWaveConvert,
                            target.Name)
                            );
                    }
                }
                catch (DirectoryNotFoundException exception)
                {
                    FormsApplicationCommon.Instance.UIService.ShowMessageBox(exception.Message);
                    return false;
                }
                catch (Exception exception)
                {
                    FormsApplicationCommon.Instance.UIService.ShowMessageBox(
                        string.Format(
                        FrameworkResources.MessageResource.Message_FailedToWaveConvert,
                        target.Name,
                        exception.Message)
                        );
                    return false;
                }
                finally
                {
                    try
                    {
                        if (waveOutInfo != null && !string.IsNullOrEmpty(waveOutInfo.TempWaveFilePath))
                        {
                            File.Delete(waveOutInfo.TempWaveFilePath);
                        }
                    }
                    catch { }
                }
            }

            if (targets.Length == 1 && this.IsPreview)
            {
                // (SIGLO-7835) 動作が変わらないように addPlayMuteParameter: false を指定しておきますが、積極的な意味はありません。
                PreviewPlayerOperator.Play(targets[0], addPlayMuteParameter: false);
            }

            return true;
        }

        /// <summary>
        /// 対象プロジェクトサービスを取得します。
        /// </summary>
        protected SoundProjectService GetTargetProjectService(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }
            return parameters.GetParameter(CommandParameterNames.TargetComponentService)
                   as SoundProjectService;
        }

        /// <summary>
        /// 対象サウンドの一覧を取得します。
        /// </summary>
        protected Sound[] GetTargetSounds(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            return (from Component component in
                        (parameters.GetParameter(CommandParameterNames.TargetComponents)
                          as IEnumerable<Component>)
                    where component is Sound
                    select component as Sound).ToArray();
        }

        private WaveOutInfo GetWaveOutInfo(WaveSoundBase sound)
        {
            string targetText = sound.Comment7.Trim(' ');

            if (string.IsNullOrEmpty(targetText))
            {
                return null;
            }

            if (targetText.StartsWith("//"))
            {
                return null;
            }

            var match = Regex.Match(targetText, "([^\\|]+) *\\| *(\\d) *\\| *(\\d+)( *\\| *([^\\|]*))?( *\\| *([^\\|]*))?( *\\| *([^\\|]*))?");

            if (match.Groups.Count < 4)
            {
                return null;
            }

            try
            {
                var result = new WaveOutInfo()
                {
                    SequenceFilePath = match.Groups[1].Value.Trim(' '),
                    ChannelCount = int.Parse(match.Groups[2].Value),
                    Volume = int.Parse(match.Groups[3].Value),
                };

                if (match.Groups.Count > 5)
                {
                    result.PostCommand = match.Groups[5].Value.Trim(' ').Replace('/', '\\');

                    if (!string.IsNullOrEmpty(result.PostCommand))
                    {
                        result.TempWaveFilePath = Path.GetTempFileName();

                        result.PostCommandArgs = string.Format(
                            "\"{0}\" \"{1}\"",
                            result.TempWaveFilePath,
                            PathEx.MakeRelative(sound.FilePath, Path.GetDirectoryName(sound.FilePath))
                            );

                        if (match.Groups.Count >= 7)
                        {
                            result.PostCommandArgs += " " + match.Groups[7].Value.Trim(' ');
                        }
                    }
                }

                if (match.Groups.Count > 9)
                {
                    result.DummyBankName = match.Groups[9].Value.Trim(' ');
                }

                if (string.IsNullOrEmpty(result.SequenceFilePath) ||
                    result.ChannelCount < 0 ||
                    result.ChannelCount > 2 ||
                    result.Volume < 0 ||
                    result.Volume > 255)
                {
                    return null;
                }

                return result;
            }
            catch
            {
                return null;
            }
        }

        private void OutputWaveFile(Sound sound, int maxDuration, string filePath, bool isMono)
        {
            if (null == sound) { throw new ArgumentNullException("sound"); }
            if (string.IsNullOrEmpty(filePath)) { throw new ArgumentException("filePath"); }

            FormsApplicationCommon.Instance.ExecuteCommand(PreviewCommands.StopAll);
            FormsApplicationCommon.Instance.ExecuteCommand(ToolCommands.StopAllSounds);

            try
            {
                FormsApplicationCommon.Instance.ShutdownWaveOutput();

                Preview.PreviewSound previewSound = Preview.PreviewPlayer.CreatePreviewSound(sound, OutputWaveFileRenderType.k32KHz);

                if (null == previewSound)
                {
                    throw new ApplicationException(FrameworkResources.MessageResource.Message_UnknownError);
                }

                previewSound.OutputWaveFile(
                    filePath,
                    maxDuration,
                    isMono
                    );
            }
            finally
            {
                FormsApplicationCommon.Instance.StartupWaveOutput();
            }
        }
    }
}
