﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Preview
{
    using System;
    using System.Diagnostics;
    using System.Threading;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Preview.Communications;
    using NintendoWare.SoundMaker.Preview.MCS;
    using NintendoWare.SoundMaker.Windows.Forms;
    using NintendoWare.SoundMakerPlugin;
    using NintendoWare.SoundFoundation.Projects;

    public partial class PreviewPlayer
    {
        private const uint RepeatPlayInterval = 1000;	// リピート再生間隔 2000 ms

        protected delegate void PlayerStateEventHandler(object sender, PlayerStateEventArgs e);

        /// <summary>
        /// 内部用イベントクラス
        /// </summary>
        protected class PlayerStateEventArgs : EventArgs
        {
            private PreviewPlayerState _newState = PreviewPlayerState.Stopped;

            public PlayerStateEventArgs(PreviewPlayerState newState)
            {
                _newState = newState;
            }

            #region ** プロパティ

            public PreviewPlayerState NewState
            {
                get { return _newState; }
            }

            #endregion
        }

        #region ** プレイヤー

        protected abstract class BasePlayer : IDisposable
        {
            #region ** デリゲート

            private delegate void RepeatEventHandler();

            #endregion

            #region ** 内部クラス

            private class RepeatInvoker
            {
                #region ** パラメータ

                // リピート状態
                private PreviewPlayerState _state = PreviewPlayerState.Disabled;
                private bool _enabled = false;
                private bool _waiting = false;

                // リピート再生ハンドラ
                private Timer _repeatTimer = null;	// タイマ
                private RepeatEventHandler _handler = null; // ハンドラ

                #endregion

                public RepeatInvoker(RepeatEventHandler handler)
                {
                    if (null == handler) { throw new ArgumentNullException("handler"); }

                    _handler = handler;
                    _repeatTimer = new Timer(OnTimer, null, Timeout.Infinite, Timeout.Infinite);
                }

                #region ** プロパティ

                public bool Enabled
                {
                    get { return _enabled; }
                    set
                    {
                        if (value == _enabled) { return; }

                        _enabled = value;

                        if (!value)
                        {
                            _state = PreviewPlayerState.Disabled;
                        }
                        else
                        {
                            if (PreviewPlayerState.Disabled == _state)
                            {
                                _state = PreviewPlayerState.Stopped;
                            }
                        }
                    }
                }

                public PreviewPlayerState State
                {
                    get { return _state; }
                    set { _state = value; }
                }

                public bool Waiting
                {
                    get
                    {
                        if (!Enabled) { return false; }
                        return _waiting;
                    }
                }

                /// <summary>
                /// 再生可能な状態かどうかを取得します。
                /// </summary>
                public bool CanPlay
                {
                    get { return Enabled; }
                }

                /// <summary>
                /// 一時停止可能な状態かどうかを取得します。
                /// </summary>
                public bool CanPause
                {
                    get
                    {
                        switch (_state)
                        {
                            case PreviewPlayerState.Playing:
                            case PreviewPlayerState.Paused:
                                return true;
                        }

                        return false;
                    }
                }

                /// <summary>
                /// 停止可能な状態かどうかを取得します。
                /// </summary>
                public bool CanStop
                {
                    get
                    {
                        switch (_state)
                        {
                            case PreviewPlayerState.Playing:
                            case PreviewPlayerState.Paused:
                                return true;
                        }

                        return false;
                    }
                }

                public bool Repeat
                {
                    get
                    {
                        switch (_state)
                        {
                            case PreviewPlayerState.Playing:
                            case PreviewPlayerState.Paused:
                                return true;
                        }

                        return false;
                    }
                }

                #endregion

                #region ** イベントハンドラ

                private void OnTimer(object state)
                {
                    Debug.Assert(null != _repeatTimer, "unexpected error");

                    if (Enabled)
                    {
                        if (null == PreviewManager.Instance.InvokeControl) { return; }
                        PreviewManager.Instance.InvokeControl.Invoke(new RepeatEventHandler(OnRepeat));
                    }
                }

                private void OnRepeat()
                {
                    _waiting = false;

                    Debug.Assert(null != _handler, "unexpected error");
                    _handler();
                }

                #endregion

                #region ** メソッド

                public void Invoke()
                {
                    if (!Repeat) { return; }

#if !BUILD_WAVE_EXPORTER
                    FormsApplicationCommon.Instance.LoudnessService.Stop();
#endif

                    _waiting = true;

                    _repeatTimer.Change(RepeatPlayInterval, Timeout.Infinite);
                }

                #endregion
            }

            #endregion

            #region ** パラメータ

            // パラメータ
            private PreviewPlayer _parent = null;
            private PreviewSound _sound = null;
            private string _soundName = string.Empty;
            private PlayerState _state = null;

            // コンポーネント
            private PreviewPlayerInformation _information = null;
            private RepeatInvoker _repeatInvoker = null;	// リピート処理クラス

            #endregion

            protected BasePlayer(PreviewPlayer parent)
            {
                if (null == parent) { throw new ArgumentNullException("parent"); }
                _parent = parent;

                Initialize(DefaultInitialState);
            }

            protected BasePlayer(PreviewPlayer parent, PreviewPlayerState initState)
            {
                if (null == parent) { throw new ArgumentNullException("parent"); }
                _parent = parent;

                Initialize(initState);
            }

            #region ** プロパティ

            public PreviewSound Sound
            {
                get { return _sound; }
                set
                {
                    if (value == _sound) { return; }

                    Stop();

                    if (_sound != null)
                    {
                        _sound.Dispose();
                    }

                    _sound = value;
                    _information.Sound = value;
                    SoundName = _sound == null ? string.Empty : _sound.Name;

                    ResetState();
                }
            }

            public string SoundName
            {
                get { return _soundName; }
                set
                {
                    _soundName = string.IsNullOrEmpty(value) ? string.Empty : value;
                }
            }

            /// <summary>
            /// プレイヤーの再生状態を取得します。
            /// </summary>
            public PreviewPlayerState State
            {
                get
                {
                    if (_repeatInvoker.Waiting)
                    {
                        return _repeatInvoker.State;
                    }

                    if (null == _state) { throw new ApplicationException("invalid player state"); }
                    return _state.Type;
                }
            }

            public bool Repeat
            {
                get { return _repeatInvoker.Enabled; }
                set
                {
                    if (value == _repeatInvoker.Enabled) { return; }

                    bool waiting = _repeatInvoker.Waiting;
                    PreviewPlayerState state = _repeatInvoker.State;

                    ResetRepeatPlayer();
                    _repeatInvoker.Enabled = value;

                    if (value)
                    {
                        _repeatInvoker.State = _state.Type;
                    }
                    else
                    {

                        // リピート再生待ちの場合は停止させる
                        if (waiting && state == PreviewPlayerState.Playing)
                        {

                            if (_information.State != PreviewPlayerState.Stopped)
                            {
                                _state.Stop();
                            }
                            else
                            {
                                // サウンド停止中なら、状態変更通知のみ
                                if (null != PlayerStateChanged)
                                {
                                    PlayerStateChanged(this, new EventArgs());
                                }
                            }

                        }

                    }

                    OnRepeatChanged(new EventArgs());
                }
            }

            /// <summary>
            /// プレイヤーの情報を取得します。
            /// </summary>
            public PreviewPlayerInformation Information
            {
                get { return _information; }
            }

            /// <summary>
            /// プレイヤーの有効状態を取得または設定します。
            /// </summary>
            public bool Enabled
            {
                get { return (PreviewPlayerState.Disabled != State); }
                set
                {
                    if (value == Enabled) { return; }

                    if (value)
                    {

                        if (PreviewPlayerState.Stopped == _information.State) { return; }
                        _information.State = PreviewPlayerState.Stopped;

                    }
                    else
                    {

                        if (PreviewPlayerState.Disabled == _information.State) { return; }
                        _information.State = PreviewPlayerState.Disabled;

                    }

                    OnPlayerStateChanged(_information, new EventArgs());
                }
            }

            /// <summary>
            /// 再生可能な状態かどうかを取得します。
            /// </summary>
            public bool CanPlay
            {
                get
                {
                    if (_repeatInvoker.Waiting) { return _repeatInvoker.CanPlay; }
                    return _state.CanPlay;
                }
            }

            /// <summary>
            /// 一時停止可能な状態かどうかを取得します。
            /// </summary>
            public bool CanPause
            {
                get
                {
                    if (_repeatInvoker.Waiting) { return _repeatInvoker.CanPause; }
                    return _state.CanPause;
                }
            }

            /// <summary>
            /// 停止可能な状態かどうかを取得します。
            /// </summary>
            public bool CanStop
            {
                get
                {
                    if (_repeatInvoker.Waiting) { return _repeatInvoker.CanStop; }
                    return _state.CanStop;
                }
            }

            /// <summary>
            /// リピート可能な状態かどうかを取得します。
            /// </summary>
            public bool CanRepeat
            {
                get { return (PreviewPlayerState.Disabled != _state.Type); }
            }

            /// <summary>
            /// プレビュープレイヤーのインデックスを取得します。
            /// </summary>
            public uint PreviewPlayerIndex
            {
                get { return _parent.Index; }
            }

            protected virtual PreviewPlayerState DefaultInitialState
            {
                get { return PreviewPlayerState.Disabled; }
            }

            #endregion

            #region ** イベント

            public event EventHandler PlayerStateChanged;
            public event EventHandler RepeatChanged;

            #endregion

            #region ** イベントハンドラ

            protected virtual void OnRepeatChanged(EventArgs e)
            {
                if (null != RepeatChanged)
                {
                    RepeatChanged(this, e);
                }
            }

            private void OnPlayerStateChanged(object sender, EventArgs e)
            {
                PreviewPlayerInformation target = sender as PreviewPlayerInformation;
                Debug.Assert(null != target);

                ChangeState(target.State);
            }

            private void OnRepeatStateChanged(object sender, EventArgs e)
            {
                RepeatInvoker target = sender as RepeatInvoker;
                Debug.Assert(null != target);

                ChangeState(target.State);
            }

            private void OnRepeat()
            {
                if (!_repeatInvoker.Enabled) { return; }

                switch (_repeatInvoker.State)
                {
                    case PreviewPlayerState.Playing:
                        try
                        {
                            _state.Play();
                        }
                        catch
                        {
                            this.Stop();
                        }
                        break;
                }
            }

            #endregion

            #region ** メソッド

            ///
            public void SetParameter(string parameterName)
            {
                if (null == _state) { throw new ApplicationException("invalid player state"); }

                _state.SetParameter(parameterName);
            }

            ///
            public void Play()
            {
                if (null == _state) { throw new ApplicationException("invalid player state"); }

                if (_repeatInvoker.Enabled)
                {
                    _repeatInvoker.State = PreviewPlayerState.Playing;
                    ResetRepeatPlayer();
                }

                _state.Play();
            }

            public void Pause()
            {
                if (null == _state) { throw new ApplicationException("invalid player state"); }

                if (!_repeatInvoker.Enabled)
                {
                    _state.Pause();
                    return;
                }

                switch (_repeatInvoker.State)
                {
                    case PreviewPlayerState.Playing:
                        _repeatInvoker.State = PreviewPlayerState.Paused;

                        if (_repeatInvoker.Waiting)
                        {
                            // リピート再生待ち状態の場合は、一時停止状態への変更通知を行う
                            if (null != PlayerStateChanged)
                            {
                                PlayerStateChanged(this, new EventArgs());
                            }
                        }
                        else
                        {
                            _state.Pause();
                        }
                        break;

                    case PreviewPlayerState.Paused:
                        _repeatInvoker.State = PreviewPlayerState.Playing;
                        ResetRepeatPlayer();

                        if (_state.Type == PreviewPlayerState.Paused)
                        {
                            _state.Pause();
                        }
                        else
                        {
                            _state.Play();
                        }
                        break;

                    default:
                        throw new ApplicationException("invalid player state");
                }

            }

            /// <summary>
            ///
            /// </summary>
            public void MuteChannel(Component component, bool value)
            {
                if (this._sound == null)
                {
                    return;
                }

                this._sound.MuteChannel(component, value);
            }

            /// <summary>
            ///
            /// </summary>
            public void Stop()
            {
                if (null == _state) { throw new ApplicationException("invalid player state"); }

                if (!_repeatInvoker.Enabled)
                {
                    _state.Stop();
                    return;
                }

                // リピート再生待ち状態、または一時停止状態で、
                // Stateとの状態が食い違っている場合は、停止状態への変更通知を行う
                bool stateChanged = (PreviewPlayerState.Stopped == _state.Type && _state.Type != _repeatInvoker.State);

                _repeatInvoker.State = PreviewPlayerState.Stopped;
                ResetRepeatPlayer();

                _state.Stop();

                if (stateChanged && null != PlayerStateChanged)
                {
                    PlayerStateChanged(this, new EventArgs());
                }
            }

            public void Dispose()
            {
                if (_sound != null)
                {
                    _sound.Dispose();
                }
            }

            public void Reset()
            {
                Repeat = false;
                Sound = null;
                Enabled = false;
            }

            public void GetVariables(SeqVariableContainer variables)
            {
                if (null == _state) { throw new ApplicationException("invalid player state"); }
                _state.GetVariables(variables);
            }

            public void GetVariables(SeqVariableContainerCollection containers)
            {
                if (null == _state) { throw new ApplicationException("invalid player state"); }
                _state.GetVariables(containers);
            }

            public void SetVariables(SeqVariableContainer variables)
            {
                if (null == _state) { throw new ApplicationException("invalid player state"); }
                _state.SetVariables(variables);
            }

            public void SetVariables(SeqVariableContainerCollection containers)
            {
                if (null == _state) { throw new ApplicationException("invalid player state"); }
                _state.SetVariables(containers);
            }

            public void SetParameter()
            {
                if (null == _state) { throw new ApplicationException("invalid player state"); }
                _state.SetParameter(_parent);
            }

            public void UpdateState()
            {
                _information.UpdateState();
            }

            public virtual void ResetState()
            {
                if (null == Sound)
                {
                    // Disabled
                    _information.State = PreviewPlayerState.Disabled;
                }
                else
                {
                    // Stopped
                    _information.State = PreviewPlayerState.Stopped;
                }

                _information.UpdateState();
            }

            protected virtual PlayerState CreateNewState(PreviewPlayerState newState)
            {
                return new PlayerDisabled();
            }

            protected void ChangeState(PreviewPlayerState newStateType)
            {
                if (null != _state && newStateType == _state.Type) { return; }

                // リピート再生中かつ、
                // プレイヤー状態 [再生/一時停止→停止] のタイミングでリピートタイマを開始する
                if (_repeatInvoker.Repeat)
                {

                    if ((_state.Type == PreviewPlayerState.Playing || _state.Type == PreviewPlayerState.Paused)
                        && newStateType == PreviewPlayerState.Stopped)
                    {
                        _repeatInvoker.Invoke();
                    }

                }


                //
                // 新しい状態を作成する
                //
                PlayerState newState = CreateNewState(newStateType);
                if (null == newState) { throw new Exception("unexpected error"); }

                //
                // 状態を更新してイベント通知する
                //
                _state = newState;

                if (null != PlayerStateChanged)
                {
                    PlayerStateChanged(this, new EventArgs());
                }
            }

            private void Initialize(PreviewPlayerState initStateType)
            {
                // 状態を初期化する
                _state = CreateNewState(initStateType);

                // プレビュープレイヤー情報を生成する
                _information = new PreviewPlayerInformation(_parent, initStateType);
                _information.StateChanged += OnPlayerStateChanged;

                // リピートプレイヤーを生成する
                _repeatInvoker = new RepeatInvoker(OnRepeat);
            }

            private void ResetRepeatPlayer()
            {
                bool oldEnabled = false;
                PreviewPlayerState oldState = PreviewPlayerState.Disabled;

                if (_repeatInvoker.Enabled)
                {
                    oldEnabled = _repeatInvoker.Enabled;
                    oldState = _repeatInvoker.State;
                }

                _repeatInvoker.Enabled = false;

                _repeatInvoker = new RepeatInvoker(OnRepeat);
                _repeatInvoker.Enabled = oldEnabled;
                _repeatInvoker.State = oldState;
            }

            #endregion

            #region ** 抽象プロパティ

            public abstract PreviewPlayerType Type { get; }

            #endregion
        }

        #region ** PCPlayer

        protected class PCPlayer : BasePlayer
        {
            public PCPlayer(PreviewPlayer parent) : base(parent) { }
            public PCPlayer(PreviewPlayer parent, PreviewPlayerState initState) : base(parent, initState) { }

            #region ** メソッドのオーバーライド

            protected override PlayerState CreateNewState(PreviewPlayerState newState)
            {
                switch (newState)
                {
                    case PreviewPlayerState.Stopped:
                        return new PCPlayerStopped(this);

                    case PreviewPlayerState.Playing:
                        return new PCPlayerPlaying(this);

                    case PreviewPlayerState.Paused:
                        return new PCPlayerPaused(this);
                }

                return new PCPlayerDisabled(this);
            }

            #endregion

            #region ** BasePlayer プロパティの実装

            /// <summary>
            /// プレイヤーの種類を取得します。
            /// </summary>
            public override PreviewPlayerType Type
            {
                get { return PreviewPlayerType.PCPlayer; }
            }

            #endregion
        }

        #endregion

        #region ** ViewerPlayer

        protected class ViewerPlayer : BasePlayer
        {
            public ViewerPlayer(PreviewPlayer parent) : base(parent) { }
            public ViewerPlayer(PreviewPlayer parent, PreviewPlayerState initState) : base(parent, initState) { }

            #region ** メソッドのオーバーライド

            protected override PlayerState CreateNewState(PreviewPlayerState newState)
            {
                switch (newState)
                {
                    case PreviewPlayerState.Stopped:
                        return new ViewerPlayerStopped(this);

                    case PreviewPlayerState.Preparing:
                        return new ViewerPlayerPreparing(this);

                    case PreviewPlayerState.Playing:
                        return new ViewerPlayerPlaying(this);

                    case PreviewPlayerState.Paused:
                        return new ViewerPlayerPaused(this);
                }

                return new ViewerPlayerDisabled(this);
            }

            #endregion

            #region ** BasePlayer プロパティの実装

            /// <summary>
            /// プレイヤーの種類を取得します。
            /// </summary>
            public override PreviewPlayerType Type
            {
                get { return PreviewPlayerType.ViewerPlayer; }
            }

            #endregion
        }

        #endregion

        #endregion

        #region ** プレイヤー状態

        protected abstract class PlayerState
        {
            private BasePlayer _player = null;


            protected PlayerState() { }
            protected PlayerState(BasePlayer player)
            {
                _player = player;
            }

            #region ** プロパティ

            protected BasePlayer Player
            {
                get { return _player; }
            }

            #endregion

            #region ** 抽象プロパティ

            public abstract PreviewPlayerState Type { get; }

            public abstract bool CanPlay { get; }
            public abstract bool CanPause { get; }
            public abstract bool CanStop { get; }

            #endregion

            #region ** 抽象メソッド

            public abstract void Play();
            public abstract void Pause();
            public abstract void Stop();

            public abstract void GetVariables(SeqVariableContainer variables);
            public abstract void GetVariables(SeqVariableContainerCollection containers);
            public abstract void SetVariables(SeqVariableContainer variables);
            public abstract void SetVariables(SeqVariableContainerCollection containers);

            public abstract void SetParameter(PreviewPlayer target);

            public virtual void SetParameter(string parameterName)
            {
            }

            #endregion
        }

        protected class PlayerDisabled : PlayerState
        {
            public PlayerDisabled() : base(null) { }

            #region ** PlayerState プロパティの実装

            public override PreviewPlayerState Type
            {
                get { return PreviewPlayerState.Disabled; }
            }

            public override bool CanPlay { get { return false; } }

            public override bool CanPause { get { return false; } }

            public override bool CanStop { get { return false; } }

            #endregion

            #region ** PlayerState メソッドの実装

            public override void Play()
            {
                // 何もしない
            }

            public override void Pause()
            {
                // 何もしない
            }

            public override void Stop()
            {
                // 何もしない
            }

            public override void GetVariables(SeqVariableContainer variables)
            {
                // 何もしない
            }

            public override void GetVariables(SeqVariableContainerCollection containers)
            {
                // 何もしない
            }

            public override void SetVariables(SeqVariableContainer variables)
            {
                // 何もしない
            }

            public override void SetVariables(SeqVariableContainerCollection containers)
            {
                // 何もしない
            }

            public override void SetParameter(PreviewPlayer target)
            {
                // 何もしない
            }

            #endregion
        }

        #region ** PCPlayer状態

        protected abstract class PCPlayerState : PlayerState
        {
            protected PCPlayerState(PCPlayer player) : base(player) { }

            #region ** プロパティ

            protected new PCPlayer Player
            {
                get { return base.Player as PCPlayer; }
            }

            #endregion

            #region ** メソッド

            private void GetLocalVariablesInternal(SeqVariableContainer variables)
            {
                // シーケンスサウンド以外は対象としない
                PreviewSequenceSound seqSound = Player.Sound as PreviewSequenceSound;
                if (null == seqSound) { return; }

                if (SeqVariableType.Local != variables.ID.Type)
                {
                    throw new ArgumentException("invalid sequence variable type.");
                }

                variables.UnlockVariableChangedEvent();

                // ローカル変数を取得する
                foreach (SeqVariable target in variables.Variables)
                {
                    target.Value = seqSound.GetLocalVariable((int)target.No);
                }
            }

            private void SetLocalVariablesInternal(SeqVariableContainer variables)
            {
                // シーケンスサウンド以外は対象としない
                PreviewSequenceSound seqSound = Player.Sound as PreviewSequenceSound;
                if (null == seqSound) { return; }

                if (SeqVariableType.Local != variables.ID.Type)
                {
                    throw new ArgumentException("invalid sequence variable type.");
                }

                // ローカル変数を更新する
                foreach (SeqVariable target in variables.Variables)
                {
                    if (!target.Checked) { continue; }
                    if (SeqVariable.InvalidValue == target.Value) { continue; }
                    seqSound.SetLocalVariable((int)target.No, (short)target.Value);
                }
            }

            private void GetTrackVariablesInternal(SeqVariableContainer variables)
            {
                // シーケンスサウンド以外は対象としない
                PreviewSequenceSound seqSound = Player.Sound as PreviewSequenceSound;
                if (null == seqSound) { return; }

                if (SeqVariableType.Track != variables.ID.Type)
                {
                    throw new ArgumentException("invalid sequence variable type.");
                }

                variables.UnlockVariableChangedEvent();

                // トラック変数を取得する
                foreach (SeqVariable target in variables.Variables)
                {
                    target.Value = seqSound.GetTrackVariable((int)variables.ID.TrackNo, (int)target.No);
                }

                // 変数の値に変更があった場合、ここでイベントが発行される
                variables.UnlockVariableChangedEvent();
            }

            private void SetTrackVariablesInternal(SeqVariableContainer variables)
            {
                // シーケンスサウンド以外は対象としない
                PreviewSequenceSound seqSound = Player.Sound as PreviewSequenceSound;
                if (null == seqSound) { return; }

                if (SeqVariableType.Track != variables.ID.Type)
                {
                    throw new ArgumentException("invalid sequence variable type.");
                }

                // トラック変数を更新する
                foreach (SeqVariable target in variables.Variables)
                {
                    if (!target.Checked) { continue; }
                    if (SeqVariable.InvalidValue == target.Value) { continue; }
                    seqSound.SetTrackVariable((int)variables.ID.TrackNo, (int)target.No, (short)target.Value);
                }
            }

            #endregion

            #region ** PlayerState メソッドの実装

            public override void GetVariables(SeqVariableContainer variables)
            {
                // シーケンスサウンド以外は対象としない
                PreviewSequenceSound seqSound = Player.Sound as PreviewSequenceSound;
                if (null == seqSound) { return; }

                switch (variables.ID.Type)
                {
                    case SeqVariableType.Local:
                        GetLocalVariablesInternal(variables);
                        break;

                    case SeqVariableType.Track:
                        GetTrackVariablesInternal(variables);
                        break;

                    default:
                        throw new ArgumentException("invalid sequence variable type");
                }
            }

            public override void GetVariables(SeqVariableContainerCollection containers)
            {
                // シーケンスサウンド以外は対象としない
                PreviewSequenceSound seqSound = Player.Sound as PreviewSequenceSound;
                if (null == seqSound) { return; }

                foreach (SeqVariableContainer container in containers)
                {
                    switch (container.ID.Type)
                    {
                        case SeqVariableType.Local:
                            GetLocalVariablesInternal(container);
                            break;

                        case SeqVariableType.Track:
                            GetTrackVariablesInternal(container);
                            break;

                        default:
                            throw new ArgumentException("invalid sequence variable type");
                    }
                }
            }

            public override void SetVariables(SeqVariableContainer variables)
            {
                // シーケンスサウンド以外は対象としない
                PreviewSequenceSound seqSound = Player.Sound as PreviewSequenceSound;
                if (null == seqSound) { return; }

                switch (variables.ID.Type)
                {
                    case SeqVariableType.Local:
                        SetLocalVariablesInternal(variables);
                        break;

                    case SeqVariableType.Track:
                        SetTrackVariablesInternal(variables);
                        break;

                    default:
                        throw new ArgumentException("invalid sequence variable type");
                }
            }

            public override void SetVariables(SeqVariableContainerCollection containers)
            {
                // シーケンスサウンド以外は対象としない
                PreviewSequenceSound seqSound = Player.Sound as PreviewSequenceSound;
                if (null == seqSound) { return; }

                foreach (SeqVariableContainer container in containers)
                {
                    switch (container.ID.Type)
                    {
                        case SeqVariableType.Local:
                            SetLocalVariablesInternal(container);
                            break;

                        case SeqVariableType.Track:
                            SetTrackVariablesInternal(container);
                            break;

                        default:
                            throw new ArgumentException("invalid sequence variable type");
                    }
                }
            }

            public override void SetParameter(PreviewPlayer target)
            {
                // 何もしない
            }

            #endregion
        }

        protected class PCPlayerDisabled : PCPlayerState
        {
            public PCPlayerDisabled(PCPlayer player) : base(player) { }

            #region ** PlayerState プロパティの実装

            public override PreviewPlayerState Type
            {
                get { return PreviewPlayerState.Disabled; }
            }

            public override bool CanPlay { get { return false; } }

            public override bool CanPause { get { return false; } }

            public override bool CanStop { get { return false; } }

            #endregion

            #region ** PlayerState メソッドの実装

            public override void Play()
            {
                // 何もしない
            }

            public override void Pause()
            {
                // 何もしない
            }

            public override void Stop()
            {
                // 何もしない
            }

            #endregion
        }

        protected class PCPlayerStopped : PCPlayerState
        {
            public PCPlayerStopped(PCPlayer player) : base(player) { }

            #region ** PlayerState プロパティの実装

            public override PreviewPlayerState Type
            {
                get { return PreviewPlayerState.Stopped; }
            }

            public override bool CanPlay { get { return true; } }

            public override bool CanPause { get { return false; } }

            public override bool CanStop { get { return false; } }

            #endregion

            #region ** PlayerState メソッドの実装

            public override void Play()
            {
                if (null == Player.Sound) { return; }

                Player.Sound.Load();

                // HACK : Prepare() にて確保したチャンネルが Play() にて Start される前に
                //        解放されてしまう問題への暫定対処として、各チャンネルの Start が
                //        完了するまでの間、サウンドスレッドをロックするようにしています。
                SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.RuntimeSoundSystem_LockSoundThread();

                try
                {
                    Player.Sound.Prepare();
                    Player.Sound.Play();
                }
                finally
                {
                    SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.RuntimeSoundSystem_UnlockSoundThread();
                }
            }

            public override void Pause()
            {
                // 何もしない
            }

            public override void Stop()
            {
                // 何もしない
            }

            #endregion
        }

        protected class PCPlayerPlaying : PCPlayerState
        {
            public PCPlayerPlaying(PCPlayer player) : base(player) { }

            #region ** PlayerState プロパティの実装

            public override PreviewPlayerState Type
            {
                get { return PreviewPlayerState.Playing; }
            }

            public override bool CanPlay { get { return true; } }

            public override bool CanPause { get { return true; } }

            public override bool CanStop { get { return true; } }

            #endregion

            #region ** PlayerState メソッドの実装

            public override void SetParameter(string parameterName)
            {
                Player.Sound.SetParameter(parameterName);
            }

            public override void Play()
            {
                if (null == Player.Sound) { return; }

                Player.Sound.Prepare();
                Player.Sound.Play();
            }

            public override void Pause()
            {
                if (null == Player.Sound) { return; }

                Player.Sound.Pause(true);
            }

            public override void Stop()
            {
                if (null == Player.Sound) { return; }

                Player.Sound.Stop();
            }

            #endregion
        }

        protected class PCPlayerPaused : PCPlayerState
        {
            public PCPlayerPaused(PCPlayer player) : base(player) { }

            #region ** PlayerState プロパティの実装

            public override PreviewPlayerState Type
            {
                get { return PreviewPlayerState.Paused; }
            }

            public override bool CanPlay { get { return true; } }

            public override bool CanPause { get { return true; } }

            public override bool CanStop { get { return true; } }

            #endregion

            #region ** PlayerState メソッドの実装

            public override void Play()
            {
                if (null == Player.Sound) { return; }

                Player.Sound.Pause(false);
            }

            public override void Pause()
            {
                Play();
            }

            public override void Stop()
            {
                if (null == Player.Sound) { return; }

                Player.Sound.Stop();
            }

            #endregion
        }

        #endregion

        #region ** ViewerPlayer状態

        protected abstract class ViewerPlayerState : PlayerState
        {
            protected ViewerPlayerState(ViewerPlayer player) : base(player) { }

            #region ** プロパティ

            protected new ViewerPlayer Player
            {
                get { return base.Player as ViewerPlayer; }
            }

            #endregion

            #region ** PlayerState メソッドの実装

            public override void GetVariables(SeqVariableContainer variables)
            {
                this.CommunicationService.GetSeqVariables(variables);
            }

            public override void GetVariables(SeqVariableContainerCollection containers)
            {
                this.CommunicationService.GetSeqVariables(containers);
            }

            public override void SetVariables(SeqVariableContainer variables)
            {
                this.CommunicationService.SetSeqVariables(variables);
            }

            public override void SetVariables(SeqVariableContainerCollection containers)
            {
                this.CommunicationService.SetSeqVariables(containers);
            }

            public override void SetParameter(PreviewPlayer target)
            {
                this.CommunicationService.SetPreviewPlayerParameter(target);
            }

            protected CommunicationService CommunicationService
            {
                get
                {
#if BUILD_WAVE_EXPORTER
                    return null;
#else
                    return FormsApplicationCommon.Instance.CommunicationService;
#endif
                }
            }

            #endregion
        }

        protected class ViewerPlayerDisabled : ViewerPlayerState
        {
            public ViewerPlayerDisabled(ViewerPlayer player) : base(player) { }

            #region ** PlayerState プロパティの実装

            public override PreviewPlayerState Type
            {
                get { return PreviewPlayerState.Disabled; }
            }

            public override bool CanPlay { get { return false; } }

            public override bool CanPause { get { return false; } }

            public override bool CanStop { get { return false; } }

            #endregion

            #region ** PlayerState メソッドの実装

            public override void Play()
            {
                // 何もしない
            }

            public override void Pause()
            {
                // 何もしない
            }

            public override void Stop()
            {
                // 何もしない
            }

            #endregion
        }

        protected class ViewerPlayerStopped : ViewerPlayerState
        {
            public ViewerPlayerStopped(ViewerPlayer player) : base(player) { }

            #region ** PlayerState プロパティの実装

            public override PreviewPlayerState Type
            {
                get { return PreviewPlayerState.Stopped; }
            }

            public override bool CanPlay { get { return true; } }

            public override bool CanPause { get { return false; } }

            public override bool CanStop { get { return false; } }

            #endregion

            #region ** PlayerState メソッドの実装

            public override void Play()
            {
                this.CommunicationService.Play(Player.PreviewPlayerIndex, Player.SoundName);
            }

            public override void Pause()
            {
                // 何もしない
            }

            public override void Stop()
            {
                // 何もしない
            }

            #endregion
        }

        protected class ViewerPlayerPreparing : ViewerPlayerState
        {
            public ViewerPlayerPreparing(ViewerPlayer player) : base(player) { }

            #region ** PlayerState プロパティの実装

            public override PreviewPlayerState Type
            {
                get { return PreviewPlayerState.Preparing; }
            }

            public override bool CanPlay { get { return false; } }

            public override bool CanPause { get { return false; } }

            public override bool CanStop { get { return true; } }

            #endregion

            #region ** PlayerState メソッドの実装

            public override void Play()
            {
                // 何もしない
            }

            public override void Pause()
            {
                // 何もしない
            }

            public override void Stop()
            {
                this.CommunicationService.Stop(Player.PreviewPlayerIndex);
            }

            #endregion
        }

        protected class ViewerPlayerPlaying : ViewerPlayerState
        {
            public ViewerPlayerPlaying(ViewerPlayer player) : base(player) { }

            #region ** PlayerState プロパティの実装

            public override PreviewPlayerState Type
            {
                get { return PreviewPlayerState.Playing; }
            }

            public override bool CanPlay { get { return true; } }

            public override bool CanPause { get { return true; } }

            public override bool CanStop { get { return true; } }

            #endregion

            #region ** PlayerState メソッドの実装

            public override void Play()
            {
                this.CommunicationService.Play(Player.PreviewPlayerIndex, Player.SoundName);
            }

            public override void Pause()
            {
                this.CommunicationService.Pause(Player.PreviewPlayerIndex);
            }

            public override void Stop()
            {
                this.CommunicationService.Stop(Player.PreviewPlayerIndex);
            }

            #endregion
        }

        protected class ViewerPlayerPaused : ViewerPlayerState
        {
            public ViewerPlayerPaused(ViewerPlayer player) : base(player) { }

            #region ** PlayerState プロパティの実装

            public override PreviewPlayerState Type
            {
                get { return PreviewPlayerState.Paused; }
            }

            public override bool CanPlay { get { return true; } }

            public override bool CanPause { get { return true; } }

            public override bool CanStop { get { return true; } }

            #endregion

            #region ** PlayerState メソッドの実装

            public override void Play()
            {
                this.CommunicationService.Pause(Player.PreviewPlayerIndex);
            }

            public override void Pause()
            {
                Play();
            }

            public override void Stop()
            {
                this.CommunicationService.Stop(Player.PreviewPlayerIndex);
            }

            #endregion
        }

        #endregion

        #endregion
    }
}
