﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.ObjectModel;
using System.Diagnostics;

namespace NintendoWare.SoundMaker.Preview
{
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Preview.Communications;
    using NintendoWare.SoundMaker.Preview.MCS;
    using NintendoWare.SoundMaker.Windows.Forms;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    public class PreviewBankManager
    {
        public const uint MaxBankCount = 32;
        public const uint DefaultBankCount = 4;

        // パラメータ
        private PreviewBankCollection _previewBanks = new PreviewBankCollection();

        // 状態
        private bool _synchronizedViewer = false;             // Viewerとの接続状態
        private PreviewBank _selectedPreviewBank = null;   // 選択中のプレビューバンク


        public PreviewBankManager() { }

        //-----------------------------------------------------------------

        public event EventHandler ViewerSynchronousChanged;
        public event EventHandler PreviewBankSelectChanged;

        //-----------------------------------------------------------------

        public PreviewBankCollection PreviewBanks
        {
            get { return _previewBanks; }
        }

        public PreviewBank SelectedPreviewBank
        {
            get { return _selectedPreviewBank; }
            set
            {
                if (value == _selectedPreviewBank) { return; }
                if (!_previewBanks.Contains(value)) { return; }

                _selectedPreviewBank = value;

                OnPreviewPlayerSelectChangedEvent(new EventArgs());
            }
        }

        public uint SelectedPreviewBankIndex
        {
            get
            {
                if (null == _selectedPreviewBank) { return uint.MaxValue; }
                return _selectedPreviewBank.Index;
            }
            set
            {
                if (value == uint.MaxValue) { throw new ArgumentOutOfRangeException(); }
                if (value >= _previewBanks.Count) { throw new ArgumentOutOfRangeException(); }

                SelectedPreviewBank = _previewBanks[(int)value];
            }
        }

        public bool IsSynchronizedViewer
        {
            get { return _synchronizedViewer; }
        }

        private CommunicationService CommunicationService
        {
            get
            {
#if BUILD_WAVE_EXPORTER
                return null;
#else
                return FormsApplicationCommon.Instance.CommunicationService;
#endif
            }
        }

        //-----------------------------------------------------------------

        public void Initialize()
        {
            Initialize(DefaultBankCount);
        }

        public void Initialize(uint playerCount)
        {
            if (_previewBanks.Count == playerCount) { return; }
            if (MaxBankCount < playerCount) { throw new ArgumentOutOfRangeException(); }

            _previewBanks.Clear();

            for (int i = 0; i < playerCount; i++)
            {

                PreviewBank newBank = new PreviewBank(this, (uint)i);
                newBank.ItemChanged += OnPreviewBankItemChanged;

                _previewBanks.Add(newBank);

            }

            // バンクを更新する
            Update();

            // バンク0を選択する
            if (playerCount > 0)
            {
                SelectedPreviewBank = _previewBanks[0];
            }


            // イベントハンドラを設定する
            this.CommunicationService.ViewerConnectionChanged += OnViewerConnectionChanged;
        }

        public void Uninitialize()
        {
            _previewBanks.Clear();

            this.CommunicationService.ViewerConnectionChanged -= OnViewerConnectionChanged;
        }

        public void ResetAll()
        {
            foreach (PreviewBank bank in _previewBanks)
            {
                bank.Reset();
            }
        }

        public void SetAllLabels()
        {
            if (!IsSynchronizedViewer) { return; }

            // TODO : ★全てのバンクラベルを送信する
            //MCSSound.Instance.SetBankLabels(this);
        }

        //-----------------------------------------------------------------

        protected virtual void OnPreviewPlayerSelectChangedEvent(EventArgs e)
        {
            if (null != PreviewBankSelectChanged)
            {
                PreviewBankSelectChanged(this, e);
            }
        }

        protected virtual void OnViewerSynchronousChanged(EventArgs e)
        {
            // Viewer との接続状態を更新する
            _synchronizedViewer = (ConnectionState.Connected == this.CommunicationService.GetState());

            if (null != ViewerSynchronousChanged)
            {
                ViewerSynchronousChanged(this, e);
            }

            Update();
        }

        //-----------------------------------------------------------------

        private void Update()
        {
            // 全てのバンクラベルを送信する（Viewerのみ）
            if (this.IsSynchronizedViewer)
            {
                this.SetAllLabels();
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// Viewer との接続状態が変更されると発生します。
        /// </summary>
        /// <param name="sender">イベントの発行者</param>
        /// <param name="e">Viewer 接続状態イベント</param>
        private void OnViewerConnectionChanged(object sender, NintendoWare.SoundMaker.Framework.Preview.Communications.ConnectionChangedEventArgs e)
        {
            switch (e.State)
            {
                case ConnectionState.Connected:
                case ConnectionState.Disconnected:
                    OnViewerSynchronousChanged(new EventArgs());
                    break;
            }
        }

        private void OnPreviewBankItemChanged(object sender, EventArgs e)
        {
            if (!IsSynchronizedViewer) { return; }
            this.SetAllLabels();
        }
    }
}
