﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Preview
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Preview.MCS;
    using NintendoWare.ToolDevelopmentKit;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    public partial class PreviewBank
    {
        public const uint InvalidIndex = uint.MaxValue; // インデックスの無効値

        private PreviewBankManager parent = null;

        private SoundSetBankBase item = null;
        private BankServiceReference bankServiceReference = null;
        private uint index = 0;

        private Dictionary<int, Instrument> instrumentMap = new Dictionary<int, Instrument>();

        public PreviewBank(PreviewBankManager parent, uint index)
        {
            if (null == parent) { throw new ArgumentNullException("parent"); }

            this.parent = parent;
            this.index = index;
        }

        public event EventHandler ItemChanged;

        /// <summary>
        /// PreviewBankManager を取得します。
        /// </summary>
        public PreviewBankManager Parent
        {
            get { return this.parent; }
        }

        /// <summary>
        /// プレビュープレイヤーのインデックスを取得します。
        /// </summary>
        public uint Index
        {
            get { return this.index; }
        }

        public SoundSetBankBase Item
        {
            get { return this.item; }
            set { this.Attach(value); }
        }

        public bool IsEnabled
        {
            get { return this.item != null; }
        }

        public void Reset()
        {
            Attach(null);
        }

        public Instrument GetInstrument(int programNo)
        {
            Instrument instrument = null;
            this.instrumentMap.TryGetValue(programNo, out instrument);

            return instrument;
        }

        protected virtual void OnItemChanged(EventArgs e)
        {
            if (null != this.ItemChanged)
            {
                this.ItemChanged(this, e);
            }
        }

        private void Attach(SoundSetBankBase newItem)
        {
            if (newItem == this.item) { return; }

            BankServiceReference newBankServiceReference = null;
            if (newItem != null)
            {
                try
                {
                    newBankServiceReference =
                        ApplicationBase.Instance.BankServices.OpenItem(newItem.FilePath);
                }
                catch (Exception)
                {
                    return;
                }
            }

            //
            if (this.item != null)
            {
                this.item.Parameters.ParameterValueChanged -= this.UpdateInstrumentMap;
            }

            if (this.bankServiceReference != null)
            {
                this.bankServiceReference.Close();
                this.bankServiceReference = null;
            }

            this.item = newItem;

            if (this.item != null)
            {
                this.item.Parameters.ParameterValueChanged += this.UpdateInstrumentMap;
                this.bankServiceReference = newBankServiceReference;
            }

            this.UpdateInstrumentMap();
            this.OnItemChanged(new EventArgs());
        }

        private void UpdateInstrumentMap()
        {
            this.instrumentMap.Clear();

            if (this.item == null || this.bankServiceReference == null)
            {
                return;
            }

            foreach (Instrument instrument in this.bankServiceReference.Target.Instruments)
            {
                if (this.instrumentMap.ContainsKey(instrument.ProgramNo))
                {
                    continue;
                }

                this.instrumentMap.Add(instrument.ProgramNo, instrument);
            }
        }

        private void UpdateInstrumentMap(object sender, ParameterEventArgs e)
        {
            if (e.Key != ProjectParameterNames.Instrument.ProgramNo)
            {
                return;
            }

            this.UpdateInstrumentMap();
        }
    }

    public class PreviewBankCollection : Collection<PreviewBank> { }
}
