﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Core.Parameters;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework;
using NintendoWare.SoundMaker.Framework.FileManagement;
using NintendoWare.SoundMakerPlugin;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Preview.Service
{
    public class LoudnessService
    {
        private static readonly HashAlgorithm HashAlgorithm = new SHA1CryptoServiceProvider();

        private enum MeasureStatus
        {
            Measuring,
            Stopping,
            Stopped,
        }

        private const float WaitingTime = 5 * 10000000; // 再計測するまでの待ち時間５秒
        private readonly LoudnessMeasureStack _measureStack = new LoudnessMeasureStack();
        private readonly Timer _updateMeasureTimer = new Timer();
        private readonly Dictionary<string, List<Component>> _fileDictionary = new Dictionary<string, List<Component>>();
        private System.Threading.CancellationTokenSource _tokenSource = null;
        private Task<float> _task = null;
        private LoudnessMeasureStackItem _currentLoudnessMeasureStackItem = null;
        private bool _stopMeasureLoudnessFlag = false;
        private DateTime _stopMeasureLoudnessTime;
        private bool _cancelRetryFlag = true;
        private MeasureStatus _measureStatus = MeasureStatus.Stopped;

        public readonly HashSet<string> StreamSoundParameters = new HashSet<string>()
        {
            ProjectParameterNames.FilePath,
            ProjectParameterNames.WaveEncoding,
            ProjectParameterNames.Volume,
            ProjectParameterNames.Pan,
            ProjectParameterNames.PanMode,
            ProjectParameterNames.PanCurve,
            ProjectParameterNames.Pitch,
            ProjectParameterNames.Biquad,
            ProjectParameterNames.BiquadType,
            ProjectParameterNames.LPF,
            ProjectParameterNames.Sends.MainSend,
            ProjectParameterNames.IsResampleEnabled,
            ProjectParameterNames.SampleRate,
            ProjectParameterNames.IsDownMixEnabled,
        };

        public readonly HashSet<string> WaveSoundParameters = new HashSet<string>()
        {
            ProjectParameterNames.FilePath,
            ProjectParameterNames.WaveEncoding,
            ProjectParameterNames.Volume,
            ProjectParameterNames.Pan,
            ProjectParameterNames.PanMode,
            ProjectParameterNames.PanCurve,
            ProjectParameterNames.Pitch,
            ProjectParameterNames.Biquad,
            ProjectParameterNames.BiquadType,
            ProjectParameterNames.LPF,
            ProjectParameterNames.Sends.MainSend,
            ProjectParameterNames.WaveSound.EnvelopeRelease,
            ProjectParameterNames.IsResampleEnabled,
            ProjectParameterNames.SampleRate,
            ProjectParameterNames.IsDownMixEnabled,
        };

        public readonly HashSet<string> SequenceSoundParameters = new HashSet<string>()
        {
            ProjectParameterNames.FilePath,
            ProjectParameterNames.Volume,
            ProjectParameterNames.Pan,
            ProjectParameterNames.SequenceSound.SoundSetBankReferences,
            ProjectParameterNames.SequenceSound.StartPosition,
        };

        // <summary>
        // 計測を開始する前にイベントが発生します。
        // </summary>
        public event MeasureLoudnessEventHandler Measuring;

        // <summary>
        // 計測が終了したときにイベントが発生します。
        // </summary>
        public event MeasureLoudnessEventHandler Measured;

        public LoudnessService()
        {
            _updateMeasureTimer.Tick += this.MeasureLoudness;
            _updateMeasureTimer.Enabled = false;
        }

        // <summary>
        // 計測の実行の有効無効を設定または取得します。
        // </summary>
        public bool Enabled
        {
            get
            {
                return _updateMeasureTimer.Enabled;
            }
            set
            {
                _updateMeasureTimer.Enabled = value; // 定期的な計測呼び出しの停止または再開をします。

                if (value == false)
                {
                    this.StopMeasuring(); // 計測を停止します。
                }
            }
        }

        // <summary>
        // 計測中かどうか取得します。
        // </summary>
        public bool IsMeasuring
        {
            get
            {
                return _measureStatus != MeasureStatus.Stopped;
            }
        }

        // <summary>
        // FileWatcherService を取得します。
        // </summary>
        private FileWatcherService FileWatcherService
        {
            get
            {
                return ApplicationBase.Instance.FileWatcherService;
            }
        }

        // <summary>
        // 全てのシーケンスサウンドを取得します。
        // </summary>
        private SequenceSoundBase[] SequenceSounds
        {
            get
            {
                return ApplicationBase.Instance.ProjectService.SequenceSounds;
            }
        }

        // <summary>
        // シーケンスサウンドの最大計測時間
        // </summary>
        private int? SequenceMaxMeasureDuration
        {
            get
            {
                return ApplicationBase.Instance.ProjectService.Project.SequenceMaxMeasureDuration;
            }
        }

        // <summary>
        // 自動計測するかどうか。
        // </summary>
        private bool AutoMeasureLoudness
        {
            get
            {
                return ApplicationBase.Instance.AppConfiguration.Options.Application.Statistics.EnabledAutoMeasureLoudness;
            }
        }

        // <summary>
        // サウンドを自動計測の対象に追加します。
        // </summary>
        // <param name="components">計測対象にするサウンドを指定します。</param>
        public void AddAutoMeasure(IEnumerable<Component> components)
        {
            foreach (Component component in components)
            {
                // パラメータの変更を監視します。
                if (component is Sound == true ||
                    component is StreamSoundTrackBase == true ||
                    component is SoundSetBankBase == true)
                {
                    component.ParameterValueChanged += this.OnParameterValueChanged;
                }

                // ファイルを監視します。
                if (component is StreamSoundTrackBase == true)
                {
                    this.AddFileWatch((component as StreamSoundTrackBase).FilePath, component);
                }
                else if (component is WaveSoundBase == true)
                {
                    this.AddFileWatch((component as WaveSoundBase).FilePath, component);
                }
                else if (component is SequenceSoundBase == true)
                {
                    this.AddFileWatch((component as SequenceSoundBase).FilePath, component);
                }
                else if (component is SoundSetBankBase == true)
                {
                    this.AddFileWatch((component as SoundSetBankBase).FilePath, component);
                }

                // 計測スタックに優先度「低」で積みます。
                if (this.AutoMeasureLoudness == true && component is Sound == true && this.Enabled == true)
                {
                    this.MeasureLowPriority(component as Sound);
                }
            }
        }

        // <summary>
        // サウンドを自動計測の対象から削除します。
        // </summary>
        // <param name="components">計測対象から除外するサウンドを指定します。</param>
        public void RemoveAutoMeasure(IEnumerable<Component> components)
        {
            foreach (Component component in components)
            {
                // パラメータの変更の監視をやめます。
                if (component is Sound == true ||
                    component is StreamSoundTrackBase == true ||
                    component is SoundSetBankBase == true)
                {
                    component.ParameterValueChanged -= this.OnParameterValueChanged;
                }

                // ファイルの監視をやめます。
                if (component is StreamSoundTrackBase == true)
                {
                    this.RemoveFileWatch((component as StreamSoundTrackBase).FilePath, component);
                }
                else if (component is WaveSoundBase == true)
                {
                    this.RemoveFileWatch((component as WaveSoundBase).FilePath, component);
                }
                else if (component is SequenceSoundBase == true)
                {
                    this.RemoveFileWatch((component as SequenceSoundBase).FilePath, component);
                }
                else if (component is SoundSetBankBase == true)
                {
                    this.RemoveFileWatch((component as SoundSetBankBase).FilePath, component);
                }

                // 計測スタックから削除します。
                if (component is Sound == true)
                {
                    _measureStack.Remove(component as Sound);
                }

                // 計測中なら計測を中止します。
                if (_measureStatus == MeasureStatus.Measuring && _currentLoudnessMeasureStackItem.Sound == component as Sound)
                {
                    _cancelRetryFlag = false; // 計測を中止するサウンドをキャンセル処理で計測スタックに積まないようにします。
                    this.StopMeasuring();     // 計測を中止します。
                }
            }
        }

        // <summary>
        // 計測待ちをクリアします。
        // </summary>
        public void Clear()
        {
            _measureStack.Clear(); // 計測スタックをクリアします。
        }

        // <summary>
        // 計測を中止し、しばらくの間、計測しないようにします。
        // </summary>
        public void Stop()
        {
            // 外部からの停止の要請なので、しばらくの間、計測しないようにする。
            _stopMeasureLoudnessFlag = true;
            _stopMeasureLoudnessTime = DateTime.Now;

            this.StopMeasuring();
        }

        // <summary>
        // サウンドを優先度「高」で計測します。
        // </summary>
        // <param name="sounds">優先度「高」で計測したいサウンドを指定します。</param>
        public void MeasureHighPriority(IEnumerable<Sound> sounds)
        {
            sounds.Reverse().ForEach((sound) => this.MeasureHighPriority(sound));
        }

        // <summary>
        // サウンドを優先度「低」計測します。
        // </summary>
        // <param name="sounds">優先度「低」で計測したいサウンドを指定します。</param>
        public void MeasureLowPriority(IEnumerable<Sound> sounds)
        {
            sounds.Reverse().ForEach((sound) => this.MeasureLowPriority(sound));
        }

        // <summary>
        // 計測を中止します。
        // </summary>
        private void StopMeasuring()
        {
            if (_tokenSource != null)
            {
                Debug.Assert(_task != null); // _tokenSource と _task の寿命は同じ

                _measureStatus = MeasureStatus.Stopping; // 状態を「中止」に設定します。

                _tokenSource.Cancel(); // キャンセルすると例外が発生する。
            }
            if (_task != null)
            {
                Debug.Assert(_tokenSource != null); // _tokenSource と _task の寿命は同じ

                try
                {
                    _task.Wait();
                }
                catch // キャンセルの例外をキャッチする、でもここでは何もしない。
                {
                }
            }
        }

        // <summary>
        // サウンドを優先度「高」で計測します。
        // </summary>
        // <param name="sounds">優先度「高」で計測したいサウンドを指定します。</param>
        private void MeasureHighPriority(Sound sound)
        {
            if (_currentLoudnessMeasureStackItem?.Sound == sound)
            {
                this.StopMeasuring();
            }

            sound.IntegratedLoudnessStatus = IntegratedLoudnessStatus.WaitingMeasure;
            _measureStack.Push(sound, LoudnessMeasurePriority.High);
        }

        // <summary>
        // サウンドを優先度「低」計測します。
        // </summary>
        // <param name="sounds">優先度「低」で計測したいサウンドを指定します。</param>
        private void MeasureLowPriority(Sound sound)
        {
            sound.IntegratedLoudnessStatus = IntegratedLoudnessStatus.WaitingMeasure;
            _measureStack.Push(sound, LoudnessMeasurePriority.Low);
        }

        // <summary>
        // バックグラウンドで計測します。
        // タイマーで定期的に呼ばれるメソッドです。
        // </summary>
        private void MeasureLoudness(object sender, EventArgs e)
        {
            if (_stopMeasureLoudnessFlag == true)
            {
                if (DateTime.Now.Ticks - _stopMeasureLoudnessTime.Ticks < WaitingTime) // ５秒以上待った？
                {
                    return; // 計測再開しない。
                }

                _stopMeasureLoudnessFlag = false; // 計測再開する。
            }

            // 計測ではなく他でプレビュー再生している場合は何もしない。
            if (_measureStatus == MeasureStatus.Stopped &&
                SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.RuntimeSoundSystem_GetActiveVoiceCount() > 0)
            {
                return;
            }

            // 計測を中止している最中の場合
            if (_measureStatus == MeasureStatus.Stopping)
            {
                return;
            }

            LoudnessMeasureStackItem item = _measureStack.Peek(); // 計測スタックのトップを取得。
            if (item == null)
            {
                return;
            }

            // 計測中の場合（何もしない、または、計測を中止する）
            if (_measureStatus == MeasureStatus.Measuring)
            {
                // 現在計測中よりも高優先のサウンドが見つかった場合、計測を中止します。
                if (_currentLoudnessMeasureStackItem != item)
                {
                    // ただし、現在計測中のサウンドの優先度が低く、計測待ちの優先度も低い場合は計測は中止しない。
                    if (_currentLoudnessMeasureStackItem.LoudnessMeasurePriority == LoudnessMeasurePriority.Low &&
                        item.LoudnessMeasurePriority == LoudnessMeasurePriority.Low)
                    {
                        return;
                    }

                    this.StopMeasuring();
                }

                return;
            }

            _currentLoudnessMeasureStackItem = item;

            int? maxDuration = null;

            if (_currentLoudnessMeasureStackItem.Sound is SequenceSoundBase == true)
            {
                // シーケンスサウンドの場合は、最大計測時間を設定します。
                maxDuration = this.SequenceMaxMeasureDuration;
            }

            // メインスレッドのデフォルトのタスクスケジューラをメインスレッド上で取得しておく。
            var taskScheduler = TaskScheduler.FromCurrentSynchronizationContext();

            PreviewSound previewSound = PreviewPlayer.CreatePreviewSound(_currentLoudnessMeasureStackItem.Sound, OutputWaveFileRenderType.k48KHz, true);
            if (previewSound != null)
            {
                _measureStatus = MeasureStatus.Measuring;
                this.OnMeasuring(new Sound[] { _currentLoudnessMeasureStackItem.Sound });

                _tokenSource = new System.Threading.CancellationTokenSource();

                _task = Task.Factory.StartNew(() =>
                {
                    // ワーカースレッドで計測する。
                    return previewSound.MeasureIntegratedLoudness(maxDuration, _tokenSource.Token);
                }, _tokenSource.Token);
                _task.ContinueWith((t) =>
                {
                    // ワーカースレッドが計算を完了したあと、ContinueWith のタスクが呼ばれるまでの間に
                    // キャンセルしたときには t.IsCanceled が設定されないので、
                    // キャンセルされたかどうかは _measureStatus で判断します。
                    bool isCanceled = _measureStatus == MeasureStatus.Stopping;

                    _measureStatus = MeasureStatus.Stopped;
                    if (t.IsFaulted == true) // 計測失敗の場合は計測スタックに積まず再計測しない。
                    {
                        _measureStack.Remove(_currentLoudnessMeasureStackItem);
                        this.OnMeasured(new Sound[] { _currentLoudnessMeasureStackItem.Sound });
                    }
                    else if (isCanceled == true) // キャンセルの処理はここでする。
                    {
                        if (_cancelRetryFlag == false) // キャンセルされたサウンドを再び計測するか？
                        {
                            _measureStack.Remove(_currentLoudnessMeasureStackItem);
                        }
                        this.OnMeasured(new Sound[] { _currentLoudnessMeasureStackItem.Sound }, true);
                    }
                    else
                    {
                        _measureStack.Remove(_currentLoudnessMeasureStackItem);
                        this.OnMeasured(new Sound[] { _currentLoudnessMeasureStackItem.Sound }, new float[] { t.Result });
                    }
                    _cancelRetryFlag = true;
                    _currentLoudnessMeasureStackItem = null;
                    _tokenSource = null;
                    _task = null;
                }, taskScheduler); // 終了処理はメインスレッドで行うように、取得しておいたタスクスケジューラを渡す。
            }
            else // previewSound == null
            {
                _currentLoudnessMeasureStackItem = null;
            }
        }

        // <summary>
        // ファイルを監視に追加します。
        // </summary>
        private void AddFileWatch(string filePath, Component component)
        {
            // ファイル監視に追加
            this.FileWatcherService.Add(filePath, this.OnFileChanged);

            // ファイルパスからコンポーネントを取得できるように登録する
            if (_fileDictionary.ContainsKey(filePath) == false)
            {
                _fileDictionary.Add(filePath, new List<Component>());
            }

            _fileDictionary[filePath].Add(component);
        }

        // <summary>
        // ファイルを監視から削除します。
        // </summary>
        private void RemoveFileWatch(string filePath, Component component)
        {
            // ファイル監視から削除
            this.FileWatcherService.Remove(filePath, this.OnFileChanged);

            if (_fileDictionary.ContainsKey(filePath) == true)
            {
                _fileDictionary[filePath].Remove(component);
                if (_fileDictionary[filePath].Count <= 0)
                {
                    _fileDictionary.Remove(filePath);
                }
            }
        }

        // <summary>
        // ファイルを監視から削除します。（パスが不明になった場合）
        // </summary>
        private void RemoveFileWatch(Component component)
        {
            List<string> removePaths = new List<string>();

            foreach (var pair in _fileDictionary)
            {
                if (pair.Value.Contains(component) == true)
                {
                    // ファイル監視から削除
                    this.FileWatcherService.Remove(pair.Key, this.OnFileChanged);
                    pair.Value.Remove(component);
                    if (pair.Value.Count <= 0)
                    {
                        removePaths.Add(pair.Key);
                    }
                }
            }

            removePaths.ForEach((filePath) => _fileDictionary.Remove(filePath));
        }

        // <summary>
        // サウンドのパラメータが更新されたら呼ばれます。
        // </summary>
        private void OnParameterValueChanged(object sender, ParameterEventArgs e)
        {
            Component component = sender as Component;

            if (e.Key == ProjectParameterNames.FilePath)
            {
                // ファイルパスの変更の場合はファイル監視の変更があるか確認します。
                this.FilePathParameterValueChanged(component);
            }

            // 自動計測が無効の場合は何もしない。
            if (this.AutoMeasureLoudness == false)
            {
                return;
            }

            // ラウドネスに関係あるパラメータが変更されたので計測スタックに優先度「高」で積みます。
            if (component is StreamSoundBase == true && this.StreamSoundParameters.Contains(e.Key) == true)
            {
                this.MeasureHighPriority(component as StreamSoundBase);
            }
            else if (component is StreamSoundTrackBase == true && this.StreamSoundParameters.Contains(e.Key) == true)
            {
                this.MeasureHighPriority(component.Parent as StreamSoundBase);
            }
            else if (component is WaveSoundBase == true && this.WaveSoundParameters.Contains(e.Key) == true)
            {
                this.MeasureHighPriority(component as WaveSoundBase);
            }
            else if (component is SequenceSoundBase == true && this.SequenceSoundParameters.Contains(e.Key) == true)
            {
                this.MeasureHighPriority(component as SequenceSoundBase);
            }
            else if (component is SoundSetBankBase == true && e.Key == ProjectParameterNames.FilePath)
            {
                // サウンドセットバンクを参照しているシーケンスサウンドを探し計測スタックに優先度「高」で積みます。
                SoundSetBankBase soundSetBank = component as SoundSetBankBase;

                foreach (SequenceSoundBase sequence in this.SequenceSounds)
                {
                    foreach (ComponentReference componentReference in sequence.SoundSetBankReferences)
                    {
                        if (componentReference.TargetName == soundSetBank.Name)
                        {
                            this.MeasureHighPriority(sequence);
                        }
                    }
                }
            }
        }

        // <summary>
        // ファイルパスパラメータが変更された場合の処理
        // </summary>
        private void FilePathParameterValueChanged(Component component)
        {
            string filePath = (string)component.Parameters[ProjectParameterNames.FilePath].Value;

            if (_fileDictionary.ContainsKey(filePath) == true &&
                _fileDictionary[filePath].Contains(component) == true)
            {
                // ファイルパスの変更はないので何もしない。
                return;
            }

            // ファイルパスの変更があったので元のファイルを監視から削除する。
            this.RemoveFileWatch(component);
            // 現在のファイルを監視に追加する。
            this.AddFileWatch(filePath, component);
        }

        // <summary>
        // ファイルが更新されたら呼ばれます。
        // </summary>
        private void OnFileChanged(string filePath)
        {
            if (_fileDictionary.ContainsKey(filePath) == true)
            {
                foreach (Component component in _fileDictionary[filePath])
                {
                    this.OnParameterValueChanged(component, new ParameterEventArgs(ProjectParameterNames.FilePath, new FilePathParameterValue(filePath)));
                }
            }
        }

        // <summary>
        // 計測を開始する前にイベントを発生させます。
        // <param name="sounds">今から計測するサウンド</param>
        // </summary>
        private void OnMeasuring(IEnumerable<Sound> sounds)
        {
            this.Measuring?.Invoke(this, new MeasureLoudnessEventArgs(sounds));
        }

        // <summary>
        // 計測が終了したらイベントを発生させます。
        // </summary>
        // <param name="sounds">計測したサウンド</param>
        // <param name="sounds">計測した平均ラウドネス値</param>
        private void OnMeasured(IEnumerable<Sound> sounds, IEnumerable<float> integratedLoudnesses)
        {
            this.Measured?.Invoke(this, new MeasureLoudnessEventArgs(sounds, integratedLoudnesses));
        }

        // <summary>
        // 計測が（中断）終了したらイベントを発生させます。
        // </summary>
        // <param name="sounds">計測したサウンド</param>
        private void OnMeasured(IEnumerable<Sound> sounds, bool isCancel = false)
        {
            this.Measured?.Invoke(this, new MeasureLoudnessEventArgs(sounds, false, isCancel));
        }

        // <summary>
        // サウンドのハッシュ値を取得します。
        // </summary>
        public string GetHashCode(Sound sound)
        {
            if (sound is StreamSoundBase == true)
            {
                HashCode hash = this.GetParameterHashCode(sound, name => this.StreamSoundParameters.Contains(name));

                // ストリームサウンドのハッシュとトラックのハッシュを連結してハッシュ計算します。
                var bytes = hash.Value.Concat(
                    sound.Children
                    .Where<Component>(component => component.IsEnabled)
                    .Select<Component, HashCode>(component => this.GetParameterHashCode(component, name => this.StreamSoundParameters.Contains(name)))
                    .SelectMany<HashCode, byte>(hashCode => hashCode.Value));
                return new HashCode(HashAlgorithm.ComputeHash(bytes.ToArray())).ToString();
            }
            else if (sound is WaveSoundBase == true)
            {
                return this.GetParameterHashCode(sound, name => this.WaveSoundParameters.Contains(name)).ToString();
            }
            else if (sound is SequenceSoundBase == true)
            {
                return this.GetParameterHashCode(sound, name => this.SequenceSoundParameters.Contains(name)).ToString();
            }

            return string.Empty;
        }

        private HashCode GetParameterHashCode(Component component, Func<string, bool> filter)
        {
            Debug.Assert(component != null);
            Debug.Assert(filter != null);

            HashCode valueHash = HashCode.Empty;

            foreach (var key in component.Parameters.Keys)
            {
                // フィルタを通過できない場合は次のパラメータへ。
                if (filter != null && filter(key) == false)
                {
                    continue;
                }

                IParameterValue value = component.Parameters[key];

                var hash = value.GetParameterHashCode(HashAlgorithm, key, p => true);

                if (valueHash == HashCode.Empty)
                {
                    valueHash = hash;
                }
                else
                {
                    if (hash != HashCode.Empty)
                    {
                        valueHash ^= hash;
                    }
                }
            }

            if (valueHash == HashCode.Empty)
            {
                return valueHash;
            }

            // キーのバイト列と値のバイト列を連結してハッシュ計算します。
            var bytes = Encoding.Unicode.GetBytes(component.Name).Concat(valueHash.Value);

            // シーケンスサウンドの場合は、最大計測時間をハッシュに含めます。
            if (component is SequenceSoundBase == true)
            {
                Debug.Assert(this.SequenceMaxMeasureDuration.HasValue == true);

                bytes = bytes.Concat(BitConverter.GetBytes(this.SequenceMaxMeasureDuration.Value));
            }

            return new HashCode(HashAlgorithm.ComputeHash(bytes.ToArray()));
        }
    }
}
