﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Collections;
using System.Net;
using System.Threading;
//using FrameworkResources = AnimSoundMakerGeneric.Resources;

namespace NintendoWare.Generic.Preview.MCS.Tool
{
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Error;
    //using NW4F.ProtocolSound;
    ///using NintendoWare.Generic.Preview.MCS.Error;

    /// <summary>
    /// ツール接続
    /// <para>
    /// ツールからビューアに送るパケットを処理する接続です。
    /// </para>
    /// </summary>
    public class MCSToolConnection : MCSConnection
    {
        #region ** 定数

        //---------------------------------------------------------------------
        // 定数
        //---------------------------------------------------------------------
        // パケットがない時のインターバル
        private const int intervalNothingPacket = 50;
        // パケットがある時のインターバル
        private const int intervalExistPacket = 50;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MCSToolConnection() { }

        /// <summary>
        /// チャンネルの取得
        /// </summary>
        public override uint Channel
        {
#if ANIMATION
            get { return _baseChannel + 0x2; }
#else
            get { return _baseChannel + 0x0; }
#endif
        }

        //---------------------------------------------------------------------
        // 接続関係
        //---------------------------------------------------------------------
        /// <summary>
        /// 接続
        /// </summary>
        public override bool Connect(IPEndPoint ipEndPoint, string hostName)
        {
            Debug.Assert(_packetQueue.Count == 0);
            bool result = base.Connect(ipEndPoint, hostName);
            if (result)
            {
                // プロトコルのバージョン確認
                VersionPacket.Send();

                // リセット
                ResetPacket.Send();
            }
            return result;
        }

        /// <summary>
        /// 接続を閉じる
        /// </summary>
        public override void Disconnect(int timeout)
        {
            DisconnectingFlag = true;
            base.Disconnect(timeout);
            lock (_packetQueue) { _packetQueue.Clear(); }
        }

        //---------------------------------------------------------------------
        // パケット関係
        //---------------------------------------------------------------------
        /// <summary>
        /// パケット送信
        /// </summary>
        public override void SendPacket(CommPacket commPacket)
        {
            ToolPacket packet = commPacket as ToolPacket;

            // キューにパケットを追加する
            lock (_packetQueue)
            {
                if (_packetQueue.Count > 512)
                {
                    return;
                }

                // パケット圧縮
                if (_packetQueue.Count > 0)
                {
                    if (_lastPacket.Compress(packet)) { return; }
                }

                _packetQueue.Enqueue(packet);
                _lastPacket = packet;

                _queuingEvent.Set();

                // Debug.WriteLine( string.Format( "+ Queue : {0:C}", _packetQueue.Count ) );
            }

            //ShowMessage("Tool Packet Queue {0}", _packetQueue.Count);
        }

        /// <summary>
        /// パケット数の取得
        /// </summary>
        public int GetPacketCount()
        {
            lock (_packetQueue) { return _packetQueue.Count; }
        }

        /// <summary>
        /// パケット取得
        /// </summary>
        public ToolPacket GetPacket()
        {
            // キューからパケットを取り出す
            ToolPacket packet = null;
            lock (_packetQueue)
            {
                if (_packetQueue.Count > 0)
                {
                    packet = (ToolPacket)_packetQueue.Dequeue();
                    _queuingEvent.Reset();

                    //					Debug.WriteLine( string.Format( "- Dequeue : {0:N}", _packetQueue.Count ) );
                }
            }
            return packet;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// メインループ
        /// </summary>
        public override void MainLoop()
        {
            ToolPacket packet = GetPacket();
            if (packet == null)
            {
                Stopwatch stopwatch = new Stopwatch();
                stopwatch.Start();
                _queuingEvent.WaitOne(intervalNothingPacket, false);
                stopwatch.Stop();

                //				Debug.WriteLine( string.Format( "* WaitTime : {0:N}", stopwatch.ElapsedMilliseconds ) );
                return;
            }
            // パケットの書き込み
            CommPacketHeader packetHeader = packet.Write(Writer);
            Writer.Flush();
            //ShowMessage( "  Tool → " + packet.ToString() );
            if (packet.Reply)
            {
                CommPacketHeader replyHeader = new CommPacketHeader(Reader);
                //ShowMessage( "       ← " + replyHeader.ToString() );

                if (!replyHeader.Error)
                {
                    // 返信処理
                    Debug.Assert(packet.Category == replyHeader.Category);
                    Debug.Assert(packet.Command == replyHeader.Command);
                    Debug.Assert(packet.Error == replyHeader.Error);
                    packet.Read(Reader, replyHeader);
                }
                else
                {
                    // エラー処理
                    ErrorPacket errorPacket = ErrorPacket.Create(replyHeader, packet);
                    if (errorPacket != null)
                    {
                        errorPacket.Read(Reader, replyHeader);
                    }
                    else
                    {
                        // 未知のエラーパケットはメッセージを出して読み飛ばす
                        ShowMessage(
                            //FrameworkResources.MessageResource.Message_ReceiveUnknownPacket +
                            "Recieve unkown packet {0}" +
                            "\r\n" + replyHeader.ToString());
                        Reader.ReadBytes((int)replyHeader.Size);
                    }
                }
            }
            else
            {
                // パケットを休み無く送り続けるとソケットのバッファがあふれる
                //				Thread.Sleep( 0/*intervalExistPacket*/ );
            }
        }

        #region ** フィールド

        //---------------------------------------------------------------------
        // メンバ
        //---------------------------------------------------------------------
        // パケットキュー
        private Queue _packetQueue = new Queue();

        // 最後のパケット
        private ToolPacket _lastPacket = null;

        // キューイベント
        private ManualResetEvent _queuingEvent = new ManualResetEvent(false);

        #endregion
    }
}
