﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using NW4R.ProtocolSound;

namespace NintendoWare.SoundMaker.Preview.MCS.Tool.Sound
{

    /// <summary>
    /// MCSツールサウンドパケット
    /// </summary>
    public abstract class MCSToolSoundPacket : MCSToolPacket
    {
        public MCSToolSoundPacket()
        {
            Debug.Assert( Category == Constants.CATEGORY_SOUND );
        }
    }

    #region ** 同期パケットクラス

    #region ** イベントクラス

    public class ReplyEventArgsT<EventArgsType> : EventArgs
        where EventArgsType : EventArgs
    {
        #region ** イベントハンドラ デリゲート

        public delegate void EventHandler( object sender, EventArgsType e );

        #endregion

        #region ** パラメータ

        private uint   _errorCode;
        private object _userData;

        #endregion

        public ReplyEventArgsT( uint errorCode, object userData )
        {
            _errorCode = errorCode;
            _userData = userData;
        }

        #region ** プロパティ

        public bool Failed
        {
            get { return ( 0 != ( Constants.MCSSOUND_ERRORFLAG & _errorCode ) ); }
        }

        public uint ErrorCode
        {
            get { return _errorCode; }
        }

        public object UserData
        {
            get { return _userData; }
        }

        #endregion
    }

    public class ReplyEventArgs : ReplyEventArgsT<ReplyEventArgs>
    {
        public ReplyEventArgs( uint errorCode, object userData ) : base( errorCode, userData ) { }
    }

    #endregion

    /// <summary>
    /// MCSツールサウンド同期パケット
    /// </summary>
    public abstract class MCSToolSoundSyncPacketT<EventArgsType> : MCSToolSoundPacket, IDisposable
        where EventArgsType : ReplyEventArgsT<EventArgsType>
    {
        #region ** パラメータ

        private ReplyEventArgsT<EventArgsType>.EventHandler _handler;
        private object                                      _userData;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="handler">結果通知用デリゲート</param>
        /// <param name="userData">ユーザデータ</param>
        public MCSToolSoundSyncPacketT( ReplyEventArgsT<EventArgsType>.EventHandler handler, object userData )
        {
            _handler = handler;
            _userData = userData;
        }

        #region ** プロパティ

        protected object UserData
        {
            get { return _userData; }
        }

        #endregion

        #region ** イベント

        public event ReplyEventArgsT<EventArgsType>.EventHandler ReplyRecieved;

        #endregion

        #region ** イベントハンドラ

        protected virtual MCSPacketHeader OnWrite( ProtocolSoundWriter writer ) { return base.Write( writer ); }
        protected virtual EventArgsType OnRead( ProtocolSoundReader reader, MCSPacketHeader header ) { return null; }
        protected virtual EventArgsType OnError( Error.Sound.MCSErrorPacketRecievedEventArgs e ) { return null; }

        protected virtual void OnReply( EventArgsType e )
        {
            try {
                if ( null != ReplyRecieved ) {
                    ReplyRecieved( this, e );
                }
            }
            finally {
                ClearHandler();
            }
        }

        private void OnErrorPacketReceived( object sender, Error.Sound.MCSErrorPacketRecievedEventArgs e )
        {
            // エラーパケットを受け取るためのイベント設定解除
            MCS.Error.Sound.MCSSoundErrorPacket.PacketRecieved -= OnErrorPacketReceived;

            EventArgsType errorEvent = OnError( e );

            if ( null != errorEvent ) {
                OnReply( errorEvent );
            }
        }

        #endregion

        #region ** メソッド

        private void SetHandler()
        {
            if ( null != _handler ) {
                ReplyRecieved -= _handler;
                ReplyRecieved += _handler;
            }
        }

        private void ClearHandler()
        {
            if ( null != _handler ) {
                ReplyRecieved -= _handler;
                _handler = null;
            }

        }

        #endregion

        #region ** メソッドのオーバーライド

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public sealed override void Read( ProtocolSoundReader reader, MCSPacketHeader header )
        {
            // エラーパケットを受け取るためのイベント設定解除
            MCS.Error.Sound.MCSSoundErrorPacket.PacketRecieved -= OnErrorPacketReceived;

            EventArgsType recvEvent = OnRead( reader, header );
            Debug.Assert( null != recvEvent );

            OnReply( recvEvent );
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public sealed override MCSPacketHeader Write( ProtocolSoundWriter writer )
        {
            MCSPacketHeader header = OnWrite( writer );

            // エラーパケットを受け取るためのイベント設定
            MCS.Error.Sound.MCSSoundErrorPacket.PacketRecieved += OnErrorPacketReceived;

            // 返信イベントを設定する
            SetHandler();

            return header;
        }

        #endregion

        #region ** IDisposable の実装

        void IDisposable.Dispose()
        {
            ClearHandler();
        }

        #endregion
    }

    public abstract class MCSToolSoundSyncPacket : MCSToolSoundSyncPacketT<ReplyEventArgs>
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="handler">結果通知用デリゲート</param>
        /// <param name="userData">ユーザデータ</param>
        public MCSToolSoundSyncPacket( ReplyEventArgs.EventHandler handler, object userData ) : base( handler, userData ) { }
    }

    #endregion

    /// <summary>
    /// サウンドアーカイブを読み込むパケットです。
    /// </summary>
    public class MCSLoadSarSoundPacket : MCSToolSoundSyncPacket
    {
        #region ** パラメータ

        private string _soundArchiveFileName;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="modelFileName">サウンドアーカイブのファイルパス</param>
        /// <param name="handler">結果通知用デリゲート</param>
        /// <param name="userData">ユーザデータ</param>
        public MCSLoadSarSoundPacket( string soundArchiveFileName, ReplyEventArgs.EventHandler handler, object userData )
            : base( handler, userData )
        {
            _soundArchiveFileName = soundArchiveFileName;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_LOADSAR; }
        }

        /// <summary>
        /// データサイズ
        /// </summary>
        public override ushort Size
        {
            get { return (ushort)( ToolSoundLoadSar.StructSize ); }
        }

        #endregion

        #region ** イベントハンドラのオーバーライド

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        protected override MCSPacketHeader OnWrite( ProtocolSoundWriter writer )
        {
            Debug.Assert( null != writer );

            MemoryStream memStream = new MemoryStream();
            ProtocolSoundWriter memWriter = new ProtocolSoundWriter( memStream );


            // パケットヘッダを出力する
            MCSPacketHeader header = CreateHeader();
            header.Write( memWriter );

            // パラメータを出力する
            ToolSoundLoadSar parameter = new ToolSoundLoadSar();
            EncodeFilePath( _soundArchiveFileName, parameter.pathName );

            parameter.Write( memWriter );


            writer.Write( memStream.GetBuffer(), 0, (int)memStream.Length );

            return header;
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        protected override ReplyEventArgs OnRead( ProtocolSoundReader reader, MCSPacketHeader header )
        {
            ToolSoundLoadSarReply reply = new ToolSoundLoadSarReply();
            reply.Read( reader );

            ShowMessage( string.Format(
                         "SoundArchive loaded. \"{0}\" ID 0x{1:x8}",
                         Path.GetFileNameWithoutExtension( _soundArchiveFileName ), reply.id ) );

            return new ReplyEventArgs( Constants.MCSSOUND_S_OK, UserData );
        }

        protected override ReplyEventArgs OnError( Error.Sound.MCSErrorPacketRecievedEventArgs e )
        {
            return new ReplyEventArgs( e.ErrorPacket.ErrorInfo.errorCode, UserData );
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// アニメーションプレイヤー作成パケットを送信します。
        /// </summary>
        public static bool Send( string soundArchiveFilePath )
        {
            return Send( soundArchiveFilePath, null );
        }
        public static bool Send( string soundArchiveFilePath, ReplyEventArgs.EventHandler handler )
        {
            return Send( soundArchiveFilePath, handler, null );
        }
        public static bool Send( string soundArchiveFilePath, ReplyEventArgs.EventHandler handler, object userData )
        {
#if false
            if ( !MCSManager.IsConnected ) { return false; }
#else
            if ( !MCSSound.Instance.IsConnected ) { return false; }
#endif

            SendPacket( new MCSLoadSarSoundPacket( soundArchiveFilePath, handler, userData ) );
            return true;
        }

        #endregion
    }

    ///
    public class MCSCloseSarSoundPacket : MCSToolSoundSyncPacket
    {
        public MCSCloseSarSoundPacket( ReplyEventArgs.EventHandler handler, object userData)
            : base( handler, userData )
        {
        }

        ///
        public static bool Send()
        {
            if ( !( MCSManager.IsConnecting || MCSManager.IsConnected ) ) { return false; }

            //MCSManager.SendPacket( new MCSCloseSarSoundPacket( null, null));
            bool replied = false;
            bool success = false;

            MCSCloseSarSoundPacket packet = new MCSCloseSarSoundPacket
                ( delegate( object sender, ReplyEventArgs e)
                        {
                            replied = true;
                            success = !e.Failed;
                        },
                  null);
            MCSManager.SendPacket( packet);

            //
            while (replied == false)
            {
                System.Threading.Thread.Sleep(1);
            }
            return success;
        }

        ///
        protected override MCSPacketHeader OnWrite( ProtocolSoundWriter writer )
        {
            Debug.Assert( null != writer );

            MCSPacketHeader header = CreateHeader();
            header.Write( writer );
            return header;
        }

        ///
        protected override ReplyEventArgs OnRead( ProtocolSoundReader reader, MCSPacketHeader header )
        {
            ToolSoundCloseSarReply reply = new ToolSoundCloseSarReply();
            reply.Read( reader );

            ShowMessage( string.Format( "SoundArchive closed. Result 0x{0:x8}",
                                        reply.result));

            return new ReplyEventArgs
                ( reply.result != 0 ? Constants.MCSSOUND_S_OK : Constants.MCSSOUND_E_FAILED
                  , UserData );
        }

        ///
        protected override ReplyEventArgs OnError( Error.Sound.MCSErrorPacketRecievedEventArgs e )
        {
            return new ReplyEventArgs( e.ErrorPacket.ErrorInfo.errorCode, UserData );
        }

        /// <summary>
        /// メッセージタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_CLOSESAR; }
        }
    }

    /// <summary>
    /// 強制的にサウンドモードに切り替えるパケットです。
    /// </summary>
    public class MCSFocusSoundPacket : MCSToolSoundPacket
    {
        public static MCSFocusSoundPacket Send()
        {
            if ( !MCSManager.IsConnected ) { return null; }
            MCSFocusSoundPacket packet = new MCSFocusSoundPacket();
            MCSManager.SendPacket( packet );
            return packet;
        }
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_FOCUS; }
        }
    }

    /// <summary>
    /// 呼びかけて返信を要求するパケットです。
    /// 返信があると、ReadFlag が true になります。
    /// </summary>
    public class MCSEchoSoundPacket : MCSToolSoundPacket
    {
        public bool ReadFlag;
        public static MCSEchoSoundPacket Send()
        {
            if ( !MCSManager.IsConnected ) {
                return null;
            }
            MCSEchoSoundPacket packet = new MCSEchoSoundPacket();
            SendPacket( packet );
            return packet;
        }
        public override void Read( ProtocolSoundReader reader, MCSPacketHeader header )
        {
            ReadFlag = true;
        }
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_ECHO; }
        }
    }

    /// <summary>
    /// 全サウンドを停止するパケットです。
    /// </summary>
    public class MCSAllStopSoundPacket : MCSToolSoundPacket
    {
        public static void Send()
        {
            if ( !MCSManager.IsConnected ) {
                return;
            }
            SendPacket( new MCSAllStopSoundPacket() );
        }
        /// <summary>
        /// パケットタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_ALLSTOP; }
        }
    }

    /// <summary>
    /// サウンドシステム設定パケット
    /// </summary>
    public class MCSSetSoundSystemParametersPacket : MCSToolSoundPacket
    {
        private ToolSoundOutputMode _outputMode = ToolSoundOutputMode.OutputMode_Invalid;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="outputMode">出力モード</param>
        public MCSSetSoundSystemParametersPacket( ToolSoundOutputMode outputMode )
        {
            _outputMode = outputMode;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_SET_SYSTEMPARAMETERS; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get
            {
                return (ushort)ToolSoundSystemParameters.StructSize;
            }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// サウンド設定パケットを送信します。
        /// </summary>
        /// <param name="outputMode">出力モード</param>
        public static void Send( ToolSoundOutputMode outputMode )
        {
            if ( !MCSManager.IsConnected ) { return; }
            SendPacket( new MCSSetSoundSystemParametersPacket( outputMode ) );
        }

        #endregion

        #region ** メソッドのオーバーライド

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read( ProtocolSoundReader reader, MCSPacketHeader header )
        {
            // Write のみ
            Debug.Assert( false );
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override MCSPacketHeader Write( ProtocolSoundWriter writer )
        {
            Debug.Assert( null != writer );

            MemoryStream memStream = new MemoryStream();
            ProtocolSoundWriter memWriter = new ProtocolSoundWriter( memStream );


            // パケットヘッダを出力する
            MCSPacketHeader header = CreateHeader();
            header.Write( memWriter );

            // パラメータを出力する
            ToolSoundSystemParameters parameters = new ToolSoundSystemParameters();
            parameters.outputMode = _outputMode;
            parameters.Write( memWriter );

            writer.Write( memStream.GetBuffer(), 0, (int)memStream.Length );

            return header;
        }

        #endregion
    }
}
