﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Text;
using NW4R.ProtocolSound;

namespace NintendoWare.SoundMaker.Preview.MCS
{
    /// <summary>
    /// MCSパケット
    /// <para>
    /// MCS通信で送受信されるパケットのベースクラスです。
    /// </para>
    /// </summary>
    public abstract class MCSPacket
    {
        /// <summary>
        /// ヘッダの作成
        /// </summary>
        public MCSPacketHeader CreateHeader()
        {
            return new MCSPacketHeader(PacketType, Size);
        }

        /// <summary>
        /// 読み込み
        /// </summary>
        public virtual void Read(
            ProtocolSoundReader reader, MCSPacketHeader header)
        {
            // デフォルトの動作は内容無しのパケットを想定
            Debug.Assert(header.Size == 0);
        }

        /// <summary>
        /// 書き出し
        /// </summary>
        public virtual MCSPacketHeader Write(ProtocolSoundWriter writer)
        {
            // デフォルトの動作は内容無しのパケットを想定
            Debug.Assert(Size == 0);
            MCSPacketHeader header = CreateHeader();
            header.Write(writer);
            return header;
        }

        /// <summary>
        /// 文字列変換
        /// </summary>
        public override string ToString()
        {
            return this.GetType().Name + " " + CreateHeader().ToString();
        }

        //---------------------------------------------------------------------
        // プロパティ
        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージタイプ
        /// </summary>
        public abstract ushort PacketType{ get; }

        /// <summary>
        /// パケットのカテゴリ
        /// </summary>
        public ushort Category
        {
            get
            {
                return (ushort)(PacketType & Constants.HEADER_CATEGORY_MASK);
            }
        }

        /// <summary>
        /// パケットのカテゴリ内のコマンド
        /// </summary>
        public ushort Command
        {
            get
            {
                return (ushort)(PacketType & Constants.HEADER_COMMAND_MASK);
            }
        }

        /// <summary>
        /// 返信フラグ
        /// </summary>
        public bool Reply
        {
            get
            {
                return ((PacketType & Constants.HEADER_REPLY_FLAG) != 0);
            }
        }

        /// <summary>
        /// エラーフラグ
        /// </summary>
        public bool Error
        {
            get{ return ((PacketType & Constants.HEADER_ERROR_FLAG) != 0); }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public virtual ushort Size{ get{ return 0; } }

        //---------------------------------------------------------------------
        // 非公開メンバ
        //---------------------------------------------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        protected MCSPacket(){}

        //---------------------------------------------------------------------
        /// <summary>
        /// ファイルパスのエンコード
        /// </summary>
        protected void EncodeFilePath(string filePath, byte[] byteArray)
        {
            if(Encoding.Default.GetByteCount(filePath) > byteArray.Length)
            {
                ShowError( "ファイルパスが長すぎます" );
            }
            Encoding.Default.GetBytes(
                filePath, 0, filePath.Length, byteArray, 0);
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージ
        /// </summary>
        protected static void ShowMessage( string text )
        {
            MCSManager.Console.WriteMCSMessage( text );
        }

        /// <summary>
        /// エラー
        /// </summary>
        protected static void ShowError( string text )
        {
            MCSManager.Console.WriteMCSError( text );
            throw new ApplicationException( text );
        }

    }
}
