﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Preview.Htcs.Tool.Sound
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Text;

    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NW4F.ProtocolSound;
    using NW4R.ProtocolSound;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;


    /// <summary>
    /// サウンド情報取得パケット
    /// </summary>
    public class HtcsGetSoundInformationPacket : ToolSoundPacket
    {
        private PreviewPlayerInformationCollectionCommon _informations;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="previewPlayers">対象プレビュープレイヤーのコレクション</param>
        public HtcsGetSoundInformationPacket(PreviewPlayerInformationCollectionCommon informations)
        {
            if (null == informations) { throw new ArgumentNullException("informations"); }
            _informations = informations;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_GET_SOUNDINFORMATIONS; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get { return 0; }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// サウンド情報の取得パケットを送信します。
        /// </summary>
        public static void Send(PreviewPlayerInformationCollectionCommon informations)
        {
            if (!CommManager.Instance.IsConnected) { return; }
            SendPacket(new HtcsGetSoundInformationPacket(informations));
        }

        #endregion

        #region ** メソッドのオーバーライド

        /// <summary>
        /// 送信パケットを圧縮します。
        /// </summary>
        public override bool Compress(ToolPacket nextPacket)
        {
            return (nextPacket is HtcsGetSoundInformationPacket);
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            throw new Exception("unexpected call function");
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write(writer);

            // この１行は仮実装
            // Send用ToolPacket, Recv用ViewerPacketとして１つのクラスにまとめる
            Htcs.Viewer.HtcsGetSoundInformationPacket.Received = false;

            return header;
        }

        #endregion
    }


    /// <summary>
    /// プレビュープレイヤーパラメータ設定パケット
    /// </summary>
    public class HtcsSetPreviewPlayerParameterPacket : ToolSoundPacket
    {
        private PreviewPlayer _previewPlayer;

        /// <summary>
        /// プレビュープレイヤーパラメータ設定パケットを送信します。
        /// </summary>
        /// <param name="previewPlayer">対象プレビュープレイヤー</param>
        public static void Send(PreviewPlayer previewPlayer)
        {
            if (!CommManager.Instance.IsConnected) { return; }
            SendPacket(new HtcsSetPreviewPlayerParameterPacket(previewPlayer));
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="previewPlayer">対象プレビュープレイヤー</param>
        public HtcsSetPreviewPlayerParameterPacket(PreviewPlayer previewPlayer)
        {
            if (null == previewPlayer) { throw new ArgumentNullException("previewPlayer"); }
            _previewPlayer = previewPlayer;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_SET_PREVIEWPLAYERPARAMETER; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get
            {
                return (ushort)ToolSoundPreviewPlayerParameter.StructSize;
            }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// 送信パケットの圧縮
        /// </summary>
        public override bool Compress(ToolPacket nextPacket)
        {
            if (!(nextPacket is HtcsSetPreviewPlayerParameterPacket)) { return false; }
            return (_previewPlayer == (nextPacket as HtcsSetPreviewPlayerParameterPacket)._previewPlayer);
        }

        #endregion

        #region ** メソッドのオーバーライド

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            // Write のみ
            Debug.Assert(false, "Don't call this method");
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write(writer);

            Debug.Assert(null != _previewPlayer, "Preview player is null");

            //
            // パラメータを出力する
            //
            // コンテナヘッダ
            ToolSoundPreviewPlayerParameter parameter = new ToolSoundPreviewPlayerParameter();

            parameter.previewPlayerIndex = _previewPlayer.Index;

            parameter.volume = _previewPlayer.Paramters.Volume;
            parameter.pitch = _previewPlayer.Paramters.Pitch;

            parameter.pan = _previewPlayer.Paramters.Pan;
            parameter.surroundPan = _previewPlayer.Paramters.SurroundPan;

            parameter.mainPan = _previewPlayer.Paramters.MainPan;
            parameter.mainSurroundPan = _previewPlayer.Paramters.MainSurroundPan;

            parameter.drcPan = _previewPlayer.Paramters.DrcPan;
            parameter.drcSurroundPan = _previewPlayer.Paramters.DrcSurroundPan;

            parameter.lpfFrequency = _previewPlayer.Paramters.LpfFrequency;
            parameter.biquadType = BiquadTypeConverter.ConvertToToolSoundBiquadType(_previewPlayer.Paramters.BiquadType);
            parameter.biquadValue = _previewPlayer.Paramters.BiquadValue;
            parameter.startOffsetTime = _previewPlayer.Paramters.StartOffsetTime;

            parameter.mainOutEnabled = _previewPlayer.Paramters.MainOut.Enabled ? 1 : 0;
            parameter.drcOutEnabled = _previewPlayer.Paramters.DrcOut.Enabled ? 1 : 0;
            parameter.remote0OutEnabled = _previewPlayer.Paramters.RemoteOut[0].Enabled ? 1 : 0;
            parameter.remote1OutEnabled = _previewPlayer.Paramters.RemoteOut[1].Enabled ? 1 : 0;
            parameter.remote2OutEnabled = _previewPlayer.Paramters.RemoteOut[2].Enabled ? 1 : 0;
            parameter.remote3OutEnabled = _previewPlayer.Paramters.RemoteOut[3].Enabled ? 1 : 0;

            parameter.mainOutVolume = _previewPlayer.Paramters.MainOut.Volume;
            parameter.drcOutVolume = _previewPlayer.Paramters.DrcOut.Volume;
            parameter.remote0OutVolume = _previewPlayer.Paramters.RemoteOut[0].Volume;
            parameter.remote1OutVolume = _previewPlayer.Paramters.RemoteOut[1].Volume;
            parameter.remote2OutVolume = _previewPlayer.Paramters.RemoteOut[2].Volume;
            parameter.remote3OutVolume = _previewPlayer.Paramters.RemoteOut[3].Volume;

            parameter.mainSend = _previewPlayer.Paramters.MainSend;
            parameter.effectAuxASend = _previewPlayer.Paramters.EffectAuxASend;
            parameter.effectAuxBSend = _previewPlayer.Paramters.EffectAuxBSend;
            parameter.effectAuxCSend = _previewPlayer.Paramters.EffectAuxCSend;

            parameter.mainMainSend = _previewPlayer.Paramters.MainMainSend;
            parameter.mainEffectAuxASend = _previewPlayer.Paramters.MainEffectAuxASend;
            parameter.mainEffectAuxBSend = _previewPlayer.Paramters.MainEffectAuxBSend;
            parameter.mainEffectAuxCSend = _previewPlayer.Paramters.MainEffectAuxCSend;

            parameter.drcMainSend = _previewPlayer.Paramters.DrcMainSend;
            parameter.drcEffectAuxASend = _previewPlayer.Paramters.DrcEffectAuxASend;
            parameter.drcEffectAuxBSend = _previewPlayer.Paramters.DrcEffectAuxBSend;
            parameter.drcEffectAuxCSend = _previewPlayer.Paramters.DrcEffectAuxCSend;

            parameter.fadeInFrames = (int)_previewPlayer.Paramters.FadeInFrames;
            parameter.fadeOutFrames = (int)_previewPlayer.Paramters.FadeOutFrames;
            parameter.pauseFadeFrames = (int)_previewPlayer.Paramters.PauseFadeFrames;

            for (int i = 0; i < PreviewPlayer.Parameters.TracksCountDefault; i++)
            {
                parameter.tracksMute[i] = !(_previewPlayer.Paramters.TracksParameter[i].Enabled);
                parameter.tracksVolume[i] = _previewPlayer.Paramters.TracksParameter[i].Volume;
            }

            parameter.Write(writer);

            Htcs.Viewer.HtcsSetPreviewPlayerParameterPacket.Received = false;

            return header;
        }

        #endregion
    }
}
