﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Generic.Preview.Htcs
{
    using System;
    using System.IO;
    using System.Diagnostics;
    using System.Net;
    using System.Net.Sockets;
    using System.Threading;
    using System.Text;

    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using Nintendo.Alto.Foundation.Communications;

    /// <summary>
    /// Htcs接続
    /// <para>
    /// ツール接続、ビューア接続、Ping接続の親クラスです。Htcs接続の共通部分を
    /// 実装しています。実際に別スレッドで通信を行う部分の実装は、内部クラスの
    /// CommunicationThreadクラスにて行っています。
    /// </para>
    /// </summary>
    public abstract class HtcsConnection : CommConnection
    {
        //---------------------------------------------------------------------
        // 定数
        //---------------------------------------------------------------------
        protected const uint _baseChannel =
        (((uint)'S' << 8) + (uint)'D') + 0x8000;

        // 通信
        private volatile HtcsCommunicationThread _communication = null;

        //---------------------------------------------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public HtcsConnection() { }

        /// <summary>
        /// メインループ
        /// </summary>
        public abstract void MainLoop();

        /// <summary>
        /// チャンネルの取得
        /// </summary>
        public abstract uint Channel { get; }

        /// <summary>
        /// 文字列変換
        /// </summary>
        public override string ToString() { return GetType().Name; }

        //---------------------------------------------------------------------
        // 接続関係
        //---------------------------------------------------------------------
        public override bool Connect()
        {
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public IComEndPoint ComEndPoint
        {
            get;
            set;
        }

        public object Port
        {
            get;
            set;
        }

        /// <summary>
        /// 接続
        /// </summary>
        public virtual bool Connect(IPEndPoint ipEndPoint, string hostName)
        {
            Debug.Assert(_communication == null);

            _communication = new HtcsCommunicationThread(this, hostName, 0, 0);


            if (_communication.Connect(ComEndPoint, Port) == false)
            {
                _communication = null;
                return false;
            }
            ShowMessage("通信開始 : " + _communication);
            return true;
        }

        /// <summary>
        /// 接続中か
        /// </summary>
        public override bool IsConnected { get { return (_communication != null); } }

        /// <summary>
        /// 切断
        /// </summary>
        public override void Disconnect()
        {
            Disconnect(Timeout.Infinite);
        }

        /// <summary>
        /// 切断
        /// </summary>
        public override void Disconnect(int timeout)
        {
            if (!IsConnected) { return; }
            string name = _communication.ToString();
            _communication.Disconnect(timeout);
            _communication = null;
            ShowMessage("通信停止 : " + name);
        }

        /// <summary>
        /// 終了待ち
        /// </summary>
        public bool IsWantToExit
        {
            get
            {
                if (_communication.DisconnectingFlag)
                    return true;
                if (!IsConnected)
                    return false;
                return _communication.IsWantToExit;
            }
        }

        /// <summary>
        /// 切断中
        /// </summary>
        public bool DisconnectingFlag
        {
            get
            {
                if (_communication != null)
                {
                    return _communication.DisconnectingFlag;
                }
                else
                {
                    return false;
                }
            }
            set
            {
                if (_communication != null)
                {
                    _communication.DisconnectingFlag = value;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void SendPacket(CommPacket packet)
        {
            Debug.Assert(false, "SendPacket is not implimented");
        }

        //---------------------------------------------------------------------
        // 非公開メンバ
        //---------------------------------------------------------------------
        /// <summary>
        /// メッセージ
        /// </summary>
        protected void ShowMessage(string text)
        {
#if HtcsDEBUG
           HtcsManager.Console.WriteHtcsMessage( text );
#endif
            Debug.WriteLine(text);
        }

        /// <summary>
        /// エラー
        /// </summary>
        protected void ShowError(string text)
        {
#if HtcsDEBUG
           HtcsManager.Console.WriteHtcsError( text );
#endif
            Debug.WriteLine(text);
        }


        /// <summary>
        /// リーダ
        /// </summary>
        protected ProtocolSoundReader Reader
        {
            get { return _communication.Reader; }
        }

        /// <summary>
        /// ライタ
        /// </summary>
        protected ProtocolSoundWriter Writer
        {
            get { return _communication.Writer; }
        }

        /// <summary>
        ///
        /// </summary>
        protected int ReceiveTimeout
        {
            set
            {
                this._communication.ReceiveTimeout = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected int SendTimeout
        {
            set
            {
                this._communication.SendTimeout = value;
            }
        }

        //=====================================================================
        // 通信スレッド内部クラス
        //=====================================================================
        #region HtcsCommunicationThread
        /// <summary>
        /// 通信スレッド
        /// <para>
        /// スレッド実行によるソケット通信を行うクラスです。接続時にHtcsチャンネルの
        /// 初期化も行います。
        /// </para>
        /// </summary>
        protected class HtcsCommunicationThread
        {
            //-----------------------------------------------------------------
            // 定数
            //-----------------------------------------------------------------
            /// <summary>Htcsデフォルトタイムアウト(ms)</summary>
            private const int HtcsDefaultTimeout = 5000;
            /// <summary>Htcsチャンネルオープンタイムアウト(ms)</summary>
            private const int HtcsChannelOpenTimeout = 5000;
            /// <summary>スレッド終了待ち時間(ms)</summary>
            private const int WaitForThreadExitTime = 100;

            //-----------------------------------------------------------------
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public HtcsCommunicationThread(HtcsConnection connection,
                string hostName, int port, uint channel)
            {
                Debug.Assert(connection != null);
                _connection = connection;
                _name = _connection + " " + hostName + ":" + port +
                    " channel 0x" + channel.ToString("x");
            }

            //-----------------------------------------------------------------
            /// <summary>
            /// 接続
            /// </summary>
            public bool Connect(IComEndPoint comEndPoint, object port)
            {
                Debug.Assert((this.comChannel == null) && (_writer == null) &&
                              (_reader == null) && (_thread == null));

                // TCP/IP接続
                try
                {
                    IComChannel result = null;

                    using (var timer = new Timer(
                        state => comEndPoint.Disconnect(),
                        null,
                        3000,
                        Timeout.Infinite))
                    {
                        while (true)
                        {
                            lock (comEndPoint)
                            {
                                if (!comEndPoint.IsConnected)
                                {
                                    break;
                                }

                                result = comEndPoint.OpenChannel(port);
                            }

                            if (result != null)
                            {
                                break;
                            }
                            else
                            {
                                Thread.Sleep(10);
                            }
                        }
                    }

                    this.comChannel = result;
                    if (this.comChannel == null)
                    {
                        return false;
                    }

                    Stream stream = null;
                    if (this.comChannel is HostIOChannel)
                    {
                        HostIOChannel channel = this.comChannel as HostIOChannel;
                        stream = channel.TcpClient.GetStream();
                    }
                    else
                    {
                        stream = this.comChannel.GetStream();
                    }

                    this._writer = new HtcsProtocolSoundWriter(stream);
                    this._reader = new HtcsProtocolSoundReader(stream);
                }
                catch (Exception exception)
                {
                    ShowError(string.Format(
                        "TCP/IP接続に失敗しました。{0}\r\n{1}",
                        ToString(), exception.Message));
                    Disconnect(0);
                    return false;
                }


                // スレッドの開始
                try
                {
                    _thread = new Thread(new ThreadStart(ThreadMain));
                    _thread.Name = ToString();
                    _thread.IsBackground = true;
                    _thread.Start();
                }
                catch (Exception exception)
                {
                    ShowError(string.Format(
                        "スレッドの開始に失敗しました。{0}\r\n{1}",
                        ToString(), exception.Message));
                    Disconnect(0);
                    return false;
                }
                return true;
            }

            /// <summary>
            /// スレッドのメインループ
            /// </summary>
            public void ThreadMain()
            {
                try
                {
                    while (!_wantToExit)
                    {
                        _connection.MainLoop();
                        // ループ毎にyieldする
                        Thread.Sleep(0);
                    }
                    Disconnect(Timeout.Infinite);
                }
                catch (Exception exception)
                {
                    // 終了フラグを立てていないときならば、例外処理をする
                    if (!_wantToExit)
                    {
                        string message = exception.Message;
                        if (exception.InnerException != null)
                        {
                            message += "\r\n" +
                                exception.InnerException.Message;
                        }
                        ShowError(string.Format(
                            "Htcs通信に問題が発生しました。{0}\r\n{1}",
                            ToString(), message));
                    }
                    // 終了フラグを立てて、Htcs接続全てを閉じる
                    _wantToExit = true;

                    if (null != exception.Message)
                    {
                        //HtcsManager.LastError = exception.Message;
                        CommManager.Instance.LastError = exception.Message;
                    }
                    //HtcsManager.Terminate();
                    CommManager.Instance.Terminate();
                }
            }

            /// <summary>
            /// 切断
            /// </summary>
            public void Disconnect(int timeout)
            {
                // スレッドの停止を待つ
                if (_thread != null)
                {
                    // 終了フラグが立っていない時に閉じられたら、
                    // 終了フラグを立てて待つ
                    if (!_wantToExit)
                    {
                        _wantToExit = true;
                        _thread.Join(timeout);

                        if (null != _thread)
                        {
                            _thread.Abort();
                        }

                    }
                    else
                    {
                        // すでに終了しようとしているので待つ
                        Thread.Sleep(WaitForThreadExitTime);
                    }
                    _thread = null;
                }

                // TCP/IP接続を閉じる
                if (_reader != null)
                {
                    _reader.Close();
                    _reader = null;
                }

                if (_writer != null)
                {
                    _writer.Close();
                    _writer = null;
                }

                if (this.comChannel != null)
                {
                    this.comChannel.Close();
                    this.comChannel = null;
                }
            }

            /// <summary>
            ///
            /// </summary>
            public int ReceiveTimeout
            {
                set
                {
                    if (this.comChannel != null)
                    {
                        this.comChannel.ReceiveTimeout = value;
                    }
                }
            }

            /// <summary>
            ///
            /// </summary>
            public int SendTimeout
            {
                set
                {
                    if (this.comChannel != null)
                    {
                        this.comChannel.SendTimeout = value;
                    }
                }
            }


            //-----------------------------------------------------------------
            // 公開メンバ
            //-----------------------------------------------------------------

            /// <summary>
            /// リーダ
            /// </summary>
            public ProtocolSoundReader Reader { get { return _reader; } }

            /// <summary>
            /// ライタ
            /// </summary>
            public ProtocolSoundWriter Writer { get { return _writer; } }

            /// <summary>
            /// 文字列変換
            /// </summary>
            public override string ToString() { return _name; }

            /// <summary>
            /// 終了待ち
            /// </summary>
            public bool IsWantToExit { get { return _wantToExit; } }

            /// <summary>
            /// 切断中
            /// </summary>
            public bool DisconnectingFlag
            {
                get { return _disconnectingFlag; }
                set { _disconnectingFlag = value; }
            }

            //-----------------------------------------------------------------
            // 非公開メンバ
            //-----------------------------------------------------------------
            /// <summary>
            /// メッセージ
            /// </summary>
            protected void ShowMessage(string text)
            {
#if HtcsDEBUG
                HtcsManager.Console.WriteHtcsMessage(text);
#endif
            }

            /// <summary>
            /// エラー
            /// </summary>
            protected void ShowError(string text)
            {
#if HtcsDEBUG
                HtcsManager.Console.WriteHtcsError(text);
#endif
            }

            //-----------------------------------------------------------------
            // 接続
            private volatile HtcsConnection _connection;
            // 名前
            private volatile string _name;

            // TCPクライアント
            private IComChannel comChannel = null;

            // ライタ
            private volatile ProtocolSoundWriter _writer = null;
            // リーダ
            private volatile ProtocolSoundReader _reader = null;
            // スレッド
            private volatile Thread _thread = null;
            // 終了フラグ
            private volatile bool _wantToExit = false;
            // 切断中フラグ
            private volatile bool _disconnectingFlag = false;
        }
        #endregion
    }

    //-----------------------------------------------------------------------
    /// <summary>
    /// ProtocolSoundReader
    /// short,ushort,int,uint,floatのエンディアンを変換し、読み込みます
    /// </summary>
    public class HtcsProtocolSoundReader : ProtocolSoundReader
    {
        /// <summary>コンストラクタ</summary>
        public HtcsProtocolSoundReader(Stream input) : base(input) { }
        /// <summary>コンストラクタ</summary>
        public HtcsProtocolSoundReader(Stream input, Encoding encoding) :
            base(input, encoding)
        { }

        ///	<summary>short読み込み</summary>
        public override short ReadInt16()
        {
            return IPAddress.NetworkToHostOrder(base.ReadInt16());
            //return base.ReadInt16();
        }

        ///	<summary>ushort読み込み</summary>
        public override ushort ReadUInt16()
        {
            short result = (short)base.ReadUInt16();
            return (ushort)IPAddress.NetworkToHostOrder(result);
        }

        ///	<summary>int読み込み</summary>
        public override int ReadInt32()
        {
            int result = (int)base.ReadInt32();
            return (int)IPAddress.NetworkToHostOrder(result);
        }

        ///	<summary>uint読み込み</summary>
        public override uint ReadUInt32()
        {
            int result = (int)base.ReadUInt32();
            return (uint)IPAddress.NetworkToHostOrder(result);
        }

        ///	<summary>float読み込み</summary>
        public override float ReadSingle()
        {
            // ネットワークとエンディアンが違うことが前提になっています
            byte[] floatBytes = BitConverter.GetBytes(base.ReadSingle());
            Array.Reverse(floatBytes);
            return BitConverter.ToSingle(floatBytes, 0);
        }
    }

    //-----------------------------------------------------------------------
    /// <summary>
    /// ProtocolSoundWriter
    /// short,ushort,int,uint,floatのエンディアンを変換し、書き込みます
    /// </summary>
    public class HtcsProtocolSoundWriter : ProtocolSoundWriter
    {
        /// <summary>コンストラクタ</summary>
        public HtcsProtocolSoundWriter() : base() { }
        /// <summary>コンストラクタ</summary>
        public HtcsProtocolSoundWriter(Stream input) : base(input) { }
        /// <summary>コンストラクタ</summary>
        public HtcsProtocolSoundWriter(Stream input, Encoding encoding) :
            base(input, encoding)
        { }

        ///	<summary>short書き出し</summary>
        public override void Write(short value)
        {
            base.Write(IPAddress.HostToNetworkOrder(value));
        }

        ///	<summary>ushort書き出し</summary>
        public override void Write(ushort value)
        {
            base.Write((ushort)IPAddress.HostToNetworkOrder((short)value));
            //base.Write(value);
        }

        ///	<summary>int書き出し</summary>
        public override void Write(int value)
        {
            base.Write(IPAddress.HostToNetworkOrder(value));
            //base.Write(value);
        }

        ///	<summary>uint書き出し</summary>
        public override void Write(uint value)
        {
            base.Write((uint)IPAddress.HostToNetworkOrder((int)value));
            //base.Write(value);
        }

        ///	<summary>float書き出し</summary>
        public override void Write(float value)
        {
            // ネットワークとエンディアンが違うことが前提になっています
            byte[] floatBytes = BitConverter.GetBytes(value);
            Array.Reverse(floatBytes);
            base.Write(BitConverter.ToSingle(floatBytes, 0));
        }
    }
}
