﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Preview.HIO
{
    using System.IO;

    /// <summary>
    /// サウンド用プロトコルのパケットヘッダです。
    /// </summary>
    public struct PacketHeader
    {
        /// <summary>
        /// 構造体のサイズを取得します。
        /// </summary>
        public const int StructSize = 4 + 4 + 4;

        private Message message;
        private int dataSize;
        private HIO.ErrorCode errorCode;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="message">メッセージの種類を指定します。</param>
        /// <param name="dataSize">パケットのデータサイズを指定します。</param>
        public PacketHeader(Message message, int dataSize)
        {
            this.message = message;
            this.dataSize = dataSize;
            this.errorCode = HIO.ErrorCode.Succeeded;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="message">メッセージの種類を指定します。</param>
        /// <param name="dataSize">パケットのデータサイズを指定します。</param>
        /// <param name="errorCode">エラーコードを指定します。</param>
        public PacketHeader(Message message, int dataSize, HIO.ErrorCode errorCode)
            : this(message, dataSize)
        {
            this.errorCode = errorCode;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// メッセージを取得します。
        /// </summary>
        public Message Message
        {
            get
            {
                return this.message;
            }
        }

        /// <summary>
        /// メッセージをカテゴリを取得します。
        /// </summary>
        public MessageCategory MessageCategory
        {
            get
            {
                return (MessageCategory)(this.Message & MessageUtilitiy.CategoryMask);
            }
        }

        /// <summary>
        /// パケットのデータサイズを取得します。
        /// </summary>
        public int DataSize
        {
            get
            {
                return this.dataSize;
            }
        }

        /// <summary>
        /// エラーコードを取得します。
        /// </summary>
        public HIO.ErrorCode ErrorCode
        {
            get
            {
                return this.errorCode;
            }
        }

        /// <summary>
        /// 処理に成功したかどうかを取得します。
        /// </summary>
        public bool IsSucceeded
        {
            get
            {
                return !this.IsFailed;
            }
        }

        /// <summary>
        /// 処理に失敗かどうかを取得します。
        /// </summary>
        public bool IsFailed
        {
            get
            {
                return (this.ErrorCode & ErrorCode.Failed) != 0;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 指定ストリームから入力します。
        /// </summary>
        /// <param name="reader">入力ストリームを指定します。</param>
        public void Read(BinaryReader reader)
        {
            // TODO : ★Assert処理を追加

            this.message = (Message)reader.ReadInt32();
            this.dataSize = reader.ReadInt32();
        }

        /// <summary>
        /// 指定ストリームに出力します。
        /// </summary>
        /// <param name="writer">出力ストリームを指定します。</param>
        public void Write(BinaryWriter writer)
        {
            // TODO : ★Assert処理を追加

            writer.Write((int)this.message);
            writer.Write(this.dataSize);
        }
    }
}
