﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;

//using NWSL.Components;

using NW4R.ProtocolSound;

// preview
//using NW4R_SoundMaker.Preview.Models;

namespace NintendoWare.Preview.HIO.Sound.Tool
{
    using Communications;
    using NintendoWare.SoundMaker.Preview;

  /// <summary>
  /// サウンド設定パケット
  /// </summary>
  public class SetSoundLabelPacket : HIOToolSoundPacket
  {
    private PreviewPlayerCollectionGeneric _previewPlayers;

    /// <summary>
    /// コンストラクタ
    /// </summary>
    /// <param name="previewPlayers">対象プレビュープレイヤーのコレクション</param>
    public SetSoundLabelPacket(PreviewPlayerCollectionGeneric previewPlayers)
    {
      //Validator.NullAssert( previewPlayers );
      _previewPlayers = previewPlayers;
    }

    #region ** プロパティのオーバーライド

    /// <summary>
    /// パケットの種類
    /// </summary>
    public override ushort PacketType
    {
      get { return Constants.TOOL_SOUND_SET_SOUNDLABELS; }
    }

    /// <summary>
    /// サイズ
    /// </summary>
    public override ushort Size
    {
      get
      {
        return (ushort)( ToolSoundItemContainer.StructSize +  ( ToolSoundPreviewSoundLabel.StructSize * _previewPlayers.Count ) );
      }
    }

    #endregion

    #region ** メソッド

    /// <summary>
    /// サウンド設定パケットを送信します。
    /// </summary>
    /// <param name="previewPlayers">対象プレビュープレイヤーのコレクション</param>
    public static void Send(PreviewPlayerCollectionGeneric previewPlayers)
      {
        //if( !MCSManager.IsConnected ) { return; }
        if( CommManager.Instance.IsConnected == false )
        {
            return;
        }
      SendPacket( new SetSoundLabelPacket( previewPlayers ) );
    }

    #endregion

    #region ** メソッドのオーバーライド

    /// <summary>
    /// ストリームからデータを読み込みます。
    /// </summary>
    /// <param name="reader">対象ストリーム</param>
    /// <param name="header">パケットヘッダ</param>
    public override void Read(ProtocolSoundReader reader, HIOPacketHeader header)
    {
      // TOOL_SOUND_SET_EFFECTPARAMETER は Write のみ
      Debug.Assert( false, "Not override Read method");
    }

    /// <summary>
    /// ストリームにデータを書き出します。
    /// </summary>
    /// <param name="writer">対象ストリーム</param>
    /// <returns>出力したパケットヘッダ</returns>
    public override HIOPacketHeader Write(ProtocolSoundWriter writer)
    {
      Debug.Assert( null != writer, "Writer is null");

      MemoryStream memStream = new MemoryStream();
      ProtocolSoundWriter memWriter = new ProtocolSoundWriter( memStream );


      // パケットヘッダを出力する
      HIOPacketHeader header = CreateHeader();
      header.Write( memWriter );

      Debug.Assert( null != _previewPlayers, "PreviewPlayers is null");


      //
      // サウンドラベル一覧を出力する
      //
      // コンテナヘッダ
      ToolSoundItemContainer container = new ToolSoundItemContainer();

      container.offsetForData = ToolSoundItemContainer.StructSize;
      container.itemCount     = (uint)_previewPlayers.Count;
      container.itemSize      = ToolSoundPreviewSoundLabel.StructSize;
      container.Write( memWriter );

      // サウンドラベル
      foreach( PreviewPlayerGeneric player in _previewPlayers ) {

        ToolSoundPreviewSoundLabel outputItem = new ToolSoundPreviewSoundLabel();

        outputItem.index = player.Index;

        //if( null != player.Item && 0 < player.Item.Label.Length ) {
        if( null != player.Item && 0 < player.Item.Name.Length ) {

            //byte[] workLabel = Encoding.UTF8.GetBytes( player.Item.Label );
            byte[] workLabel = Encoding.UTF8.GetBytes( player.Item.Name );

          if( workLabel.Length >= outputItem.label.Length ) {
            //ShowError( GetMessage( "NameIsTooLong" ) );
            return null;
          }

          workLabel.CopyTo( outputItem.label, 0 );

        }
        else {
          outputItem.label[ 0 ] = 0;
        }

        outputItem.Write( memWriter );

      }

      writer.Write( memStream.GetBuffer(), 0, (int)memStream.Length );

      return header;
    }

    #endregion
  }

  /// <summary>
  /// サウンド情報取得パケット
  /// </summary>
  public class GetSoundInformationPacket : HIOToolSoundPacket
  {
    private PreviewPlayerInformationCollectionGeneric _informations;

    /// <summary>
    /// コンストラクタ
    /// </summary>
    /// <param name="previewPlayers">対象プレビュープレイヤーのコレクション</param>
    public GetSoundInformationPacket(PreviewPlayerInformationCollectionGeneric informations)
    {
      //Validator.NullAssert( informations );
      _informations = informations;
    }

    #region ** プロパティのオーバーライド

    /// <summary>
    /// パケットの種類
    /// </summary>
    public override ushort PacketType
    {
      get { return Constants.TOOL_SOUND_GET_SOUNDINFORMATIONS; }
    }

    /// <summary>
    /// サイズ
    /// </summary>
    public override ushort Size
    {
      get { return 0; }
    }

    #endregion

    #region ** メソッド

    /// <summary>
    /// サウンド情報の取得パケットを送信します。
    /// </summary>
    public static void Send(PreviewPlayerInformationCollectionGeneric informations)
    {
        //if( !MCSManager.IsConnected ) { return; }
        if( CommManager.Instance.IsConnected == false )
        {
            return;
        }
      SendPacket( new GetSoundInformationPacket( informations ) );
    }

    #endregion

    #region ** メソッドのオーバーライド

    /// <summary>
    /// 送信パケットを圧縮します。
    /// </summary>
    public override bool Compress(HIOToolPacket nextPacket)
    {
      return ( nextPacket is GetSoundInformationPacket );
    }

    /// <summary>
    /// ストリームからデータを読み込みます。
    /// </summary>
    /// <param name="reader">対象ストリーム</param>
    /// <param name="header">パケットヘッダ</param>
    public override void Read(ProtocolSoundReader reader, HIOPacketHeader header)
    {
      //Validator.Assert( false );
    }

    /// <summary>
    /// ストリームにデータを書き出します。
    /// </summary>
    /// <param name="writer">対象ストリーム</param>
    /// <returns>出力したパケットヘッダ</returns>
    public override HIOPacketHeader Write(ProtocolSoundWriter writer)
    {
      Debug.Assert( null != writer, "Writer is null");

      MemoryStream memStream = new MemoryStream();
      ProtocolSoundWriter memWriter = new ProtocolSoundWriter( memStream );


      // パケットヘッダを出力する
      HIOPacketHeader header = CreateHeader();
      header.Write( memWriter );

      // この１行は仮実装
      // Send用ToolPacket, Recv用ViewerPacketとして１つのクラスにまとめる
      Viewer.GetSoundInformationPacket.Received = false;

      writer.Write( memStream.GetBuffer(), 0, (int)memStream.Length );

      return header;
    }

    #endregion
  }

  /// <summary>
  /// サウンド情報表示パケット
  /// </summary>
  public class ShowSoundInformationPacket : HIOToolSoundPacket
  {
    private PreviewPlayerGeneric _target;

    /// <summary>
    /// コンストラクタ
    /// </summary>
    /// <param name="target">対象プレビュープレイヤー</param>
    public ShowSoundInformationPacket(PreviewPlayerGeneric target)
    {
      //Validator.NullAssert( target );
      _target = target;
    }

    #region ** プロパティのオーバーライド

    /// <summary>
    /// パケットの種類
    /// </summary>
    public override ushort PacketType
    {
      get { return Constants.TOOL_SOUND_SHOWSOUNDINFORMATION; }
    }

    /// <summary>
    /// サイズ
    /// </summary>
    public override ushort Size
    {
      get { return ToolSoundPreviewPlayerID.StructSize; }
    }

    #endregion

    #region ** メソッド

    /// <summary>
    /// サウンド情報表示パケットを送信します。
    /// </summary>
    public static void Send(PreviewPlayerGeneric target)
    {
        //if( !MCSManager.IsConnected ) { return; }
        if( CommManager.Instance.IsConnected == false )
        {
            return;
        }
      SendPacket( new ShowSoundInformationPacket( target ) );
    }

    #endregion

    #region ** メソッドのオーバーライド

    /// <summary>
    /// 送信パケットを圧縮します。
    /// </summary>
    public override bool Compress(HIOToolPacket nextPacket)
    {
      return ( nextPacket is ShowSoundInformationPacket );
    }

    /// <summary>
    /// ストリームからデータを読み込みます。
    /// </summary>
    /// <param name="reader">対象ストリーム</param>
    /// <param name="header">パケットヘッダ</param>
    public override void Read(ProtocolSoundReader reader, HIOPacketHeader header)
    {
      //Validator.Assert( false );
    }

    /// <summary>
    /// ストリームにデータを書き出します。
    /// </summary>
    /// <param name="writer">対象ストリーム</param>
    /// <returns>出力したパケットヘッダ</returns>
    public override HIOPacketHeader Write(ProtocolSoundWriter writer)
    {
      Debug.Assert( null != writer, "Writer is null");

      MemoryStream memStream = new MemoryStream();
      ProtocolSoundWriter memWriter = new ProtocolSoundWriter( memStream );


      // パケットヘッダを出力する
      HIOPacketHeader header = CreateHeader();
      header.Write( memWriter );

      //Validator.NullAssert( _target );

      // パラメータを出力する
      ToolSoundPreviewPlayerID parameter = new ToolSoundPreviewPlayerID();
      parameter.index = _target.Index;
      parameter.Write( memWriter );


      writer.Write( memStream.GetBuffer(), 0, (int)memStream.Length );

      return header;
    }

    #endregion
  }

  /// <summary>
  /// プレビュープレイヤーパラメータ設定パケット
  /// </summary>
  public class SetPreviewPlayerParameterPacket : HIOToolSoundPacket
  {
    private PreviewPlayerGeneric _previewPlayer;

    /// <summary>
    /// プレビュープレイヤーパラメータ設定パケットを送信します。
    /// </summary>
    /// <param name="previewPlayer">対象プレビュープレイヤー</param>
    public static void Send(PreviewPlayerGeneric previewPlayer)
    {
        //if( !MCSManager.IsConnected ) { return; }
        if( CommManager.Instance.IsConnected == false )
        {
            return;
        }

      SendPacket( new SetPreviewPlayerParameterPacket( previewPlayer ) );
    }

    /// <summary>
    /// コンストラクタ
    /// </summary>
    /// <param name="previewPlayer">対象プレビュープレイヤー</param>
    public SetPreviewPlayerParameterPacket(PreviewPlayerGeneric previewPlayer)
    {
      //Validator.NullAssert( previewPlayer );
      _previewPlayer = previewPlayer;
    }

    #region ** プロパティのオーバーライド

    /// <summary>
    /// パケットの種類
    /// </summary>
    public override ushort PacketType
    {
      get { return Constants.TOOL_SOUND_SET_PREVIEWPLAYERPARAMETER; }
    }

    /// <summary>
    /// サイズ
    /// </summary>
    public override ushort Size
    {
      get
      {
        return (ushort)ToolSoundPreviewPlayerParameter.StructSize;
      }
    }

    #endregion

    #region ** メソッド

    /// <summary>
    /// 送信パケットの圧縮
    /// </summary>
    public override bool Compress(HIOToolPacket nextPacket)
    {
      if( !( nextPacket is SetPreviewPlayerParameterPacket ) ) { return false; }
      return ( _previewPlayer == ( nextPacket as SetPreviewPlayerParameterPacket )._previewPlayer );
    }

    #endregion

    #region ** メソッドのオーバーライド

    /// <summary>
    /// ストリームからデータを読み込みます。
    /// </summary>
    /// <param name="reader">対象ストリーム</param>
    /// <param name="header">パケットヘッダ</param>
    public override void Read(ProtocolSoundReader reader, HIOPacketHeader header)
    {
      // Write のみ
      Debug.Assert( false, "Not override Read method");
    }

    /// <summary>
    /// ストリームにデータを書き出します。
    /// </summary>
    /// <param name="writer">対象ストリーム</param>
    /// <returns>出力したパケットヘッダ</returns>
    public override HIOPacketHeader Write(ProtocolSoundWriter writer)
    {
      Debug.Assert( null != writer, "Writer is null");

      MemoryStream memStream = new MemoryStream();
      ProtocolSoundWriter memWriter = new ProtocolSoundWriter( memStream );


      // パケットヘッダを出力する
      HIOPacketHeader header = CreateHeader();
      header.Write( memWriter );

      Debug.Assert( null != _previewPlayer, "PreviewPlayer is null");


      //
      // パラメータを出力する
      //
      // コンテナヘッダ
      ToolSoundPreviewPlayerParameter parameter = new ToolSoundPreviewPlayerParameter();

      parameter.previewPlayerIndex = _previewPlayer.Index;

      parameter.volume          = _previewPlayer.Paramters.Volume;
      parameter.pitch           = _previewPlayer.Paramters.Pitch;
      parameter.pan             = _previewPlayer.Paramters.Pan;
      parameter.surroundPan     = _previewPlayer.Paramters.SurroundPan;
      parameter.lpfFrequency    = _previewPlayer.Paramters.LpfFrequency;
      parameter.biquadType      = (ToolSoundBiquadType)_previewPlayer.Paramters.BiquadType;
      parameter.biquadValue     = _previewPlayer.Paramters.BiquadValue;
      parameter.startOffsetTime = _previewPlayer.Paramters.StartOffsetTime;

      parameter.mainOutEnabled    = _previewPlayer.Paramters.MainOut.Enabled ? 1 : 0;
      parameter.remote0OutEnabled = _previewPlayer.Paramters.RemoteOut[ 0 ].Enabled ? 1 : 0;
      parameter.remote1OutEnabled = _previewPlayer.Paramters.RemoteOut[ 1 ].Enabled ? 1 : 0;
      parameter.remote2OutEnabled = _previewPlayer.Paramters.RemoteOut[ 2 ].Enabled ? 1 : 0;
      parameter.remote3OutEnabled = _previewPlayer.Paramters.RemoteOut[ 3 ].Enabled ? 1 : 0;

      parameter.mainOutVolume    = _previewPlayer.Paramters.MainOut.Volume;
      parameter.remote0OutVolume = _previewPlayer.Paramters.RemoteOut[ 0 ].Volume;
      parameter.remote1OutVolume = _previewPlayer.Paramters.RemoteOut[ 1 ].Volume;
      parameter.remote2OutVolume = _previewPlayer.Paramters.RemoteOut[ 2 ].Volume;
      parameter.remote3OutVolume = _previewPlayer.Paramters.RemoteOut[ 3 ].Volume;

      parameter.mainSend       = _previewPlayer.Paramters.MainSend;
      parameter.effectAuxASend = _previewPlayer.Paramters.EffectAuxASend;
      parameter.effectAuxBSend = _previewPlayer.Paramters.EffectAuxBSend;
      parameter.effectAuxCSend = _previewPlayer.Paramters.EffectAuxCSend;

      parameter.fadeInFrames    = (int)_previewPlayer.Paramters.FadeInFrames;
      parameter.fadeOutFrames   = (int)_previewPlayer.Paramters.FadeOutFrames;
      parameter.pauseFadeFrames = (int)_previewPlayer.Paramters.PauseFadeFrames;

      parameter.Write( memWriter );


      writer.Write( memStream.GetBuffer(), 0, (int)memStream.Length );

      return header;
    }

    #endregion
  }
}
