﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;

//using NWSL.Components;

using NW4R.ProtocolSound;
//using NW4F_SoundMaker;

namespace NintendoWare.Preview.HIO.Sound.Tool
{
  using Communications;

    /// <summary>
    /// MCSツールサウンドパケット
    /// </summary>
    public abstract class HIOToolSoundPacket : HIOToolPacket {
#if false
        protected static string GetMessage(string id) {
            return NW4F_SoundMaker.AnimeSoundApplication.MessageService.Message(id);
        }
#endif
                                                                 public HIOToolSoundPacket() {
            Debug.Assert(Category == Constants.CATEGORY_SOUND, "Category is not sound");
        }
    }

    #region ** 同期パケットクラス

    #region ** イベントクラス

    public class ReplyEventArgsT<EventArgsType> : EventArgs
        where EventArgsType : EventArgs
    {
        #region ** イベントハンドラ デリゲート

        public delegate void EventHandler(object sender, EventArgsType e);

        #endregion

        #region ** パラメータ

        private uint   _errorCode;
        private object _userData;

        #endregion

        public ReplyEventArgsT(uint errorCode, object userData)
        {
            _errorCode = errorCode;
            _userData  = userData;
        }

        #region ** プロパティ

        public bool Failed
        {
            get { return ( 0 != ( Constants.MCSSOUND_ERRORFLAG & _errorCode ) ); }
        }

        public uint ErrorCode
        {
            get { return _errorCode; }
        }

        public object UserData
        {
            get { return _userData; }
        }

        #endregion
    }

    public class ReplyEventArgs : ReplyEventArgsT<ReplyEventArgs>
    {
        public ReplyEventArgs(uint errorCode, object userData) : base( errorCode, userData ) { }
    }

    #endregion

    /// <summary>
    /// MCSツールサウンド同期パケット
    /// </summary>
    public abstract class MCSToolSoundSyncPacketT<EventArgsType> : HIOToolSoundPacket, IDisposable
        where EventArgsType : ReplyEventArgsT<EventArgsType>
    {
        #region ** パラメータ

        private ReplyEventArgsT<EventArgsType>.EventHandler _handler;
        private object                                      _userData;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="handler">結果通知用デリゲート</param>
        /// <param name="userData">ユーザデータ</param>
        public MCSToolSoundSyncPacketT(ReplyEventArgsT<EventArgsType>.EventHandler handler, object userData)
        {
            _handler  = handler;
            _userData = userData;
        }

        #region ** メソッドのオーバーライド

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public sealed override void Read(ProtocolSoundReader reader, HIOPacketHeader header)
        {
            // エラーパケットを受け取るためのイベント設定解除
            NintendoWare.Preview.HIO.Error.Sound.HIOSoundErrorPacket.PacketRecieved -= OnErrorPacketReceived;

            EventArgsType recvEvent = OnRead( reader, header );
            Debug.Assert( null != recvEvent, "recvEvent is null");

            OnReply( recvEvent );
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public sealed override HIOPacketHeader Write(ProtocolSoundWriter writer)
        {
            HIOPacketHeader header = OnWrite( writer );

            // エラーパケットを受け取るためのイベント設定
            NintendoWare.Preview.HIO.Error.Sound.HIOSoundErrorPacket.PacketRecieved += OnErrorPacketReceived;

            // 返信イベントを設定する
            SetHandler();

            return header;
        }

        #endregion

        #region ** IDisposable の実装

        void IDisposable.Dispose()
        {
            ClearHandler();
        }

        #endregion

        #region ** プロパティ

        protected object UserData
        {
            get { return _userData; }
        }

        #endregion

        #region ** イベント

        public event ReplyEventArgsT<EventArgsType>.EventHandler ReplyRecieved;

        #endregion

        #region ** イベントハンドラ

        protected virtual HIOPacketHeader OnWrite(ProtocolSoundWriter writer) { return base.Write( writer ); }
        protected virtual EventArgsType   OnRead(ProtocolSoundReader reader, HIOPacketHeader header) { return null; }
        protected virtual EventArgsType   OnError(Error.Sound.HIOErrorPacketRecievedEventArgs e) { return null; }

        protected virtual void OnReply(EventArgsType e)
        {
            try {
                if( null != ReplyRecieved ) {
                    ReplyRecieved( this, e );
                }
            }
            finally {
                ClearHandler();
            }
        }

        private void OnErrorPacketReceived(object sender, Error.Sound.HIOErrorPacketRecievedEventArgs e)
        {
            // エラーパケットを受け取るためのイベント設定解除
            NintendoWare.Preview.HIO.Error.Sound.HIOSoundErrorPacket.PacketRecieved -= OnErrorPacketReceived;

            EventArgsType errorEvent = OnError( e );

            if( null != errorEvent ) {
                OnReply( errorEvent );
            }
        }

        #endregion

        #region ** メソッド

        private void SetHandler()
        {
            if( null != _handler ) {
                ReplyRecieved -= _handler;
                ReplyRecieved += _handler;
            }
        }

        private void ClearHandler()
        {
            if( null != _handler ) {
                ReplyRecieved -= _handler;
                _handler = null;
            }

        }

        #endregion
    }

    public abstract class MCSToolSoundSyncPacket : MCSToolSoundSyncPacketT<ReplyEventArgs>
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="handler">結果通知用デリゲート</param>
        /// <param name="userData">ユーザデータ</param>
        public MCSToolSoundSyncPacket(ReplyEventArgs.EventHandler handler, object userData) : base( handler, userData ) { }
    }

    #endregion

    /// <summary>
    /// サウンドアーカイブを読み込むパケットです。
    /// </summary>
    public class HIOLoadSarSoundPacket : MCSToolSoundSyncPacket
    {
        #region ** パラメータ

        private string _soundArchiveFileName;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="modelFileName">サウンドアーカイブのファイルパス</param>
        /// <param name="handler">結果通知用デリゲート</param>
        /// <param name="userData">ユーザデータ</param>
        public HIOLoadSarSoundPacket(string soundArchiveFileName, ReplyEventArgs.EventHandler handler, object userData)
            : base( handler, userData )
        {
            _soundArchiveFileName = soundArchiveFileName;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットタイプ
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_LOADSAR; }
        }

        /// <summary>
        /// データサイズ
        /// </summary>
        public override ushort Size
        {
            get { return (ushort)( ToolSoundLoadSar.StructSize ); }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// アニメーションプレイヤー作成パケットを送信します。
        /// </summary>
        public static bool Send(string soundArchiveFilePath)
        {
            return Send( soundArchiveFilePath, null );
        }
        public static bool Send(string soundArchiveFilePath, ReplyEventArgs.EventHandler handler)
        {
            return Send( soundArchiveFilePath, handler, null );
        }
        public static bool Send(string soundArchiveFilePath, ReplyEventArgs.EventHandler handler, object userData)
        {
            if( !CommManager.Instance.IsConnected ) { return false; }

            SendPacket( new HIOLoadSarSoundPacket( soundArchiveFilePath, handler, userData ) );
            return true;
        }

        #endregion

        #region ** イベントハンドラのオーバーライド

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        protected override HIOPacketHeader OnWrite(ProtocolSoundWriter writer)
        {
            Debug.Assert( null != writer, "Writer is null" );

            MemoryStream memStream = new MemoryStream();
            ProtocolSoundWriter memWriter = new ProtocolSoundWriter( memStream );


            // パケットヘッダを出力する
            HIOPacketHeader header = CreateHeader();
            header.Write( memWriter );

            // パラメータを出力する
            ToolSoundLoadSar parameter = new ToolSoundLoadSar();
            EncodeFilePath( _soundArchiveFileName, parameter.pathName );

            parameter.Write( memWriter );

            try
            {
                writer.Write( memStream.GetBuffer(), 0, (int)memStream.Length );
            }
            catch( Exception /*e*/)
            {
            }

            return header;
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        protected override ReplyEventArgs OnRead(ProtocolSoundReader reader, HIOPacketHeader header)
        {
            ToolSoundLoadSarReply reply = new ToolSoundLoadSarReply();
            reply.Read( reader );

            ShowMessage( "SoundArchive loaded. \"{0}\" ID 0x{1:x8}",
                            Path.GetFileNameWithoutExtension( _soundArchiveFileName ), reply.id );

            return new ReplyEventArgs( Constants.MCSSOUND_S_OK, UserData );
        }

        protected override ReplyEventArgs OnError(Error.Sound.HIOErrorPacketRecievedEventArgs e)
        {
            return new ReplyEventArgs( e.ErrorPacket.ErrorInfo.errorCode, UserData );
        }

        #endregion
    }

    /// <summary>
    /// 強制的にサウンドモードに切り替えるパケットです。
    /// </summary>
    public class HIOFocusSoundPacket : HIOToolSoundPacket {
        public static HIOFocusSoundPacket Send() {
            if (!CommManager.Instance.IsConnected) { return null; }
            HIOFocusSoundPacket packet = new HIOFocusSoundPacket();
            CommManager.Instance.SendPacket(packet);
            return packet;
        }
        public override ushort PacketType {
            get { return Constants.TOOL_SOUND_FOCUS; }
        }
    }

    /// <summary>
    /// 呼びかけて返信を要求するパケットです。
    /// 返信があると、ReadFlag が true になります。
    /// </summary>
    public class MCSEchoSoundPacket : HIOToolSoundPacket {
        public bool ReadFlag { get; set; }
        public static MCSEchoSoundPacket Send() {
            if (!CommManager.Instance.IsConnected) {
                return null;
            }
            MCSEchoSoundPacket packet = new MCSEchoSoundPacket();
            SendPacket(packet);
            return packet;
        }
        public override void Read(ProtocolSoundReader reader, HIOPacketHeader header) {
            ReadFlag = true;
        }
        public override ushort PacketType {
            get { return Constants.TOOL_SOUND_ECHO; }
        }
    }

    /// <summary>
    /// 全サウンドを停止するパケットです。
    /// </summary>
    public class MCSAllStopSoundPacket : HIOToolSoundPacket {
        public static void Send() {
            if (!CommManager.Instance.IsConnected) {
                return;
            }
            SendPacket(new MCSAllStopSoundPacket());
        }
        /// <summary>
        /// パケットタイプ
        /// </summary>
        public override ushort PacketType {
            get { return Constants.TOOL_SOUND_ALLSTOP; }
        }
    }

    /// <summary>
    /// サウンドシステム設定パケット
    /// </summary>
    public class HIOSetSoundSystemParametersPacket : HIOToolSoundPacket
    {
        private ToolSoundOutputMode _outputMode = ToolSoundOutputMode.OutputMode_Invalid;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="outputMode">出力モード</param>
        public HIOSetSoundSystemParametersPacket(ToolSoundOutputMode outputMode)
        {
            _outputMode = outputMode;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_SET_SYSTEMPARAMETERS; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get
            {
                return (ushort)ToolSoundSystemParameters.StructSize;
            }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// サウンド設定パケットを送信します。
        /// </summary>
        /// <param name="outputMode">出力モード</param>
        public static void Send(ToolSoundOutputMode outputMode)
        {
            if( !CommManager.Instance.IsConnected ) { return; }
            SendPacket( new HIOSetSoundSystemParametersPacket( outputMode ) );
        }

        #endregion

        #region ** メソッドのオーバーライド

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, HIOPacketHeader header)
        {
            // Write のみ
            Debug.Assert( false, "Not override Read method");
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override HIOPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert( null != writer, "Writer is null");

            MemoryStream memStream = new MemoryStream();
            ProtocolSoundWriter memWriter = new ProtocolSoundWriter( memStream );


            // パケットヘッダを出力する
            HIOPacketHeader header = CreateHeader();
            header.Write( memWriter );

            // パラメータを出力する
            ToolSoundSystemParameters parameters = new ToolSoundSystemParameters();
            parameters.outputMode = _outputMode;
            parameters.Write( memWriter );

            writer.Write( memStream.GetBuffer(), 0, (int)memStream.Length );

            return header;
        }

        #endregion
    }
}
