﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using NW4R.ProtocolSound;

namespace NintendoWare.Preview.HIO
{
    /// <summary>
    /// HIOパケットヘッダ
    /// <para>
    /// HIO通信で送受信されるパケットのヘッダクラスです。
    /// ヘッダの内容に関してはProtocolSample.csをご覧ください。
    /// </para>
    /// </summary>
    public class HIOPacketHeader
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public HIOPacketHeader(){}

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public HIOPacketHeader(ushort packetType, ushort size)
        {
            Set(packetType, size);
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public HIOPacketHeader(ushort category, ushort command,
            bool reply, bool error, ushort size)
        {
            Set(category, command, reply, error, size);
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public HIOPacketHeader(ProtocolSoundReader reader)
        {
            Read(reader);
        }

        /// <summary>
        /// 文字列の取得
        /// </summary>
        public override string ToString()
        {
            string message =
                "[ 0x" + PacketType.ToString("x4") + " " + Size + "Byte";
            if(Reply){ message += " Reply"; }
            if(Error){ message += " Error"; }
            message += " ]";
            return message;
        }

        //---------------------------------------------------------------------
        // アクセサ
        //---------------------------------------------------------------------
        /// <summary>
        /// パケットタイプ
        /// </summary>
        public ushort PacketType
        {
            get
            {
                ushort packetType = (ushort)(Category | Command);
                if(Reply){ packetType |= Constants.HEADER_REPLY_FLAG; }
                if(Error){ packetType |= Constants.HEADER_ERROR_FLAG; }
                return packetType;
            }
        }

        /// <summary>
        /// カテゴリ
        /// </summary>
        public ushort Category
        {
            get{ return _category; }
            set
            {
                Debug.Assert(_category ==
                    (_category & Constants.HEADER_CATEGORY_MASK), "Category is illegal");
                _category = value;
            }
        }

        /// <summary>
        /// コマンド
        /// </summary>
        public ushort Command
        {
            get{ return _command; }
            set
            {
                Debug.Assert(_command ==
                    (_command & Constants.HEADER_COMMAND_MASK), "Command is illegal");
                _command = value;
            }
        }

        /// <summary>
        /// 返信
        /// </summary>
        public bool Reply
        {
            get{ return _reply; }
            set{ _reply = value; }
        }

        /// <summary>
        /// エラー
        /// </summary>
        public bool Error
        {
            get{ return _error; }
            set{ _error = value; }
        }

        /// <summary>
        /// サイズ。
        /// </summary>
    public ushort Size
        {
            get{ return _size; }
            set{ _size = value; }
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 設定
        /// </summary>
        public void Set(ushort category, ushort command,
            bool reply, bool error, ushort size)
        {
            Category = category;
            Command = command;
            Reply = reply;
            Error = error;
            Size = size;
        }

        /// <summary>
        /// 設定
        /// </summary>
        public void Set(ushort packetType, ushort size)
        {
            Category = (ushort)(packetType & Constants.HEADER_CATEGORY_MASK);
            Command = (ushort)(packetType & Constants.HEADER_COMMAND_MASK);
            Reply = ((packetType & Constants.HEADER_REPLY_FLAG) != 0);
            Error = ((packetType & Constants.HEADER_ERROR_FLAG) != 0);
            Size = size;
        }

        //---------------------------------------------------------------------
        // 読み書き
        //---------------------------------------------------------------------
        /// <summary>
        /// 読み込み
        /// </summary>
        public void Read(ProtocolSoundReader reader)
        {
            ushort packetType = reader.ReadUInt16();
            ushort size = reader.ReadUInt16();
            Set(packetType, size);
        }

        /// <summary>
        /// 書き込み
        /// </summary>
        public void Write(ProtocolSoundWriter writer)
        {
#if false
            writer.Write(PacketType);
      writer.Write((ushort)0);
            writer.Write((int)Size);
#else
      MemoryStream memStream = new MemoryStream();
            ProtocolSoundWriter memWriter = new ProtocolSoundWriter( memStream );
            memWriter.Write(PacketType);
      memWriter.Write((ushort)0xFFEE);
            memWriter.Write((int)Size);
      writer.Write( memStream.GetBuffer(), 0, (int)memStream.Length );

#endif
            writer.Flush();
        }

        //---------------------------------------------------------------------
        // 非公開メンバ
        //---------------------------------------------------------------------
        // カテゴリ
        private ushort _category;
        // コマンド
        private ushort _command;
        // サイズ
        private ushort _size;
        // 返信
        private bool _reply;
        // エラー
        private bool _error;

    }
}
