﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;


namespace NintendoWare.SoundMaker.Preview.Communications
{
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NintendoWare.SoundMaker.Preview.MCS;
    using NintendoWare.SoundMaker.Preview.Htcs;
    using NintendoWare.SoundMaker.Resources;
    using NintendoWare.SoundMakerPlugin;
    using NW4R.ProtocolSound;

    using CommTool = NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using CommViewer = NintendoWare.SoundMaker.Framework.Preview.Communications.Viewer;
    using CommFunc = NintendoWare.SoundMaker.Framework.Preview.Communications.Func;
    using MCSTool = NintendoWare.SoundMaker.Preview.MCS.Tool.Sound;
    using Ctrl = NintendoWare.SoundMaker.Framework.Preview.Communications.Ctrl;


    public class CommunicationService
    {
        private ConnectionInfo currnetConnectionInfo;
        private CommunicationManager communicationManager;

        private EffectParameterCollection effectParameterSets = null;
        private uint deviceTypeValue;

        private PreviewPlayer target = null;

        private string soundArchivePath = null;
        private bool soundArchiveDirty = true;

        private MCSManager defaultMCSManager;
        private HtcsManager htcsManager;

        private ConnectionInfo[] defaultConnectionInfos;

        /// <summary>
        /// Viewerとの接続/切断イベントが発生すると実行されます。
        /// </summary>
        public delegate void ViewerConnectionDelegate(object sender, ConnectionChangedEventArgs e);
        public event ViewerConnectionDelegate ViewerConnectionChanged;

        /// <summary>
        /// SndEditとの接続/切断イベントが発生すると実行されます。
        /// </summary>
        public delegate void SndEditConnectionDelegate(object sender, ConnectionChangedEventArgs e);
        public event SndEditConnectionDelegate SndEditConnectionChanged;
        public event EventHandler SoundArchiveSending;

        public CommunicationService(IEnumerable<CommunicationInfo> communicationInfos)
        {
            communicationManager = new CommunicationManager(communicationInfos);
            CommViewer.ViewerPacket.Factories.Add(new CommViewer.SoundPacketFactoryCommon());
            CommFunc.FuncPacket.Factories.Add(new CommFunc.SoundPacketFactory());

            foreach (CommunicationInfo comInfo in communicationInfos)
            {
                foreach (ConnectionInfo info in comInfo.ConnectionInfos.Where(i => i.CommManager != null))
                {
                    info.CommManager.ConnectionChanged += OnConnectionChanged;
                    info.CommManager.SndEditConnectionChanged += OnSndEditConnectionChanged;
                }
            }

            this.htcsManager = new HtcsManager();
            this.htcsManager.Initialize(null, new HtcsConsole());
            this.htcsManager.ConnectionChanged += OnConnectionChanged;
            this.htcsManager.SndEditConnectionChanged += OnSndEditConnectionChanged;

            this.defaultMCSManager = new MCSManager();
            this.defaultMCSManager.Initialize(null, new MCSConsole());
            this.defaultMCSManager.ConnectionChanged += OnConnectionChanged;
            this.defaultMCSManager.SndEditConnectionChanged += OnSndEditConnectionChanged;

            CommManager.Instance = this.defaultMCSManager;
            this.defaultConnectionInfos = new ConnectionInfo[]
                {
#if USE_NWSND
                    new ConnectionInfo("PCApplication_nwsnd", MessageResource.PCApplication_nwsnd, true, this.DefaultConnect, this.DefaultDisconnect, this.DefaultIsConnected, this.DefaultIsConnecting, this.DefaultIsConnectedSndEdit, this.DefaultIsConnectingSndEdit, this.DefaultGetState, this.defaultMCSManager),
#endif
#if USE_PCSOUNDPLAYER
                    new ConnectionInfo("PCSoundPlayer_nwsnd", MessageResource.PCSoundPlayer_nwsnd, true, this.DefaultConnect, this.DefaultDisconnect, this.DefaultIsConnected, this.DefaultIsConnecting, this.DefaultIsConnectedSndEdit, this.DefaultIsConnectingSndEdit, this.DefaultGetState, this.defaultMCSManager),
#endif
                    new ConnectionInfo("PCApplication", MessageResource.PCApplication, true, this.HtcsConnect, this.HtcsDisconnect, this.HtcsIsConnected, this.HtcsIsConnecting, this.HtcsIsConnectedSndEdit, this.HtcsIsConnectingSndEdit, this.HtcsGetState, this.htcsManager),
                    new ConnectionInfo("TargetApplication", MessageResource.TargetApplication, true, this.HtcsConnect, this.HtcsDisconnect, this.HtcsIsConnected, this.HtcsIsConnecting, this.HtcsIsConnectedSndEdit, this.HtcsIsConnectingSndEdit, this.HtcsGetState, this.htcsManager),
                };
        }

        public bool Connect(string platform, string target)
        {
            ConnectionInfo connectionInfo = communicationManager.CommunicationInfos.Where(c => c.Platform == platform).Select(i => (i.ConnectionInfos.Where(b => b.TargetName == target).First())).First();

            if (connectionInfo.UseCustomConnection == false) // デフォルト接続を使います。
            {
                ConnectionInfo newInfo = this.defaultConnectionInfos.Where(b => b.TargetName == target).First();

                if (newInfo.CommManager is HtcsManager == true)
                {
                    //htcs 接続のため PeerType を設定します。
                    (newInfo.CommManager as HtcsManager).PeerType = connectionInfo.PeerType;
                }

                connectionInfo = newInfo;
            }

            this.currnetConnectionInfo = connectionInfo;
            CommManager.Instance = this.currnetConnectionInfo.CommManager;

            return this.currnetConnectionInfo.Connect();
        }

        public void Disconnect()
        {
            if (this.currnetConnectionInfo != null)
            {
                this.currnetConnectionInfo.Disconnect();
                CommManager.Instance = this.defaultMCSManager;
                this.currnetConnectionInfo = null;
            }
        }

        //

        public void GetSeqVariables(SeqVariableContainer variables)
        {
            SeqVariableContainerCollection tempCollection = new SeqVariableContainerCollection();
            tempCollection.Add(variables);

            this.GetSeqVariables(tempCollection);
        }

        public void GetSeqVariables(SeqVariableContainerCollection variables)
        {
            MCSTool.GetSeqVariablesPacket.Send(variables);
        }

        public void SetEffectParameters()
        {
            if (this.effectParameterSets != null)
            {
                MCSTool.SetEffectParametersPacket.Send(this.effectParameterSets, this.deviceTypeValue);
                this.effectParameterSets = null;
            }
        }

        public void SetEffectParameters(EffectParameterCollection parameterSets, uint deviceTypeValue)
        {
            this.effectParameterSets = parameterSets;
            this.deviceTypeValue = deviceTypeValue;
        }

        public void SetOutputMode(OutputMode mainOutputMode, OutputMode drcOutputMode)
        {
            MCSTool.SetOutputModePacket.Send((ToolSoundOutputMode)mainOutputMode, (ToolSoundOutputMode)drcOutputMode);
        }

        public void SetPreviewPlayerParameter()
        {
            if (this.target != null)
            {
                MCSTool.SetPreviewPlayerParameterPacket.Send(this.target);
                this.target = null;
            }
        }

        public void SetPreviewPlayerParameter(PreviewPlayer target)
        {
            this.target = target;
        }

        public void SetSeqVariables(SeqVariableContainer variables)
        {
            SeqVariableContainerCollection tempCollection = new SeqVariableContainerCollection();
            tempCollection.Add(variables);

            SetSeqVariables(tempCollection);
        }

        public void SetSeqVariables(SeqVariableContainerCollection variables)
        {
            MCSTool.SetSeqVariablesPacket.Send(variables);
        }

        public void SetSoundLabel(PreviewPlayer previewPlayer)
        {
            PreviewPlayerCollection tempCollection = new PreviewPlayerCollection();
            tempCollection.Add(previewPlayer);
            CommTool.SetSoundLabelPacket.Send(tempCollection);
        }

        public void SetSoundLabels(PreviewPlayerManager previewPlayerManager)
        {
            PreviewPlayerCollection players = new PreviewPlayerCollection();
            foreach (IPreviewPlayer previewPlayer in previewPlayerManager.PreviewPlayers)
            {
                players.Add(previewPlayer);
            }

            CommTool.SetSoundLabelPacket.Send(players);
        }

        /// <summary>
        /// 接続反応がなくなってから切断するまでの待ち時間（秒）。
        /// </summary>
        public int Timeout
        {
            get
            {
                return this.htcsManager.Timeout;
            }
            set
            {
                this.htcsManager.Timeout = value;
            }
        }

        public bool IsConnected
        {
            get { return this.currnetConnectionInfo != null && this.currnetConnectionInfo.IsConnected; }
        }

        public bool IsConnecting
        {
            get { return this.currnetConnectionInfo != null && this.currnetConnectionInfo.IsConnecting; }
        }

        public bool IsConnectedSndEdit
        {
            get { return this.currnetConnectionInfo != null && this.currnetConnectionInfo.IsConnectedSndEdit; }
        }

        public bool IsConnectingSndEdit
        {
            get { return this.currnetConnectionInfo != null && this.currnetConnectionInfo.IsConnectingSndEdit; }
        }

        public void Play(uint soundHandleIndex, string soundName)
        {
            Ctrl.PlaySoundPacket.Send((int)soundHandleIndex, soundName);
        }

        public void Pause(uint previewPlayerIndex)
        {
            Ctrl.PauseSoundPacket.Send((int)previewPlayerIndex);
        }

        public void Stop(uint previewPlayerIndex)
        {
            Ctrl.StopSoundPacket.Send((int)previewPlayerIndex);
        }

        // htcs 対応で削除予定↓
        public void SetPCPort(int SyncPortPC, int FuncPortPC, int CtrlPortPC, int ToolPortPC, int ViewerPortPC, int PingPortPC)
        {
            this.defaultMCSManager.SyncPortPC = SyncPortPC;
            this.defaultMCSManager.FuncPortPC = FuncPortPC;
            this.defaultMCSManager.CtrlPortPC = CtrlPortPC;
            this.defaultMCSManager.ToolPortPC = ToolPortPC;
            this.defaultMCSManager.ViewerPortPC = ViewerPortPC;
            this.defaultMCSManager.PingPortPC = PingPortPC;
        }

        public ConnectionState GetState()
        {
            if (this.currnetConnectionInfo != null)
            {
                return this.currnetConnectionInfo.GetState();
            }

            return ConnectionState.Disconnected;
        }

        public bool IsTargetPC
        {
            get
            {
                return CommManager.Instance == this.defaultMCSManager;
            }
        }

        public string[] GetTargetNameLabels(string platform)
        {
            return this.communicationManager.CommunicationInfos.Where(a => a.Platform == platform).Select(b => b.ConnectionInfos.Select(c => c.TargetNameLabel)).First().ToArray();
        }

        public string GetTargetNameByTargetNameLabel(string platform, string targetNameLabel)
        {
            if (string.IsNullOrEmpty(platform) == true || string.IsNullOrEmpty(targetNameLabel) == true)
            {
                return string.Empty;
            }

            return this.communicationManager.CommunicationInfos.Where(a => a.Platform == platform).Select(b => b.ConnectionInfos.Where(c => c.TargetNameLabel == targetNameLabel).Select(d => d.TargetName)).First().First();
        }

        public bool IsTargetSoundPlayer(string targetName)
        {
            return targetName.Contains("SoundPlayer");
        }

        public void TransferSoundArchive(string path)
        {
            this.soundArchivePath = path;
            this.soundArchiveDirty = true;

            if (IsConnected != false)
            {
                SendSoundArchive();
            }
        }

        private void SendSoundArchive()
        {
            if (this.soundArchiveDirty == false || this.soundArchivePath == null || this.soundArchivePath.Length == 0) { return; }

            LoadSarSoundPacket.Send(this.soundArchivePath);
            OnSoundArchiveSending(EventArgs.Empty);

            this.soundArchiveDirty = false;
        }

        private void OnSoundArchiveSending(EventArgs e)
        {
            if (null != SoundArchiveSending)
            {
                SoundArchiveSending(this, e);
            }
        }

        protected void OnSndEditConnectionChanged(ConnectionChangedEventArgs e)
        {
            if (SndEditConnectionChanged != null)
            {
                ConnectionChangedEventArgs args =
                    new ConnectionChangedEventArgs(e.State, e.ErrorText);
                SndEditConnectionChanged(this, args);
            }
        }

        private void OnConnectionChanged(ConnectionChangedEventArgs e)
        {
            if (null != ViewerConnectionChanged)
            {
                ConnectionChangedEventArgs args =
                    new ConnectionChangedEventArgs(e.State, e.ErrorText);
                ViewerConnectionChanged(this, args);
            }
        }

        //

        private bool DefaultConnect(object param)
        {
            Debug.WriteLine("Default Connect");

            return this.defaultMCSManager.Connect();
        }

        private void DefaultDisconnect()
        {
            Debug.WriteLine("Default Disconnect");

            this.defaultMCSManager.Disconnect();
        }

        private bool DefaultIsConnected()
        {
            Debug.WriteLine("Default IsConnected");

            return this.defaultMCSManager.IsConnected;
        }

        private bool DefaultIsConnecting()
        {
            Debug.WriteLine("Default IsConnecting");

            return this.defaultMCSManager.IsConnecting;
        }

        private bool DefaultIsConnectedSndEdit()
        {
            Debug.WriteLine("Default IsConnectedSndEdit");

            return this.defaultMCSManager.IsConnectedSndEdit;
        }

        private bool DefaultIsConnectingSndEdit()
        {
            Debug.WriteLine("Default IsConnectingSndEdit");

            return this.defaultMCSManager.IsConnectingSndEdit;
        }

        private ConnectionState DefaultGetState()
        {
            Debug.WriteLine("Default GetState");

            return this.defaultMCSManager.GetState();
        }

        // Htcs
        private bool HtcsConnect(object param)
        {
            Debug.WriteLine("Htcs Connect");

            return this.htcsManager.Connect();
        }

        private void HtcsDisconnect()
        {
            Debug.WriteLine("Htcs Disconnect");

            this.htcsManager.Disconnect();
        }

        private bool HtcsIsConnected()
        {
            Debug.WriteLine("Htcs IsConnected");

            return this.htcsManager.IsConnected;
        }

        private bool HtcsIsConnecting()
        {
            Debug.WriteLine("Htcs IsConnecting");

            return this.htcsManager.IsConnecting;
        }

        private bool HtcsIsConnectedSndEdit()
        {
            Debug.WriteLine("Htcs IsConnectedSndEdit");

            return this.htcsManager.IsConnectedSndEdit;
        }

        private bool HtcsIsConnectingSndEdit()
        {
            Debug.WriteLine("Htcs IsConnectingSndEdit");

            return this.htcsManager.IsConnectingSndEdit;
        }

        private ConnectionState HtcsGetState()
        {
            Debug.WriteLine("Htcs GetState");

            return this.htcsManager.GetState();
        }

        //

        private class CommunicationManager
        {
            private List<CommunicationInfo> communicationInfos = new List<CommunicationInfo>();

            public CommunicationManager(IEnumerable<CommunicationInfo> communicationInfos)
            {
                foreach (CommunicationInfo info in communicationInfos)
                {
                    this.communicationInfos.Add(info);
                }
            }

            public IEnumerable<CommunicationInfo> CommunicationInfos
            {
                get
                {
                    return this.communicationInfos;
                }
            }
        }
    }
}
