﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.FileFormats
{
    using System;
    using System.Collections;
    using System.IO;
    using System.Runtime.InteropServices;
    using NintendoWare.SoundFoundation.Codecs;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.FileFormats.Wave;

    public delegate IWaveFileInfo CreateWaveFileInfoDelegate();

    /// <summary>
    /// WaveFileManager の概要の説明です。
    /// </summary>
    public class WaveFileManager
    {
        private static WaveFileManager InstanceInternal = new WaveFileManager();
        private CreateWaveFileInfoDelegate CreateWaveFileInfo;

        private Hashtable waveFiles = new Hashtable();

        private WaveFileManager() { }

        public static WaveFileManager Instance
        {
            get { return WaveFileManager.InstanceInternal; }
        }

        public string PreprocessExePath { get; set; }

        public void SetCreateWaveFileInfo(CreateWaveFileInfoDelegate createWaveFileInfo)
        {
            this.CreateWaveFileInfo = createWaveFileInfo;
        }

        public IWaveFileInfo LoadFile(string filePath)
        {
            return LoadFile(filePath, null, null, null, null, null);
        }

        public IWaveFileInfo LoadFile(string filePath, bool? isLoop, int? loopStart, int? loopEnd)
        {
            return LoadFile(filePath, isLoop, loopStart, loopEnd, null, null);
        }

        public IWaveFileInfo LoadFile(string filePath, bool? isLoop, int? loopStart, int? loopEnd, int? sampleRate, int? channelCount)
        {
            IWaveFileInfo winfo;

            string fullPath = filePath.GetFullPath();
            string key = GetKey(filePath, isLoop, loopStart, loopEnd, sampleRate, channelCount);

            if (waveFiles.Contains(key))
            {
                winfo = (IWaveFileInfo)waveFiles[key];

                DateTime time = File.GetLastWriteTime(filePath);
                if (winfo.LastModifiedTime != time)
                {
                    try
                    {
                        winfo.Open(filePath, isLoop, loopStart, loopEnd, sampleRate, channelCount);
                    }
                    catch
                    {
                        return null;
                    }
                }
            }
            else
            {
                winfo = this.CreateWaveFileInfo();

                winfo.Open(filePath, isLoop, loopStart, loopEnd, sampleRate, channelCount);
                waveFiles[key] = winfo;
            }

            return winfo;
        }

        public void Reload()
        {
            foreach (IWaveFileInfo winfo in waveFiles.Values)
            {
                DateTime time = File.GetLastWriteTime(winfo.FilePath);
                if (winfo.LastModifiedTime != time)
                {
                    try
                    {
                        winfo.Open(winfo.FilePath);
                    }
                    catch
                    {
                        winfo.Invalidate();
                    }
                }
            }
        }

        public void ClearWaveFileCache()
        {
            foreach (IWaveFileInfo winfo in waveFiles.Values)
            {
                winfo.Dispose();
            }
            waveFiles.Clear();
        }

        private string GetKey(string filePath, bool? isLoop, int? loopStart, int? loopEnd, int? sampleRate, int? channelCount)
        {
            if (!isLoop.HasValue && !loopStart.HasValue && !loopEnd.HasValue && !sampleRate.HasValue && !channelCount.HasValue)
            {
                return filePath;
            }

            var builder = new System.Text.StringBuilder(filePath);
            builder.Append("?");

            if (isLoop.HasValue)
            {
                builder.Append("isLoop=");
                builder.Append(isLoop.Value.ToString());
            }

            if (loopStart.HasValue)
            {
                if (isLoop.HasValue)
                {
                    builder.Append("&");
                }

                builder.Append("loopStart=");
                builder.Append(loopStart.Value.ToString());
            }

            if (loopEnd.HasValue)
            {
                if (loopStart.HasValue || isLoop.HasValue)
                {
                    builder.Append("&");
                }

                builder.Append("loopEnd=");
                builder.Append(loopEnd.Value.ToString());
            }

            if (sampleRate.HasValue)
            {
                if (loopEnd.HasValue || loopStart.HasValue || isLoop.HasValue)
                {
                    builder.Append("&");
                }

                builder.Append("sampleRate=");
                builder.Append(sampleRate.Value.ToString());
            }

            if (channelCount.HasValue)
            {
                if (sampleRate.HasValue || loopEnd.HasValue || loopStart.HasValue || isLoop.HasValue)
                {
                    builder.Append("&");
                }

                builder.Append("channelCount=");
                builder.Append(channelCount.Value.ToString());
            }

            return builder.ToString();
        }
    }
}
