﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.IO;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.ToolDevelopmentKit.Collections;

    /// <summary>
    /// シーケンスサウンドバイナリから情報を読み取るためのクラスです。
    /// </summary>
    public class SequenceSoundBinaryReader : ISequenceSoundBinaryReader
    {
        private const string Signature_DataBlock = "DATA";
        private const string Signature_LabelBlock = "LABL";

        private const Byte Command_MmlAllockTrack = 0xfe;

        private readonly Dictionary<string, BlockInfo> blockInfoDictionary = new Dictionary<string, BlockInfo>();
        private readonly LabelInfoList labelInfos = new LabelInfoList();

        private CommonFileHeader header = null;

        /// <summary>
        /// シーケンスサウンドバイナリを解析します。
        /// </summary>
        /// <param name="reader">シーケンスサウンドバイナリを読み込むリーダを指定します。</param>
        public void Parse(BinaryReader reader)
        {
            Ensure.Argument.NotNull(reader);

            this.Clear();

            int blockCount = 0;

            this.ReadHeader(reader, out blockCount);
            this.ReadBlocks(reader, this.ReadReferenceWithSizeHeaderTable(reader, blockCount));
        }

        /// <summary>
        /// ラベル情報リストを取得します。
        /// </summary>
        public IKeyedList<string, LabelInfo> LabelInfos
        {
            get { return this.labelInfos; }
        }

        /// <summary>
        /// 指定ラベルの再生開始位置情報を取得します。
        /// </summary>
        /// <param name="reader">シーケンスサウンドバイナリを読み込むリーダを指定します。</param>
        /// <param name="label">ラベルを指定します。</param>
        /// <returns>再生開始位置情報を返します。</returns>
        public StartPositionInfo GetStartPositionInfo(BinaryReader reader, string label)
        {
            Ensure.Argument.NotNull(reader);

            long startOffset = 0;

            if (label != null && label.Length > 0)
            {
                if (!this.LabelInfos.ContainsKey(label))
                {
                    throw new KeyNotFoundException(label);
                }

                startOffset = this.LabelInfos.GetValue(label).OffsetFromDataBlockBody;
            }

            uint allocateTrackFlags = this.GetAllocateTrackFlags(reader, startOffset);

            // AllocTrack コマンドがある場合は、コマンド分だけオフセットを進めておく
            if (allocateTrackFlags != 1)
            {
                startOffset += 3;
            }

            return new StartPositionInfo()
            {
                OffsetFromDataBlockBody = startOffset,
                AllocateTrackFlags = allocateTrackFlags,
            };
        }

        /// <summary>
        /// 指定位置にあるコマンドからトラックアロケートフラグを取得します。
        /// </summary>
        /// <param name="reader">シーケンスサウンドバイナリを読み込むリーダを指定します。</param>
        /// <param name="commandOffset">コマンド位置を指定します。</param>
        /// <returns>トラックアロケートフラグを返します。</returns>
        private uint GetAllocateTrackFlags(BinaryReader reader, long commandOffset)
        {
            Ensure.Argument.NotNull(reader);
            Ensure.Argument.True(commandOffset >= 0);

            reader.BaseStream.Position =
                this.blockInfoDictionary[Signature_DataBlock].BodyAddress + commandOffset;

            Byte[] command = reader.ReadBytes(3);

            // 使用するトラックのビットが立ったフラグを返します。
            // AllocTrack コマンドでない場合は、0番トラックのみとして返します。
            // AllocTrack コマンドの場合は、0番トラック + ビット演算
            // （2バイトにわかれたフラグを連結した）結果を返します。
            if (Command_MmlAllockTrack != command[0]) { return 1; }
            return (uint)(1 | command[2] << 8 | command[1]);
        }

        private void Clear()
        {
            this.header = null;
            this.blockInfoDictionary.Clear();
            this.labelInfos.Clear();
        }

        //-----------------------------------------------------------------
        // バイナリデータの読み込み
        //-----------------------------------------------------------------

        private void ReadHeader(BinaryReader reader, out int blockCount)
        {
            Assertion.Argument.NotNull(reader);

            this.header = new CommonFileHeader();
            this.header.Signature = new string(reader.ReadChars(4));

            reader.ReadUInt32();    // バイトオーダー + ヘッダサイズ

            this.header.Version = reader.ReadUInt32();

            reader.ReadUInt32();    // ファイルサイズ

            blockCount = reader.ReadUInt16();

            reader.ReadUInt16();    // パディング
        }

        private void ReadBlocks(BinaryReader reader, IList<long> blockOffsets)
        {
            Assertion.Argument.NotNull(reader);

            this.blockInfoDictionary.Clear();

            foreach (long blockOffset in blockOffsets)
            {
                reader.BaseStream.Position = blockOffset;

                BlockInfo blockInfo = this.ReadBlockHeader(reader);
                this.blockInfoDictionary.Add(blockInfo.Signature, blockInfo);

                switch (blockInfo.Signature)
                {
                    case Signature_LabelBlock:
                        reader.BaseStream.Position = blockInfo.BodyAddress;
                        this.ReadLabelBlockBody(reader);
                        break;
                }
            }
        }

        private BlockInfo ReadBlockHeader(BinaryReader reader)
        {
            Assertion.Argument.NotNull(reader);

            BlockInfo blockInfo = new BlockInfo();
            blockInfo.Address = reader.BaseStream.Position;
            blockInfo.Signature = new string(reader.ReadChars(4));
            blockInfo.Size = reader.ReadUInt32();    // ブロックサイズ
            blockInfo.BodyAddress = reader.BaseStream.Position;

            return blockInfo;
        }

        private void ReadLabelBlockBody(BinaryReader reader)
        {
            Assertion.Argument.NotNull(reader);

            this.labelInfos.Clear();

            IList<long> offsets = this.ReadReferenceTable(reader);

            foreach (uint offset in offsets)
            {
                reader.BaseStream.Position = offset;

                LabelInfo labelInfo = this.ReadLabelInfo(reader);
                this.labelInfos.Add(labelInfo);
            }
        }

        private IList<long> ReadReferenceWithSizeHeaderTable(BinaryReader reader, int itemCount)
        {
            Assertion.Argument.NotNull(reader);
            Assertion.Argument.True(itemCount > 0);

            List<long> offsets = new List<long>();

            for (int i = 0; i < itemCount; i++)
            {
                reader.ReadUInt32();    // タイプ情報 + パディング
                offsets.Add(reader.ReadUInt32());
                reader.ReadUInt32();    // サイズ
            }

            return offsets;
        }

        private IList<long> ReadReferenceTable(BinaryReader reader)
        {
            List<long> offsets = new List<long>();

            long tablePosition = reader.BaseStream.Position;
            uint itemCount = reader.ReadUInt32();

            for (int i = 0; i < itemCount; i++)
            {
                reader.ReadUInt32();    // タイプ情報 + パディング
                offsets.Add(tablePosition + reader.ReadUInt32());
            }

            return offsets;
        }

        private LabelInfo ReadLabelInfo(BinaryReader reader)
        {
            LabelInfo labelInfo = new LabelInfo();

            reader.ReadUInt32();    // タイプ情報 + パディング

            labelInfo.OffsetFromDataBlockBody = reader.ReadUInt32();
            labelInfo.Address =
                this.blockInfoDictionary[Signature_DataBlock].Address + labelInfo.OffsetFromDataBlockBody;

            int labelLength = reader.ReadInt32();

            if (labelLength > 0)
            {
                labelInfo.Label = new string(reader.ReadChars(labelLength));
            }
            else
            {
                labelInfo.Label = string.Empty;
            }

            return labelInfo;
        }

        //-----------------------------------------------------------------
        // バイナリ情報
        //-----------------------------------------------------------------

        private class BlockInfo
        {
            public string Signature { get; set; }
            public long Address { get; set; }
            public long BodyAddress { get; set; }
            public long Size { get; set; }
        }

        private class LabelInfoList : KeyedCollection<string, LabelInfo>, IKeyedList<string, LabelInfo>
        {
            /// <summary>
            /// 指定されたキーがコレクション内に存在するかどうか調べます。
            /// </summary>
            /// <param name="key">アイテムのキーを指定します。</param>
            /// <returns>
            /// 指定されたキーがコレクション内に存在する場合は true、
            /// 存在しない場合は false を返します。
            /// </returns>
            public bool ContainsKey(string key)
            {
                return base.Contains(key);
            }

            /// <summary>
            /// 指定されたキーに関連付けられている値を取得します。
            /// </summary>
            /// <param name="key">アイテムのキーを指定します。</param>
            /// <returns>指定されたキーに関連付けられているアイテムを返します。</returns>
            public LabelInfo GetValue(string key)
            {
                Ensure.Argument.NotNull(key);
                return this[key];
            }

            /// <summary>
            /// 指定されたキーに関連付けられている値を取得します。
            /// </summary>
            /// <param name="key">アイテムのキーを指定します。</param>
            /// <param name="value">
            /// 指定されたキーが存在する場合、そのキーに関連付けられている値を返します。
            /// それ以外の場合は value パラメータの型に対する既定の値を返します。
            /// </param>
            /// <returns>指定されたキーに関連付けられているアイテムを返します。</returns>
            public bool TryGetValue(string key, out LabelInfo value)
            {
                Ensure.Argument.NotNull(key);

                if (base.Contains(key))
                {
                    value = this[key];
                    return true;
                }
                else
                {
                    value = null;
                    return false;
                }
            }

            protected override string GetKeyForItem(LabelInfo item)
            {
                return item.Label;
            }
        }
    }
}
