﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.FileFormats.Binary
{
    using System;
    using NintendoWare.SoundFoundation.Binarization;
    using NintendoWare.SoundFoundation.Projects;

    internal sealed class BiquadType2ByteConverter : TypeConverter<BiquadType, Byte>
    {
        private enum BinaryBiquadType : byte
        {
            None = 0,
            Lpf = 1,
            Hpf = 2,
            Bpf512 = 3,
            Bpf1024 = 4,
            Bpf2048 = 5,
            UserMin = 64,
            UserMax = 127,
        };

        /// <summary>
        /// 値を変換します。
        /// </summary>
        /// <param name="value">変換する値を指定します。</param>
        /// <returns>変換後の値を返します。</returns>
        public override Byte ConvertTo(BiquadType value)
        {
            switch (value)
            {
                case BiquadType.None:
                    return (byte)BinaryBiquadType.None;

                case BiquadType.LPF:
                    return (byte)BinaryBiquadType.Lpf;

                case BiquadType.HPF:
                    return (byte)BinaryBiquadType.Hpf;

                case BiquadType.BPF512:
                    return (byte)BinaryBiquadType.Bpf512;

                case BiquadType.BPF1024:
                    return (byte)BinaryBiquadType.Bpf1024;

                case BiquadType.BPF2048:
                    return (byte)BinaryBiquadType.Bpf2048;
            }

            if (BiquadType.USR0 <= value && value <= BiquadType.USR63)
            {
                return (byte)((int)BinaryBiquadType.UserMin + ((int)value - BiquadType.USR0));
            }

            throw new Exception("internal error : invalid BiquadType.");
        }

        /// <summary>
        /// 値を変換します。
        /// </summary>
        /// <param name="value">変換する値を指定します。</param>
        /// <returns>変換後の値を返します。</returns>
        public override BiquadType ConvertFrom(Byte value)
        {
            switch (value)
            {
                case 0:
                    return BiquadType.None;

                case 1:
                    return BiquadType.LPF;

                case 2:
                    return BiquadType.HPF;

                case 3:
                    return BiquadType.BPF512;

                case 4:
                    return BiquadType.BPF1024;

                case 5:
                    return BiquadType.BPF2048;
            }

            if ((byte)BinaryBiquadType.UserMin <= value && value <= (byte)BinaryBiquadType.UserMax)
            {
                return (BiquadType)value;
            }

            throw new Exception("internal error : invalid BiquadType.");
        }
    }
}
