﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.StreamSoundElements
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Runtime.InteropServices;
    using NintendoWare.SoundFoundation.Binarization;
    using NintendoWare.SoundFoundation.Projects;

    [StructLayout(LayoutKind.Sequential)]
    [DomElement(Tags = new string[] { "Element" })]
    [DomElementType(Value = (UInt16)ElementType.StreamSoundFile_RegionInfo)]
    internal class RegionInfo
    {
        public const int Size = 256;            /// この構造体のサイズです。
        public const int MaxName = 64;          /// 終端文字を含む文字列長
        public const int MaxChannels = 2 * 8;   /// 2ch × 8track

        //-----------------------------------------------------------------

        private readonly DspAdpcmContext[] contexts = new DspAdpcmContext[MaxChannels];
        private readonly byte[] padding = new byte[87];

        private string name = string.Empty;

        //-----------------------------------------------------------------

        public RegionInfo()
        {
            for (int index = 0; index < this.contexts.Length; ++index)
            {
                this.contexts[index] = new DspAdpcmContext();
            }
        }

        //-----------------------------------------------------------------
        // データ構造
        //-----------------------------------------------------------------

        /// <summary>
        /// スタートフレームを取得または設定します。
        /// </summary>
        public UInt32 StartFrame { get; set; }

        /// <summary>
        /// エンドフレームを取得または設定します。
        /// </summary>
        public UInt32 EndFrame { get; set; }

        /// <summary>
        /// 各ストリームチャンネル向けの DspAdpcm コンテキスト配列を取得します。
        /// </summary>
        public DspAdpcmContext[] Contexts
        {
            get { return this.contexts; }
        }

        /// <summary>
        /// 有効なリージョンかどうかを取得または設定します。
        /// </summary>
        public bool IsRegionEnabled { get; set; }

        /// <summary>
        /// 構造体のサイズを 256 バイトに調整するためのパディングを取得します。
        /// </summary>
        public byte[] Padding
        {
            get { return this.padding; }
        }

        /// <summary>
        /// リージョンの名前を取得または設定します。
        /// </summary>
        public string Name
        {
            get { return this.name; }

            set
            {
                if (value == null || value.Length >= MaxName)
                {
                    throw new Exception("invalid region name.");
                }

                this.name = value;
            }
        }

        /// <summary>
        /// Name を 64 バイトに調整するためのパディングを取得します。
        /// </summary>
        public byte[] NamePadding
        {
            get { return new byte[MaxName - this.Name.Length]; }
        }
    }
}
