﻿namespace NintendoWare.SoundFoundation.FileFormats.NintendoSdkBinary
{
    using System;
    using System.Runtime.InteropServices;
    using NintendoWare.SoundFoundation.Binarization;
    using NintendoWare.ToolDevelopmentKit;

    [StructLayout(LayoutKind.Sequential)]
    [DomElement(Tags = new string[] { "Element" })]
    internal class BinaryFileHeader
    {
        private const int SignatureLength = 8;

        private string signature = "        ";

        public BinaryFileHeader()
        {
            this.ByteOrderMark = 0xfeff;
            this.AlignmentShift = 5;        // 2^5 = 32 バイトアライメント
            this.ResourceAddressSize = 64;  // 64 bit
        }

        //-----------------------------------------------------------------
        // データ構造
        //-----------------------------------------------------------------

        /// <summary>
        /// シグネチャを取得または設定します。
        /// 値は、英数字 8 文字にする必要があります。
        /// </summary>
        public string Signature
        {
            get { return this.signature; }
            set
            {
                if (null == value) { throw new ArgumentNullException("value"); }
                if (SignatureLength != value.Length) { throw new ArgumentOutOfRangeException("value"); }
                this.signature = value;
            }
        }

        /// <summary>
        /// リソースファイルバージョンを取得します。
        /// </summary>
        public UInt32 Version
        {
            get
            {
                // major version 16bit (MSB)
                // minor version  8bit
                // micro version  8bit (LSB)
                return (UInt32)((this.MajorVersion << 16) | (this.MinorVersion << 8) | this.MicroVersion);
            }
        }

        /// <summary>
        /// バイトオーダーマークを取得します。
        /// </summary>
        public UInt16 ByteOrderMark { get; private set; }

        /// <summary>
        /// リソース配置の際に必要なアライメントの指数部を取得または設定します。
        /// 2 を底とする指数表現です。
        /// </summary>
        public byte AlignmentShift { get; set; }

        /// <summary>
        /// リソースのアドレスサイズを取得または設定します。
        /// </summary>
        public byte ResourceAddressSize { get; set; }

        /// <summary>
        /// リソースファイル名へのオフセットを取得または設定します。
        /// </summary>
        public UInt32 OffsetToResourceFileName { get; set; }

        /// <summary>
        /// オフセットからポインタへ解決済みかどうかを示すフラグを取得します。
        /// </summary>
        public UInt16 Flags
        {
            get { return 0; }
        }

        /// <summary>
        /// 最初のブロックへの参照を取得または設定します。
        /// この参照を元にオフセットが出力されます。
        /// </summary>
        [DomObjectReference(
            HasOffset = true, OffsetValueType = typeof(UInt16), OriginTag = "Element")]
        public object FirstBlockReference { get; set; }

        /// <summary>
        /// リロケーションテーブルへのオフセットを取得します。
        /// </summary>
        public UInt32 OffsetToRelacationTable
        {
            get { return 0; }
        }

        /// <summary>
        /// ファイルへの参照を取得または設定します。
        /// この参照を元にファイルサイズが出力されます。
        /// </summary>
        [DomObjectReference(HasSize = true, SizeValueType = typeof(UInt32))]
        public object File { get; set; }

        //-----------------------------------------------------------------
        // パラメータの操作
        //-----------------------------------------------------------------

        [DomIgnoreField]
        public UInt16 MajorVersion { get; set; }

        [DomIgnoreField]
        public byte MinorVersion { get; set; }

        [DomIgnoreField]
        public byte MicroVersion { get; set; }
    }
}
