﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------


namespace NintendoWare.SoundFoundation.Conversion
{
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareReport;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.Core.Threading;
    using NintendoWare.SoundFoundation.FileFormats.NintendoSdk;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Utilities;
    using NintendoWare.ToolDevelopmentKit;
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Security.Cryptography;
    using System.Text;
    using System.Threading;
    using System.Xml.Serialization;

    public delegate ISoundProjectConverter2 CreateSoundProjectConverterDelegate(SoundProjectConversionTraits traits);
    public delegate ISoundIDCppHeaderExporter CreateSoundIDCppHeaderExporterDelegate();
    public delegate ISoundArchiveBinaryMapExporter CreateSoundArchiveBinaryMapExporterDelegate();

    /// <summary>
    /// サウンドプロジェクトのコンバート機能を提供します。
    /// </summary>
    public class SoundProjectConvertService : ISoundProjectConvertService
    {
        private const int MaxPath = 256;

        //-----------------------------------------------------------------

        private delegate void SoundProjectConvertEventHandlerInternal(SoundProjectConvertEventArgs e);
        private delegate void OutputTextEventHandlerInternal(OutputLineEventArgs e);

        private CreateSoundProjectConverterDelegate CreateSoundProjectConverter;

        private ISoundIDCppHeaderExporter SoundIDCppHeaderExporter;
        private ISoundArchiveBinaryMapExporter soundArchiveBinaryMapExporter = new SoundArchiveBinaryMapExporter();

        private readonly SoundProjectConversionTraits traits;
        private readonly BankServiceManager bankServiceManager;

        private readonly IInvoker invoker;
        private Thread convertThread;
        private ManualResetEvent endConvertEvent = new ManualResetEvent(true);

        private ISoundProjectConverter2 converter = null;

        private SoundProjectService projectService;
        private SoundSet addonSoundSet;
        private BankServiceReference[] bankServiceRefs;
        private SoundProjectReconvertFilter filter;
        private bool doGarbageCorrection = true;
        private string installDirectoryPath = string.Empty;

        private IEnumerable<SoundSetItem> partsConvertTargets;
        private string partsConvertOutputDirectoryPath;

        private List<string> outputFilePaths = new List<string>();

        private string generator = string.Empty;


        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public SoundProjectConvertService(
            SoundProjectConversionTraits traits,
            IInvoker invoker,
            BankServiceManager bankServiceManager)
        {
            Ensure.Argument.NotNull(traits);
            Ensure.Argument.NotNull(invoker);
            Ensure.Argument.NotNull(bankServiceManager);

            this.traits = traits;
            this.invoker = invoker;
            this.bankServiceManager = bankServiceManager;

            this.ParallelConversionCountMax = 0;
            this.IsPreConvertCommandsIgnored = false;
            this.IsPostConvertCommandsIgnored = false;

            this.CreateSoundProjectConverter = ((t) => new SoundProjectConverter(t));
            this.SoundIDCppHeaderExporter = new SoundIDCppHeaderExporter();
            this.soundArchiveBinaryMapExporter = new SoundArchiveBinaryMapExporter();
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 作成者を取得または設定します。
        /// </summary>
        public string Generator
        {
            get { return this.generator; }
            set
            {
                Ensure.Argument.NotNull(value);
                generator = value;
            }
        }

        /// <summary>
        /// 出力ディレクトリを取得します。
        /// </summary>
        public string OutputDirectoryPath
        {
            get
            {
                return Path.Combine(
                            Path.GetDirectoryName(TargetProjectFilePath),
                            this.projectService.Project.OutputDirectoryPath);
            }
        }

        /// <summary>
        /// バイナリサウンドアーカイブファイルパスを取得します。
        /// </summary>
        public string BinarySoundArchiveFilePath
        {
            get
            {
                return Path.Combine(
                    this.OutputDirectoryPath,
                    Path.GetFileNameWithoutExtension(TargetProjectFilePath) + "." + this.traits.BinaryOutputTraits.SoundArchiveBinaryFileExtension);
            }
        }

        /// <summary>
        /// 並列コンバートスレッドの最大数を取得または設定します。
        /// </summary>
        public uint ParallelConversionCountMax { get; set; }

        /// <summary>
        /// コンバート前コマンドを無効化するかどうかを取得または設定します。
        /// </summary>
        public bool IsPreConvertCommandsIgnored { get; set; }

        /// <summary>
        /// コンバート後コマンドを無効化するかどうかを取得または設定します。
        /// </summary>
        public bool IsPostConvertCommandsIgnored { get; set; }

        /// <summary>
        /// アイテムの除外フィルタ設定を取得または設定します。
        /// このプロパティは、NintendoSDK 専用なので ISoundProjectConvertService には存在しません。
        /// </summary>
        public ItemFilterSettings ExcludeItemFilterSettings { get; set; }

        /// <summary>
        /// アイテムの包含フィルタ設定を取得または設定します。
        /// このプロパティは、NintendoSDK 専用なので ISoundProjectConvertService には存在しません。
        /// </summary>
        public ItemFilterSettings IncludeItemFilterSettings { get; set; }

        /// <summary>
        /// コンバート中かどうかを調べます。
        /// </summary>
        public bool IsConverting
        {
            get { return this.projectService != null; }
        }

        /// <summary>
        /// 進捗状況の最大値を取得します。
        /// </summary>
        public int ProgressMax
        {
            get
            {
                return this.converter != null ? this.converter.ProgressMax : 0;
            }
        }

        /// <summary>
        /// 進捗状況の現在値を取得します。
        /// </summary>
        public int ProgressCurrent
        {
            get
            {
                return this.converter != null ? this.converter.ProgressCurrent : 0;
            }
        }

        /// <summary>
        /// キャンセルされたかどうかを調べます。
        /// </summary>
        protected bool IsCanceled
        {
            get
            {
                if (this.converter == null)
                {
                    return false;
                }

                return this.converter.IsCanceled;
            }
        }

        /// <summary>
        /// コンバート対象のサウンドプロジェクトファイルパスを取得します。
        /// </summary>
        protected string TargetProjectFilePath
        {
            get { return this.projectService.ProjectDocument.Resource.Key; }
        }

        /// <summary>
        /// 作業ディレクトリパスを取得します。
        /// </summary>
        protected string WorkDirectoryPath
        {
            get { return Path.GetDirectoryName(TargetProjectFilePath); }
        }

        /// <summary>
        /// 部分コンバートかどうかを取得します。
        /// </summary>
        private bool IsPartsConvert
        {
            get
            {
                return this.partsConvertTargets != null &&
                    !string.IsNullOrEmpty(this.partsConvertOutputDirectoryPath);
            }
        }

        /// <summary>
        /// コンバートが開始されると発生します。
        /// </summary>
        public event EventHandler<SoundProjectConvertEventArgs> BeginConvert;

        /// <summary>
        /// コンバートが終了すると発生します。
        /// </summary>
        public event EventHandler<SoundProjectConvertEventArgs> EndConvert;

        /// <summary>
        /// ライン出力の際に発生します。
        /// </summary>
        public event EventHandler<OutputLineEventArgs> OutputLine;

        /// <summary>
        /// コンバートします。
        /// </summary>
        /// <param name="projectService">コンバートするプロジェクトサービスを指定します。</param>
        /// <param name="settings">コンバート設定を指定します。</param>
        public void Convert(SoundProjectService projectService, SoundProjectConvertSettings settings)
        {
            ConvertImpl(projectService, settings);
        }

        /// <summary>
        /// 指定アイテムのみを部分コンバートします。
        /// </summary>
        /// <remarks>
        /// 部分コンバートでは他のアイテムとの波形共有はできません。
        /// グループのコンバートも非対応です。
        /// </remarks>
        /// <param name="projectService">コンバートするプロジェクトサービスを指定します。</param>
        /// <param name="settings">コンバート設定を指定します。</param>
        public void ConvertParts(SoundProjectService projectService, SoundProjectConvertPartsSettings settings)
        {
            ConvertPartsImpl(projectService, settings);
        }

        /// <summary>
        /// コンバートを中止します。
        /// </summary>
        public void Cancel()
        {
            if (this.converter != null)
            {
                this.converter.Cancel();
            }
        }

        /// <summary>
        /// コンバートが完了するまで待機します。
        /// </summary>
        public void Wait()
        {
            this.endConvertEvent.WaitOne();
        }

        /// <summary>
        /// コンバートが開始されると発生します。
        /// </summary>
        protected virtual void OnBeginConvertEvent(SoundProjectConvertEventArgs e)
        {
            if (null != BeginConvert)
            {
                BeginConvert(this, e);
            }
        }

        /// <summary>
        /// コンバートが終了すると発生します。
        /// </summary>
        /// <param name="result"></param>
        protected virtual void OnEndConvertEvent(SoundProjectConvertEventArgs e)
        {
            try
            {
                if (null != EndConvert)
                {
                    EndConvert(this, e);
                }
            }
            finally
            {
                // this.projectService等をクリアするため、必ず最後に実行します。
                this.OnEndConvertEventInternal(this, e);
            }
        }

        /// <summary>
        /// ライン出力の際に発生します。
        /// </summary>
        /// <param name="e">ライン出力イベントデータ。</param>
        protected void OnOutputLine(OutputLineEventArgs e)
        {
            if (null != OutputLine)
            {
                OutputLine(this, e);
            }
        }

        private static bool CompareFiles(string filePath1, string filePath2)
        {
            if (!File.Exists(filePath1) || !File.Exists(filePath2))
            {
                return false;
            }

            using (var fileStream1 = File.OpenRead(filePath1))
            {
                using (var fileStream2 = File.OpenRead(filePath2))
                {
                    var md5 = MD5.Create();
                    byte[] firstHash = md5.ComputeHash(fileStream1);
                    byte[] secondHash = md5.ComputeHash(fileStream2);

                    for (int i = 0; i < firstHash.Length; i++)
                    {
                        if (firstHash[i] != secondHash[i])
                        {
                            return false;
                        }
                    }

                    return true;
                }
            }
        }

        private static void InstallFile(string installFilePath, string newFilePath)
        {
            if (CompareFiles(installFilePath, newFilePath))
            {
                if (File.Exists(newFilePath))
                {
                    File.Delete(newFilePath);
                }
            }
            else
            {
                if (File.Exists(installFilePath))
                {
                    File.Delete(installFilePath);
                }

                File.Move(newFilePath, installFilePath);
            }
        }

        /// <summary>
        /// コンバートが終了すると発生します。
        /// </summary>
        /// <param name="result"></param>
        private void OnEndConvertEventInternal(object sender, SoundProjectConvertEventArgs args)
        {
            if (this.bankServiceRefs != null)
            {
                foreach (BankServiceReference bankServiceRef in this.bankServiceRefs)
                {
                    try
                    {
                        bankServiceRef.Close();
                    }
                    catch (Exception e)
                    {
                        // 一時的に開いたバンクを閉じる時の例外は Release 版では無視します。
                        Debug.Assert(false, e.Message);
                    }
                }
            }

            this.bankServiceRefs = null;
            this.addonSoundSet = null;
            this.projectService = null;
            this.converter = null;
        }

        private void RaiseOutputLine(OutputLine line)
        {
            Assertion.Argument.NotNull(line);
            this.RaiseOutputLine(new OutputLine[] { line });
        }

        private void RaiseOutputLine(OutputLine[] lines)
        {
            Assertion.Argument.NotNull(lines);
            this.invoker.BeginInvoke(
               () => this.OnOutputLine(new OutputLineEventArgs(lines))
            );
        }

        private void OnConverterLineOutput(object sender, OutputLineEventArgs e)
        {
            this.invoker.BeginInvoke(
               () => this.OnOutputLine(new OutputLineEventArgs(e.Lines))
            );
        }

        private void OnPostConvert(object sender, CustomConversionEventArgs e)
        {
            try
            {
                this.outputFilePaths.AddRange(e.OutputFilePaths);
                this.outputFilePaths.AddRange(this.OutputSoundArchiveBinaryMap());
                this.outputFilePaths.AddRange(this.OutputSoundIDCppHeader());
                this.outputFilePaths.AddRange(this.OutputHtmlSoundList());
            }
            catch (Exception exception)
            {
                e.Logger.AddLine(new ErrorLine(exception.Message));
            }
        }

        private void ConvertImpl(
            SoundProjectService projectService,
            SoundProjectConvertSettings settings)
        {
            Assertion.Argument.NotNull(projectService);
            Assertion.Argument.NotNull(projectService.ProjectDocument);

            if (this.projectService != null) { return; }

            // ファイル名の最大長を設定します。
            // 正確ではないが、拡張ファイル出力パスとキャッシュ出力パスの長さだけ差し引きます。
            int maxFileName = Math.Min(
                MaxPath - projectService.ProjectIntermediateOutputPath.Length - 1,
                MaxPath - projectService.ProjectExternalFilesOutputPath.Length - 1);

            this.traits.MaxFileName = Math.Max(0, maxFileName);

            this.bankServiceRefs = this.OpenBankServices(projectService);
            this.projectService = projectService;
            this.filter = settings.ReconvertFilter;
            this.doGarbageCorrection = true;
            this.installDirectoryPath = settings.InstallDirectoryPath;
            this.partsConvertTargets = null;
            this.partsConvertOutputDirectoryPath = null;

            this.endConvertEvent.Reset();

            try
            {
                // コンバートを開始する
                this.convertThread = new Thread(new ThreadStart(ConvertThreadProc));
                this.convertThread.Start();
            }
            catch
            {
                this.endConvertEvent.Set();
                throw;
            }
        }

        private void ConvertPartsImpl(
            SoundProjectService projectService,
            SoundProjectConvertPartsSettings settings)
        {
            Assertion.Argument.NotNull(projectService);
            Assertion.Argument.NotNull(projectService.ProjectDocument);

            if (this.projectService != null) { return; }

            // ファイル名の最大長を設定します。
            // 正確ではないが、拡張ファイル出力パスとキャッシュ出力パスの長さだけ差し引きます。
            int maxFileName = Math.Min(
                MaxPath - projectService.ProjectIntermediateOutputPath.Length - 1,
                MaxPath - projectService.ProjectExternalFilesOutputPath.Length - 1);

            this.traits.MaxFileName = Math.Max(0, maxFileName);

            this.bankServiceRefs = this.GetBankServiceReferenceByTargets(settings.Targets);
            this.projectService = projectService;
            this.filter = settings.ReconvertFilter;
            this.doGarbageCorrection = settings.DoGarbageCorrection;
            this.installDirectoryPath = null;
            this.partsConvertTargets = settings.Targets;
            this.partsConvertOutputDirectoryPath = settings.OutputDirectoryPath;
            Directory.CreateDirectory(this.partsConvertOutputDirectoryPath);

            this.endConvertEvent.Reset();

            try
            {
                // コンバートを開始する
                this.convertThread = new Thread(new ThreadStart(ConvertThreadProc));
                this.convertThread.Start();
            }
            catch
            {
                this.endConvertEvent.Set();
                throw;
            }
        }

        private BankServiceReference[] GetBankServiceReferenceByTargets(IEnumerable<SoundSetItem> targets)
        {
            List<BankServiceReference> result = new List<BankServiceReference>();

            foreach (SoundSetItem soundSetItem in targets)
            {
                IEnumerable<SoundSetBankBase> soundSetBanks = null;

                if (soundSetItem is SoundSetBankBase == true)
                {
                    soundSetBanks = new[] { soundSetItem as SoundSetBankBase };
                }
                else if (soundSetItem is SequenceSoundBase)
                {
                    SequenceSoundBase sequenceSound = soundSetItem as SequenceSoundBase;
                    soundSetBanks = sequenceSound.SoundSetBanks.Where((s) => s != null);
                }

                if (soundSetBanks == null)
                {
                    continue;
                }

                foreach (SoundSetBankBase soundSetBank in soundSetBanks)
                {
                    if (soundSetBank.IsConvertTarget() == false)
                    {
                        continue;
                    }

                    try
                    {
                        BankServiceReference bankServiceRef = this.bankServiceManager.OpenItem(soundSetBank.FilePath);

                        if (bankServiceRef != null)
                        {
                            BankServiceReference bsr = null;

                            bsr = result.Where((b) => b.Target == bankServiceRef.Target).FirstOrDefault();
                            if (bsr == null)
                            {
                                result.Add(bankServiceRef);
                            }
                            else
                            {
                                bankServiceRef.Close();
                            }
                        }
                    }
                    // 例外になって読めなかった場合はエラーにして無視します。
                    catch (FileNotFoundException e)
                    {
                        string message = string.Format(Resources.MessageResource.Message_BankFileNotFound, e.FileName);
#if false // ログからアイテムにジャンプしようとすると例外になるので今はジャンプなし版。
                        RaiseOutputLine(new ErrorLine(message, soundSetBank as Component));
#else
                        RaiseOutputLine(new ErrorLine(message));
#endif
                    }
                    catch (Exception e)
                    {
#if false // ログからアイテムにジャンプしようとすると例外になるので今はジャンプなし版。
                        RaiseOutputLine(new ErrorLine(e.Message, soundSetBank as Component[]));
#else
                        RaiseOutputLine(new ErrorLine(e.Message));
#endif
                    }
                }
            }

            return result.ToArray();
        }

        /// <summary>
        /// サウンドセットバンクに登録されているバンクファイルを開きます。
        /// ただし無効化されているサウンドセットバンクは除外されます。
        /// </summary>
        private BankServiceReference[] OpenBankServices(SoundProjectService projectService)
        {
            Assertion.Argument.NotNull(projectService);
            Assertion.Argument.NotNull(projectService.ProjectDocument);

            List<BankServiceReference> result = new List<BankServiceReference>();
            HashSet<string> bankFilePaths = new HashSet<string>();

            foreach (SoundSetBankBase soundSetBank in projectService.SoundSetBanks)
            {
                if (soundSetBank.IsConvertTarget() == false)
                {
                    continue;
                }

                if (bankFilePaths.Contains(soundSetBank.FilePath))
                {
                    continue;
                }

                result.Add(
                    this.bankServiceManager.OpenItem(soundSetBank.FilePath)
                    );
                bankFilePaths.Add(soundSetBank.FilePath);
            }

            return result.ToArray();
        }

        private void ConvertThreadProc()
        {
            Thread.CurrentThread.InitializeCurrentUICulture();

            bool succeeded = false;

            try
            {
                this.invoker.Invoke(
                    () => this.OnBeginConvertEvent(
                        new SoundProjectConvertEventArgs(this.projectService, true, this.IsCanceled, this.IsPartsConvert)
                        )
                    );

                if (this.IsPartsConvert == true)
                {
                    if (this.CanPartsConvert() == false)
                    {
                        succeeded = false;
                        return;
                    }
                }

                succeeded = ConvertProcess();
            }
            catch (Exception ex)
            {
                RaiseOutputLine(new ErrorLine(ex.Message));
                succeeded = false;
            }
            finally
            {
                this.invoker.BeginInvoke(
                    () => this.OnEndConvertEvent(
                        new SoundProjectConvertEventArgs(this.projectService, succeeded, this.IsCanceled, this.IsPartsConvert)
                        )
                    );
                this.convertThread = null;

                this.endConvertEvent.Set();
            }
        }

        private bool CanPartsConvert()
        {
            // this.partsConvertTargets は１つだけだが、
            // 他が複数対応でコーディングされているので、ここも複数対応で記述する。
            foreach (SoundSetItem soundSetItem in this.partsConvertTargets)
            {
                if (soundSetItem is SoundSetBankBase == true &&
                    this.bankServiceManager.Contains((soundSetItem as SoundSetBankBase).FilePath) == false)
                {
                    return false;
                }
            }

            return true;
        }

        private bool ConvertProcess()
        {
            this.converter = this.CreateSoundProjectConverter(this.traits);

            this.converter.Settings.ParallelConversionCountMax = this.ParallelConversionCountMax;
            this.converter.Settings.IsPreConvertCommandsIgnored = this.IsPreConvertCommandsIgnored;
            this.converter.Settings.IsPostConvertCommandsIgnored = this.IsPostConvertCommandsIgnored;
            this.converter.Settings.ExcludeStringTable = this.projectService.Project.ExcludeStringTable;
            this.converter.Settings.DoWarnUnReferencedItems = this.projectService.Project.DoWarnUnreferencedItems;
            this.converter.Settings.ExcludeItemFilterSettings = this.ExcludeItemFilterSettings;
            this.converter.Settings.IncludeItemFilterSettings = this.IncludeItemFilterSettings;

            this.converter.LineOutput += OnConverterLineOutput;
            this.converter.PostConvert += OnPostConvert;

            this.outputFilePaths.Clear();

            try
            {
                this.converter.Prepare(this.projectService);

                if (this.IsPartsConvert == false) // 通常（一括）コンバートなら
                {
                    this.converter.ExecutePreConvertCommands();

                    // サウンドアーカイブ
                    this.addonSoundSet = null;
                    this.converter.Run(
                        null,
                        this.bankServiceRefs.Select(bankServiceRef => bankServiceRef.Target),
                        this.filter != SoundProjectReconvertFilter.All ? this.IsCacheUse : (Func<string, bool>)null,
                        this.filter == SoundProjectReconvertFilter.All);
                    this.converter.Wait();

                    // 追加サウンドアーカイブ
                    foreach (var addonSoundSet in
                        this.projectService.Project.SoundArchiveOutputTypes.Where(item => item.Value == SoundArchiveOutputTypes.AddonSoundArchive).Select(item => item.Key))
                    {
                        this.addonSoundSet = addonSoundSet;
                        this.converter.Run(
                            addonSoundSet,
                            this.bankServiceRefs.Select(bankServiceRef => bankServiceRef.Target),
                            this.filter != SoundProjectReconvertFilter.All ? this.IsCacheUse : (Func<string, bool>)null,
                            this.filter == SoundProjectReconvertFilter.All);
                        this.converter.Wait();
                    }

                    if (this.projectService.Project.DoDeleteGarbageOutputBinaries)
                    {
                        this.DeleteGarbageOutputBinaries();
                    }

                    this.converter.ExecutePostConvertCommands(this.outputFilePaths);
                }
                else
                {
                    this.converter.RunParts(
                        this.bankServiceRefs.Select(bankServiceRef => bankServiceRef.Target),
                        this.partsConvertTargets,
                        this.partsConvertOutputDirectoryPath,
                        this.doGarbageCorrection);
                    this.converter.Wait();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception.ToString());
                return false;
            }
            finally
            {
                this.converter.Cleanup();
            }

            return this.converter.IsSucceeded;
        }

        private void DeleteGarbageOutputBinaries()
        {
            var binaryFileExtensions = new string[]
                    {
                        "." + this.traits.BinaryOutputTraits.SoundArchiveBinaryFileExtension,
                        "." + this.traits.BinaryOutputTraits.StreamSoundBinaryFileExtension,
                        "." + this.traits.BinaryOutputTraits.GroupBinaryFileExtension
                    };

            // 出力ディレクトリから不要なファイル（今回出力していないファイル）を削除する
            // bfsar, bfstm, bfgrp が対象
            // bfsar が見つかった場合は、付随する fsid, xml, html も合わせて削除する
            foreach (var path in Directory.EnumerateFiles(this.projectService.ProjectOutputPath, "*", SearchOption.AllDirectories)
                .Where(path => binaryFileExtensions.Contains(Path.GetExtension(path), StringComparer.CurrentCultureIgnoreCase))
                .Where(path => !this.outputFilePaths.Contains(PathUtility.GetFullPath(path), StringComparer.CurrentCultureIgnoreCase)))
            {
                try
                {
                    File.Delete(path);

                    if (string.Compare(Path.GetExtension(path), "." + this.traits.BinaryOutputTraits.SoundArchiveBinaryFileExtension, true) == 0)
                    {
                        var targetArchiveName = Path.GetFileNameWithoutExtension(path);

                        this.EnumerateOutputsFilePathsForSoundArchive(targetArchiveName)
                            .Where(targetFilePath => File.Exists(targetFilePath))
                            .ForEach(targetFilePath => File.Delete(targetFilePath));
                    }
                }
                catch
                {
                }
            }

            // 出力ディレクトリから空ディレクトリを削除する
            foreach (var path in Directory.EnumerateDirectories(this.projectService.ProjectOutputPath)
                .Where(path => Directory.EnumerateFiles(path).IsEmpty()))
            {
                try
                {
                    Directory.Delete(path, true);
                }
                catch
                {
                }
            }
        }

        /// <summary>
        /// サウンドアーカイブに関連する出力ファイルパスを列挙します。
        /// 存在しないファイルパスも含む。
        /// </summary>
        /// <param name="soundArchiveName">サウンドアーカイブ名を指定します。</param>
        /// <returns>ファイルパスの列挙子を返します。</returns>
        private IEnumerable<string> EnumerateOutputsFilePathsForSoundArchive(string soundArchiveName)
        {
            yield return this.GetSoundIDCppHeaderFilePath(soundArchiveName);

            yield return this.GetSoundIDCppHeaderFilePathWithItemName(soundArchiveName, "SOUND");
            yield return this.GetSoundIDCppHeaderFilePathWithItemName(soundArchiveName, "WSDSET");
            yield return this.GetSoundIDCppHeaderFilePathWithItemName(soundArchiveName, "SEQSET");
            yield return this.GetSoundIDCppHeaderFilePathWithItemName(soundArchiveName, "BANK");
            yield return this.GetSoundIDCppHeaderFilePathWithItemName(soundArchiveName, "PLAYER");
            yield return this.GetSoundIDCppHeaderFilePathWithItemName(soundArchiveName, "WAVEARCHIVE");
            yield return this.GetSoundIDCppHeaderFilePathWithItemName(soundArchiveName, "GROUP");

            yield return this.GetSoundArchiveBinaryXmlFilePath(soundArchiveName);

            yield return this.GetSoundArchiveBinaryMapFilePath(soundArchiveName);
        }

        private bool IsSoundProjectDependentFileDirty(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            if (!File.Exists(filePath)) { return true; }

            DateTime lastTime = File.GetLastWriteTime(filePath);

            foreach (SoundSetDocument document in this.projectService.SoundSetDocuments)
            {
                if (File.GetLastWriteTime(document.Resource.Key) >= lastTime)
                {
                    return true;
                }
            }

            if (File.GetLastWriteTime(TargetProjectFilePath) >= lastTime)
            {
                return true;
            }

            return false;
        }

        private string GetSoundArchiveName()
        {
            Assertion.Operation.ObjectNotNull(this.projectService);

            return this.addonSoundSet == null
                ? this.projectService.Project.Name
                : this.addonSoundSet.Name;
        }

        private string GetSoundArchiveBinaryXmlFilePath()
        {
            return this.GetSoundArchiveBinaryXmlFilePath(this.GetSoundArchiveName());
        }

        private string GetSoundArchiveBinaryMapFilePath()
        {
            return this.GetSoundArchiveBinaryMapFilePath(this.GetSoundArchiveName());
        }

        private string GetSoundIDCppHeaderFilePath()
        {
            return this.GetSoundIDCppHeaderFilePath(this.GetSoundArchiveName());
        }

        private string GetSoundIDCppHeaderFilePathWithItemName(string itemName)
        {
            return this.GetSoundIDCppHeaderFilePathWithItemName(this.GetSoundArchiveName(), itemName);
        }

        private string GetSoundArchiveBinaryXmlFilePath(string soundArchiveName)
        {
            return PathUtility.GetFullPath(
                Path.Combine(
                    this.projectService.ProjectOutputPath,
                    soundArchiveName + ".xml"));
        }

        private string GetSoundArchiveBinaryMapFilePath(string soundArchiveName)
        {
            return PathUtility.GetFullPath(
                Path.Combine(
                    this.projectService.ProjectOutputPath,
                    soundArchiveName + "Map.html"));
        }

        private string GetSoundIDCppHeaderFilePath(string soundArchiveName)
        {
            return PathUtility.GetFullPath(
                Path.Combine(
                    this.projectService.ProjectOutputPath,
                    soundArchiveName + "." +
                    this.traits.IntermediateOutputTraits.SoundIDCppHeaderFileExtension));
        }

        private string GetSoundIDCppHeaderFilePathWithItemName(string soundArchiveName, string itemName)
        {
            return PathUtility.GetFullPath(
                Path.Combine(
                    this.projectService.ProjectOutputPath,
                    soundArchiveName + "." +
                    itemName + "." +
                    this.traits.IntermediateOutputTraits.SoundIDCppHeaderFileExtension));
        }

        private string GetSoundIDCppHeaderTempFilePath()
        {
            return this.GetSoundIDCppHeaderFilePath() + ".tmp";
        }

        private string GetSoundIDCppHeaderTempFilePathWithItemName(string itemName)
        {
            return this.GetSoundIDCppHeaderFilePathWithItemName(itemName) + ".tmp";
        }

        private string[] OutputSoundArchiveBinaryMap()
        {
            var result = new List<string>();

            string outputFilePath = this.GetSoundArchiveBinaryMapFilePath();
            result.Add(outputFilePath);

            if (this.filter == SoundProjectReconvertFilter.All ||
                this.IsSoundProjectDependentFileDirty(outputFilePath))
            {
                using (FileStream soundArchiveBinaryXml = File.Open(
                    this.GetSoundArchiveBinaryXmlFilePath(),
                    FileMode.Open,
                    FileAccess.Read,
                    FileShare.Read))
                {
                    Directory.CreateDirectory(Path.GetDirectoryName(outputFilePath));

                    using (FileStream output =
                        File.Open(outputFilePath, FileMode.Create, FileAccess.Write))
                    {
                        this.soundArchiveBinaryMapExporter.Export(soundArchiveBinaryXml, output);
                    }
                }
            }

            return result.ToArray();
        }

        private string[] OutputSoundIDCppHeader()
        {
            var result = new List<string>();

            string outputFilePath = this.GetSoundIDCppHeaderFilePath();
            string tempFilePath = this.GetSoundIDCppHeaderTempFilePath();

            result.Add(outputFilePath);

            if (this.projectService.Project.OutputLabel)
            {
                result.Add(this.GetSoundIDCppHeaderFilePathWithItemName("SOUND"));
                result.Add(this.GetSoundIDCppHeaderFilePathWithItemName("WSDSET"));
                result.Add(this.GetSoundIDCppHeaderFilePathWithItemName("SEQSET"));
                result.Add(this.GetSoundIDCppHeaderFilePathWithItemName("BANK"));
                result.Add(this.GetSoundIDCppHeaderFilePathWithItemName("PLAYER"));
                result.Add(this.GetSoundIDCppHeaderFilePathWithItemName("WAVEARCHIVE"));
                result.Add(this.GetSoundIDCppHeaderFilePathWithItemName("GROUP"));
            }

            if (this.filter == SoundProjectReconvertFilter.All ||
                this.IsSoundProjectDependentFileDirty(outputFilePath))
            {
                using (var soundArchiveBinaryXmlFile = File.Open(
                    this.GetSoundArchiveBinaryXmlFilePath(),
                    FileMode.Open,
                    FileAccess.Read,
                    FileShare.Read))
                {
                    var soundArchiveBinaryXml = (SoundArchiveBinaryXml)new XmlSerializer(typeof(SoundArchiveBinaryXml)).Deserialize(soundArchiveBinaryXmlFile);

                    Directory.CreateDirectory(Path.GetDirectoryName(outputFilePath));

                    using (TextWriter writer = new StreamWriter(tempFilePath, false, Encoding.UTF8))
                    {
                        this.SoundIDCppHeaderExporter.Export(writer, soundArchiveBinaryXml);
                    }
                    InstallFile(outputFilePath, tempFilePath);

                    if (this.projectService.Project.OutputLabel == false)
                    {
                        return result.ToArray();
                    }

                    outputFilePath = this.GetSoundIDCppHeaderFilePathWithItemName("SOUND");
                    tempFilePath = this.GetSoundIDCppHeaderTempFilePathWithItemName("SOUND");
                    using (TextWriter writer = new StreamWriter(tempFilePath, false, Encoding.UTF8))
                    {
                        this.SoundIDCppHeaderExporter.ExportSoundIDs(writer, soundArchiveBinaryXml);
                    }
                    InstallFile(outputFilePath, tempFilePath);

                    outputFilePath = this.GetSoundIDCppHeaderFilePathWithItemName("WSDSET");
                    tempFilePath = this.GetSoundIDCppHeaderTempFilePathWithItemName("WSDSET");
                    using (TextWriter writer = new StreamWriter(tempFilePath, false, Encoding.UTF8))
                    {
                        this.SoundIDCppHeaderExporter.ExportSoundGroupWaveSoundSetIDs(writer, soundArchiveBinaryXml);
                    }
                    InstallFile(outputFilePath, tempFilePath);

                    outputFilePath = this.GetSoundIDCppHeaderFilePathWithItemName("SEQSET");
                    tempFilePath = this.GetSoundIDCppHeaderTempFilePathWithItemName("SEQSET");
                    using (TextWriter writer = new StreamWriter(tempFilePath, false, Encoding.UTF8))
                    {
                        this.SoundIDCppHeaderExporter.ExportSoundGroupSequenceSoundSetIDs(writer, soundArchiveBinaryXml);
                    }
                    InstallFile(outputFilePath, tempFilePath);

                    outputFilePath = this.GetSoundIDCppHeaderFilePathWithItemName("BANK");
                    tempFilePath = this.GetSoundIDCppHeaderTempFilePathWithItemName("BANK");
                    using (TextWriter writer = new StreamWriter(tempFilePath, false, Encoding.UTF8))
                    {
                        this.SoundIDCppHeaderExporter.ExportSoundSetBankIDs(writer, soundArchiveBinaryXml);
                    }
                    InstallFile(outputFilePath, tempFilePath);

                    outputFilePath = this.GetSoundIDCppHeaderFilePathWithItemName("PLAYER");
                    tempFilePath = this.GetSoundIDCppHeaderTempFilePathWithItemName("PLAYER");
                    using (TextWriter writer = new StreamWriter(tempFilePath, false, Encoding.UTF8))
                    {
                        this.SoundIDCppHeaderExporter.ExportPlayerIDs(writer, soundArchiveBinaryXml);
                    }
                    InstallFile(outputFilePath, tempFilePath);

                    outputFilePath = this.GetSoundIDCppHeaderFilePathWithItemName("WAVEARCHIVE");
                    tempFilePath = this.GetSoundIDCppHeaderTempFilePathWithItemName("WAVEARCHIVE");
                    using (TextWriter writer = new StreamWriter(tempFilePath, false, Encoding.UTF8))
                    {
                        this.SoundIDCppHeaderExporter.ExportWaveArchiveIDs(writer, soundArchiveBinaryXml);
                    }
                    InstallFile(outputFilePath, tempFilePath);

                    outputFilePath = this.GetSoundIDCppHeaderFilePathWithItemName("GROUP");
                    tempFilePath = this.GetSoundIDCppHeaderTempFilePathWithItemName("GROUP");
                    using (TextWriter writer = new StreamWriter(tempFilePath, false, Encoding.UTF8))
                    {
                        this.SoundIDCppHeaderExporter.ExportGroupIDs(writer, soundArchiveBinaryXml);
                    }
                    InstallFile(outputFilePath, tempFilePath);
                }
            }

            return result.ToArray();
        }

        private string[] OutputHtmlSoundList()
        {
            var result = new List<string>();

            foreach (SoundListOutput soundListOutput in
                            this.projectService.Project.SoundListOutputSettings.ListOutputs)
            {
                string outputFilePath = Path.Combine(
                    Path.GetDirectoryName(this.projectService.ProjectFilePath),
                    soundListOutput.FilePath
                    ).GetFullPath();
                result.Add(outputFilePath);

                Directory.CreateDirectory(Path.GetDirectoryName(outputFilePath));

                if (this.filter == SoundProjectReconvertFilter.All ||
                    this.IsSoundProjectDependentFileDirty(outputFilePath))
                {
                    using (TextWriter writer = new StreamWriter(outputFilePath, false, Encoding.UTF8))
                    {
                        SoundProjectReportExporter exporter = new SoundProjectReportExporter()
                        {
                            Generator = this.generator,
                        };

                        exporter.Export(writer, this.projectService, soundListOutput);
                    }
                }
            }

            return result.ToArray();
        }

        public void SetCreateSoundProjectConverter(CreateSoundProjectConverterDelegate createSoundProjectConverter)
        {
            if (createSoundProjectConverter == null)
            {
                this.CreateSoundProjectConverter = ((traits) => new SoundProjectConverter(traits));
            }
            else
            {
                this.CreateSoundProjectConverter = createSoundProjectConverter;
            }
        }

        public void SetSoundIDCppHeaderExporter(CreateSoundIDCppHeaderExporterDelegate createSoundIDCppHeaderExporterDelegate)
        {
            if (createSoundIDCppHeaderExporterDelegate == null)
            {
                this.SoundIDCppHeaderExporter = new SoundIDCppHeaderExporter();
            }
            else
            {
                this.SoundIDCppHeaderExporter = createSoundIDCppHeaderExporterDelegate();
            }
        }

        public void SetSoundArchiveBinaryMapExporter(CreateSoundArchiveBinaryMapExporterDelegate CreateSoundArchiveBinaryMapExporterDelegate)
        {
            if (CreateSoundArchiveBinaryMapExporterDelegate == null)
            {
                this.soundArchiveBinaryMapExporter = new SoundArchiveBinaryMapExporter();
            }
            else
            {
                this.soundArchiveBinaryMapExporter = CreateSoundArchiveBinaryMapExporterDelegate();
            }
        }

        /// <summary>
        /// キャッシュ利用の有無を取得します。
        /// </summary>
        /// <param name="path">キャシュファイルパスを指定します。</param>
        /// <returns>キャッシュ利用の有無を返します。</returns>
        private bool IsCacheUse(string path)
        {
            if (this.filter == SoundProjectReconvertFilter.None)
            {
                return true;
            }

            if (this.filter.True(SoundProjectReconvertFilter.StreamSound) &&
                (path.EndsWith(this.traits.BinaryOutputTraits.StreamSoundBinaryFileExtension) ||
                path.EndsWith(this.traits.BinaryOutputTraits.StreamSoundPrefetchBinaryFileExtension)))
            {
                return false;
            }

            if (this.filter.True(SoundProjectReconvertFilter.WaveSoundSet) &&
                (path.EndsWith(this.traits.BinaryOutputTraits.WaveSoundBinaryFileExtension)
                || path.EndsWith(this.traits.BinaryOutputTraits.WaveSoundBinary2FileExtension)))
            {
                return false;
            }

            if (this.filter.True(SoundProjectReconvertFilter.SequenceSound) &&
                path.EndsWith(this.traits.BinaryOutputTraits.SequenceSoundBinaryFileExtension))
            {
                return false;
            }

            if (this.filter.True(SoundProjectReconvertFilter.Bank) &&
                path.EndsWith(this.traits.BinaryOutputTraits.BankBinaryFileExtension))
            {
                return false;
            }

            if (this.filter.True(SoundProjectReconvertFilter.Group) &&
                path.EndsWith(this.traits.BinaryOutputTraits.GroupBinaryFileExtension))
            {
                return false;
            }

            return true;
        }
    }
}
