﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.IO;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// コンバートのファイル出力のうち１ファイルごとの情報を管理するオブジェクトです。
    /// </summary>
    internal class FileOutputItem : IOutputItem
    {
        public delegate bool IsDirtyHandler(string filePath);

        private string filePath;
        private IsDirtyHandler isDirtyHandler = null;

        public FileOutputItem()
        {
            this.filePath = string.Empty;
        }

        public FileOutputItem(string filePath)
        {
            Ensure.Argument.NotNull(filePath);
            Ensure.Argument.StringNotEmpty(filePath);

            this.filePath = filePath;
        }

        public FileOutputItem(string filePath, IsDirtyHandler isDirtyHandler)
        {
            Ensure.Argument.NotNull(filePath);
            Ensure.Argument.StringNotEmpty(filePath);
            Ensure.Argument.NotNull(isDirtyHandler);

            this.filePath = filePath;
            this.isDirtyHandler = isDirtyHandler;
        }

        /// <summary>
        /// 出力が Dirty かどうかを調べます。
        /// </summary>
        public bool IsDirty
        {
            get
            {
                if (this.isDirtyHandler == null)
                {
                    return true;
                }

                return this.isDirtyHandler(this.filePath);
            }
        }

        /// <summary>
        /// 出力パスを取得します。
        /// </summary>
        public string Path
        {
            get { return this.filePath; }
        }

        /// <summary>
        /// 指定したキーに関連する読み込み用の出力ストリームを開きます。
        /// </summary>
        /// <returns>入力ストリームを返します。</returns>
        public Stream OpenRead()
        {
            return File.Open(this.filePath, FileMode.Open, FileAccess.Read, FileShare.Read);
        }

        /// <summary>
        /// 指定したキーに関連する書き込み用の出力ストリームを開きます。
        /// </summary>
        /// <returns>出力ストリームを返します。</returns>
        public Stream OpenWrite()
        {
            Directory.CreateDirectory(System.IO.Path.GetDirectoryName(this.filePath));
            return File.Open(this.filePath, FileMode.Create, FileAccess.Write);
        }

        /// <summary>
        /// クリーンします。
        /// </summary>
        public void Clean()
        {
            if (File.Exists(this.filePath))
            {
                try
                {
                    File.Delete(this.filePath);
                }
                catch
                {
                }
            }
        }
    }
}
