﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// コンバート処理に関連する中間生成物、成果物を一括管理します。
    /// </summary>
    internal class FileManager
    {
        private SoundIntermediateOutputTraits intermeidateOutputTraits;
        private SoundBinaryOutputTraits binaryOutputTraits;
        private FileIDAggregateFactory fileIDFactory;
        private CacheManager cacheManager;
        private string outputDirectoryRelativePath = string.Empty;

        private Dictionary<FileID, CacheOutput> outputFiles; // 出力ファイルディクショナリ

        public FileManager()
            : this(null, null)
        {
        }

        public FileManager(
            SoundIntermediateOutputTraits intermeidateOutputTraits,
            SoundBinaryOutputTraits binaryOutputTraits)
        {
            this.intermeidateOutputTraits = (intermeidateOutputTraits == null) ?
                new SoundIntermediateOutputTraits() : intermeidateOutputTraits;

            this.binaryOutputTraits = (binaryOutputTraits == null) ?
                new SoundBinaryOutputTraits() : binaryOutputTraits;
        }

        public SoundIntermediateOutputTraits IntermeidateOutputTraits
        {
            get { return this.intermeidateOutputTraits; }
        }

        public SoundBinaryOutputTraits BinaryOutputTraits
        {
            get { return this.binaryOutputTraits; }
        }

        public void Initialize(InitializeArgs args)
        {
            Ensure.Argument.NotNull(args);
            Ensure.Argument.NotNull(args.DependFilePath);
            Ensure.Argument.NotNull(args.OutputDirectoryPath);
            Ensure.Argument.StringNotEmpty(args.DependFilePath);
            Ensure.Argument.NotNull(args.CacheDiretoryPath);

            this.outputDirectoryRelativePath = args.OutputDirectoryPath;

            this.fileIDFactory = new FileIDAggregateFactory(
                Path.GetDirectoryName(args.DependFilePath),
                args.IsWaveSound2Enabled);

            this.cacheManager = new CacheManager();
            this.cacheManager.Open(
                args.DependFilePath,
                args.OutputDirectoryPath,
                args.CacheDiretoryPath,
                args.IsCacheUseFunc,
                args.IsAllCachesClean);

            this.outputFiles = new Dictionary<FileID, CacheOutput>(NoCaseStringComparer.Instance);

            this.InitializeCaches(args.IsKeepAllCaches);
        }

        public void SaveDependencies(bool doGarbageCorrection)
        {
            this.cacheManager.Save(doGarbageCorrection);
        }

        public void Clean()
        {
            this.outputFiles.Clear();
            this.cacheManager.Clean();
        }

        /// <summary>
        /// 不要なファイルを削除します。
        /// </summary>
        public void CollectGarbage()
        {
            this.cacheManager.CollectGarbage();
        }

        public IOutput GetOutput(Component component)
        {
            Assertion.Argument.NotNull(component);

            FileID fileID = this.fileIDFactory.Create(component.GetType().Name, component);
            IOutput output = null;

            if (this.outputFiles.ContainsKey(fileID))
            {
                output = this.outputFiles[fileID];
            }
            else
            {
                output = this.CreateOutput(fileID);
            }

            this.cacheManager.KeepCache(fileID, (output as CacheOutput).Output);

            return output;
        }

        public IOutput GetOutput(Component component, string subKey)
        {
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(subKey);

            FileID fileID = this.fileIDFactory.Create(component.GetType().Name + subKey, component);
            IOutput output = null;

            if (this.outputFiles.ContainsKey(fileID))
            {
                output = this.outputFiles[fileID];
            }
            else
            {
                output = this.CreateOutput(fileID);
            }

            this.cacheManager.KeepCache(fileID, (output as CacheOutput).Output);

            return output;
        }

        public IOutputItem RegisterOutputItem(IOutput outputTarget, string key, string itemFilePath)
        {
            Ensure.Argument.NotNull(outputTarget);
            Ensure.Argument.NotNull(key);
            Ensure.Argument.NotNull(itemFilePath);

            string outputItemFilePath = Path.Combine(this.outputDirectoryRelativePath, itemFilePath).Replace('\\', '/');

            if (outputTarget.ItemDictionary.ContainsKey(key))
            {
                if ((outputTarget.ItemDictionary[key] as CacheOutputItem).OutputItem.Name == outputItemFilePath)
                {
                    return outputTarget.ItemDictionary[key];
                }

                var oldOutputTargetItem = outputTarget.ItemDictionary[key] as CacheOutputItem;
                Ensure.Operation.ObjectNotNull(oldOutputTargetItem);

                this.cacheManager.InvalidateCacheItem(oldOutputTargetItem.OutputItem);
                outputTarget.ItemDictionary.Remove(key);
            }

            IOutputItem outputTargetItem = new CacheOutputItem(
                outputTarget,
                this.cacheManager.CreateCacheItem(key, outputItemFilePath, true)
                );
            outputTarget.ItemDictionary.Add(key, outputTargetItem);

            return outputTargetItem;
        }

        public IOutputItem RegisterCacheItem(IOutput outputTarget, string key, string itemFilePath)
        {
            return this.RegisterCacheItem(outputTarget, key, itemFilePath, true);
        }

        public IOutputItem RegisterCacheItem(IOutput outputTarget, string key, string itemFilePath, bool isAutoNameCorrection)
        {
            Ensure.Argument.NotNull(outputTarget);
            Ensure.Argument.NotNull(key);
            Ensure.Argument.NotNull(itemFilePath);

            if (outputTarget.ItemDictionary.ContainsKey(key))
            {
                if ((outputTarget.ItemDictionary[key] as CacheOutputItem).OutputItem.Name == itemFilePath)
                {
                    return outputTarget.ItemDictionary[key];
                }

                outputTarget.ItemDictionary.Remove(key);
            }

            IOutputItem outputTargetItem = new CacheOutputItem(
                outputTarget,
                this.cacheManager.CreateCacheItem(key, itemFilePath, isAutoNameCorrection)
                );
            outputTarget.ItemDictionary.Add(key, outputTargetItem);

            return outputTargetItem;
        }

        public void RegisterDependedFile(IOutput outputTarget, string dependedFilePath)
        {
            Ensure.Argument.NotNull(outputTarget);
            Ensure.Argument.True(outputTarget is CacheOutput);
            Ensure.Argument.NotNull(dependedFilePath);

            this.cacheManager.RegisterDependedFile((outputTarget as CacheOutput).Output, dependedFilePath);
        }

        public void RegisterDependedFile(IOutput outputTarget, string dependedFilePath, HashCode hashCode)
        {
            Ensure.Argument.NotNull(outputTarget);
            Ensure.Argument.True(outputTarget is CacheOutput);
            Ensure.Argument.NotNull(dependedFilePath);

            this.cacheManager.RegisterDependedFile((outputTarget as CacheOutput).Output, dependedFilePath, hashCode);
        }

        public void UnregisterDependedFile(IOutput outputTarget, string dependedFilePath)
        {
            Ensure.Argument.NotNull(outputTarget);
            Ensure.Argument.True(outputTarget is CacheOutput);
            Ensure.Argument.NotNull(dependedFilePath);

            this.cacheManager.UnregisterDependedFile((outputTarget as CacheOutput).Output, dependedFilePath);
        }

        public void RegisterDependedOutputItem(IOutput outputTarget, IOutput dependedOutput)
        {
            Ensure.Argument.NotNull(outputTarget);
            Ensure.Argument.True(outputTarget is CacheOutput);
            Ensure.Argument.NotNull(dependedOutput);
            Ensure.Argument.True(dependedOutput is CacheOutput);

            this.cacheManager.RegisterDependedOutputItem(
                (outputTarget as CacheOutput).Output, (dependedOutput as CacheOutput).Output);
        }

        private void InitializeCaches(bool isKeepAllCaches)
        {
            foreach (FileID fileID in this.cacheManager.ValidCacheFileIDs)
            {
                IDependentOutput output = this.cacheManager.GetCache(fileID);
                if (output == null)
                {
                    continue;
                }

                CacheOutput cacheOutput = new CacheOutput(output);
                this.outputFiles.Add(fileID, cacheOutput);

                foreach (IDependentOutputItem outputItem in output.OutputItems)
                {
                    cacheOutput.ItemDictionary.Add(
                        outputItem.Key,
                        new CacheOutputItem(cacheOutput, outputItem)
                        );
                }

                if (isKeepAllCaches)
                {
                    foreach (var dependency in output.Dependencies)
                    {
                        dependency.CurrentHashCode = dependency.LastUpdatedHashCode;
                    }

                    this.cacheManager.KeepCache(fileID, output);
                }
            }
        }

        private CacheOutput CreateOutput(FileID fileID)
        {
            IDependentOutput output = this.cacheManager.CreateCache(fileID);

            CacheOutput newOutput = new CacheOutput(output);
            this.outputFiles.Add(fileID, newOutput);

            return newOutput;
        }

        public class InitializeArgs
        {
            public string DependFilePath { get; set; }
            public string OutputDirectoryPath { get; set; }
            public string CacheDiretoryPath { get; set; }
            public Func<string, bool> IsCacheUseFunc { get; set; }
            public bool IsAllCachesClean { get; set; }
            public bool IsKeepAllCaches { get; set; }
            public bool IsWaveSound2Enabled { get; set; }
        }

        #region ** NoCaseStringComparer

        private class NoCaseStringComparer : IEqualityComparer<FileID>
        {
            private static NoCaseStringComparer InstanceInternal = new NoCaseStringComparer();

            private NoCaseStringComparer() { }

            public static NoCaseStringComparer Instance
            {
                get { return NoCaseStringComparer.InstanceInternal; }
            }

            /// <summary>
            /// 指定した string が等しいかどうかを判断します。
            /// </summary>
            /// <param name="x">string1。</param>
            /// <param name="y">string2。</param>
            /// <returns>指定した string が等しい場合は true。それ以外の場合は false。</returns>
            public bool Equals(FileID x, FileID y)
            {
                if (null == x) { throw new ArgumentNullException("x"); }
                if (null == y) { throw new ArgumentNullException("y"); }

                return string.Compare(x.Value, y.Value, true) == 0;
            }

            /// <summary>
            /// 指定した FileID のハッシュ コードを返します。
            /// </summary>
            /// <param name="text">string。</param>
            /// <returns>ハッシュコード</returns>
            public int GetHashCode(FileID value)
            {
                if (null == value) { throw new ArgumentNullException("value"); }
                return value.Value.ToLower().GetHashCode();
            }
        }

        #endregion
    }
}
