﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.IO;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// コンバートのキャッシュ出力のうち１ファイルごとの情報を管理するオブジェクトです。
    /// </summary>
    internal class CacheOutputItem : IOutputItem
    {
        private IOutput outputTarget;
        private IDependentOutputItem dependentOutputItem;

        public CacheOutputItem(IOutput outputTarget, IDependentOutputItem dependentOutputItem)
        {
            Ensure.Argument.NotNull(outputTarget);
            Ensure.Argument.NotNull(dependentOutputItem);
            this.outputTarget = outputTarget;
            this.dependentOutputItem = dependentOutputItem;
        }

        /// <summary>
        /// 出力が Dirty かどうかを調べます。
        /// </summary>
        public bool IsDirty
        {
            get { return this.outputTarget.IsDirty; }
        }

        /// <summary>
        /// 出力アイテムを取得します。
        /// </summary>
        public IDependentOutputItem OutputItem
        {
            get { return this.dependentOutputItem; }
        }

        /// <summary>
        /// 出力パスを取得します。
        /// </summary>
        public string Path
        {
            get { return this.dependentOutputItem.AbsoluteFilePath; }
        }

        /// <summary>
        /// 読み込み用の出力ストリームを開きます。
        /// </summary>
        /// <returns>出力ストリームを返します。</returns>
        public Stream OpenRead()
        {
            return File.Open(
                this.dependentOutputItem.AbsoluteFilePath,
                FileMode.Open, FileAccess.Read, FileShare.Read);
        }

        /// <summary>
        /// 書き込み用の出力ストリームを開きます。
        /// </summary>
        /// <returns>出力ストリームを返します。</returns>
        public Stream OpenWrite()
        {
            string filePath = this.dependentOutputItem.AbsoluteFilePath;

            Directory.CreateDirectory(System.IO.Path.GetDirectoryName(filePath));
            return File.Open(filePath, FileMode.Create, FileAccess.Write);
        }

        /// <summary>
        /// クリーンします。
        /// </summary>
        public void Clean()
        {
            this.dependentOutputItem.DeleteFile();
        }
    }
}
