﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.Linq;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.Collections;

    /// <summary>
    /// コンバートのキャッシュ出力を示すオブジェクトです。
    /// 最終成果物とは区別されます。
    /// 複数のファイル出力アイテムを１つの出力としてまとめます。
    /// </summary>
    internal class CacheOutput : IOutput
    {
        private IDependentOutput output;

        private ObservableDictionary<string, IOutputItem> itemDictionary =
            new ObservableDictionary<string, IOutputItem>();

        public CacheOutput(IDependentOutput output)
        {
            Ensure.Argument.NotNull(output);
            this.output = output;

            this.itemDictionary.CollectionChanged += OnItemDirectoryChanged;
        }

        /// <summary>
        /// 出力を取得します。
        /// </summary>
        public IDependentOutput Output
        {
            get { return this.output; }
        }

        /// <summary>
        /// 出力が Dirty かどうかを調べます。
        /// </summary>
        public bool IsDirty
        {
            get { return this.output.Dirty; }
        }

        /// <summary>
        /// 出力対象アイテム辞書を取得します。
        /// </summary>
        public IDictionary<string, IOutputItem> ItemDictionary
        {
            get { return this.itemDictionary; }
        }

        /// <summary>
        /// 依存ファイルパスの列挙子を取得します。
        /// </summary>
        public IEnumerable<string> DependentFilePaths
        {
            get
            {
                return this.output.Dependencies.Select(item => item.AbsoluteFilePath);
            }
        }

        private void AddOutputItem(IOutputItem item)
        {
            Ensure.Argument.NotNull(item);
            Ensure.Argument.True(item is CacheOutputItem);

            IDependentOutputItem dependentOutputItem = (item as CacheOutputItem).OutputItem;

            if (this.output.OutputItems.Contains(dependentOutputItem.Key))
            {
                Ensure.Operation.True(this.output.OutputItems[dependentOutputItem.Key] == dependentOutputItem);
                return;
            }

            this.output.OutputItems.Add(dependentOutputItem);
        }

        private void RemoveOutputItem(IOutputItem item)
        {
            Ensure.Argument.NotNull(item);
            Ensure.Argument.True(item is CacheOutputItem);

            IDependentOutputItem dependentOutputItem = (item as CacheOutputItem).OutputItem;

            this.output.OutputItems.Remove(dependentOutputItem);
        }

        private void OnItemDirectoryChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (KeyValuePair<string, IOutputItem> item in e.NewItems)
                    {
                        this.AddOutputItem(item.Value);
                    }
                    return;

                case NotifyCollectionChangedAction.Remove:
                    foreach (KeyValuePair<string, IOutputItem> item in e.OldItems)
                    {
                        this.RemoveOutputItem(item.Value);
                    }
                    return;
            }

            throw new NotImplementedException("internal error : unexpected operation.");
        }
    }
}
