﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// コンバート処理に関連するキャッシュを一括管理します。
    /// </summary>
    internal class CacheManager
    {
        private const string CacheDirectoryPathDefault = "cache";
        private const string UserKeyFileID = "FileID";
        private const string UserParameterKeyOutputDirectoryPath = "OutputDirectoryPath";

        private string dependFilePath = string.Empty;
        private DependencyManager dependenciesOutput = null;   // 依存管理クラス（出力用）
        private DependencyManager dependenciesInput = null;    // 依存管理クラス（入力用）

        private Dictionary<FileID, IDependentOutput> validCaches = new Dictionary<FileID, IDependentOutput>();
        private HashSet<string> formatlessFileIDs = new HashSet<string>();
        private HashSet<string> validCacheFilePaths = new HashSet<string>();
        private HashSet<string> garbageCacheFilePaths = new HashSet<string>();

        public IEnumerable<FileID> ValidCacheFileIDs
        {
            get { return this.validCaches.Keys; }
        }

        public void Open(
            string dependFilePath,
            string outputDirectoryPath,
            Func<string, bool> isCacheUse,
            bool isAllCachesClean)
        {
            Open(
                dependFilePath,
                outputDirectoryPath,
                CacheDirectoryPathDefault,
                isCacheUse,
                isAllCachesClean);
        }

        public void Open(
            string dependFilePath,
            string outputDirectoryPath,
            string cacheDiretoryPath,
            Func<string, bool> isCacheUse,
            bool isAllCachesClean)
        {
            Ensure.Argument.NotNull(dependFilePath);
            Ensure.Argument.NotNull(outputDirectoryPath);
            Ensure.Argument.StringNotEmpty(dependFilePath);
            Ensure.Argument.NotNull(cacheDiretoryPath);

            this.Initialize(dependFilePath, outputDirectoryPath, cacheDiretoryPath);

            if (!File.Exists(dependFilePath)) { return; }

            try
            {
                this.dependenciesInput.Load(dependFilePath);

                if (isAllCachesClean)
                {
                    this.dependenciesInput.Clean();
                }
                else if (isCacheUse != null)
                {
                    foreach (var output in this.dependenciesInput.Outputs.Values)
                    {
                        // isCacheUse() が false を返すパスがあったら、その出力をクリーンします。
                        if (output.OutputItems.
                            Select(item => item.AbsoluteFilePath).
                            Where(path => !string.IsNullOrEmpty(path)).
                            Where(path => !isCacheUse(path)).
                            Any())
                        {
                            output.Clean();
                        }
                    }
                }

                this.CollectGarbageDirectOutputs(outputDirectoryPath);
                this.InitializeDependenciesOutput();
            }
            catch
            {
                this.dependenciesOutput.ManagementFilePath = dependFilePath;
                this.dependenciesOutput.BaseDirectoryPath = cacheDiretoryPath;

                this.dependenciesInput.ManagementFilePath = dependFilePath;
            }
        }

        public void Save(bool doGarbageCorrection)
        {
            if (doGarbageCorrection)
            {
                // ガベージコレクトします。
                CollectGarbage();
            }

            // 隠し属性を解除して保存します。
            if (File.Exists(this.dependenciesOutput.ManagementFilePath))
            {
                File.SetAttributes(this.dependenciesOutput.ManagementFilePath, FileAttributes.Normal);
            }

            this.dependenciesOutput.Save();
        }

        public void Clean()
        {
            if (this.dependenciesOutput != null)
            {
                this.dependenciesOutput.Clean();
            }
        }

        /// <summary>
        /// 不要なファイルを削除します。
        /// </summary>
        public void CollectGarbage()
        {
            HashSet<string> useFiles = new HashSet<string>();

            // 出力した項目を不要ファイルから除外します。
            foreach (IDependentOutput output in this.dependenciesOutput.Outputs.Values)
            {
                foreach (IDependentOutputItem outputItem in output.OutputItems)
                {
                    useFiles.Add(outputItem.AbsoluteFilePath);

                    if (this.garbageCacheFilePaths.Contains(outputItem.AbsoluteFilePath))
                    {
                        this.garbageCacheFilePaths.Remove(outputItem.AbsoluteFilePath);
                    }
                }
            }

            // 読み込んだキャッシュのうち、出力した項目に含まれないものを不要ファイルとします。
            foreach (FileID fileID in this.validCaches.Keys)
            {
                if (this.dependenciesOutput.Outputs.ContainsUserKey(UserKeyFileID, fileID.Value))
                {
                    continue;
                }

                FileID formatlessfileID = fileID.Clone();
                formatlessfileID.Format = "*";

                // フォーマット違いのファイルIDが含まれている場合は、有効キャッシュとみなします。
                if (this.formatlessFileIDs.Contains(formatlessfileID.Value))
                {
                    this.KeepCache(fileID, this.validCaches[fileID]);
                    continue;
                }

                IDependentOutput invalidOutput = this.validCaches[fileID];

                // 使用されなくなったキャッシュの出力ファイルを削除対象にします。
                foreach (IDependentOutputItem outputItem in invalidOutput.OutputItems)
                {
                    if (useFiles.Contains(outputItem.AbsoluteFilePath)) { continue; }
                    this.InvalidateCacheItem(outputItem);
                }
            }

            foreach (string filePath in this.garbageCacheFilePaths)
            {
                try
                {
                    File.Delete(filePath);
                }
                catch { }
            }

            this.garbageCacheFilePaths.Clear();
        }

        public void KeepCache(FileID fileID, IDependentOutput cache)
        {
            Assertion.Argument.NotNull(fileID);
            Assertion.Argument.NotNull(cache);

            if (this.dependenciesOutput.Outputs.ContainsKey(cache.Key) ||
                this.dependenciesOutput.Outputs.ContainsUserKey(UserKeyFileID, fileID.Value))
            {
                this.RegisterFormatlessFileID(fileID);
                return;
            }

            this.dependenciesOutput.Add(cache);
            this.RegisterFileID(fileID, cache);
        }

        public IDependentOutput GetCache(FileID fileID)
        {
            Assertion.Argument.NotNull(fileID);

            IDependentOutput cache = this.GetCacheInternal(this.dependenciesOutput, fileID);

            // 以前はここで Dirty なキャッシュを弾いていましたが、
            // 後のタイミングでハッシュコード比較をするために Dirty なキャッシュも残すように変更しました。
            // ※後でも Dirty フラグを評価しているので、問題はないはずです。
            if (cache == null)
            {
                if (!this.validCaches.TryGetValue(fileID, out cache))
                {
                    return null;
                }
            }

            return cache;
        }

        public IDependentOutput CreateCache(FileID fileID)
        {
            Assertion.Argument.NotNull(fileID);

            IDependentOutput cache = this.dependenciesOutput.CreateOutput();
            this.KeepCache(fileID, cache);

            return cache;
        }

        public IDependentOutputItem CreateCacheItem(string key, string itemFilePath, bool isAutoNameCorrection)
        {
            Ensure.Argument.NotNull(key);
            Ensure.Argument.NotNull(itemFilePath);

            string targetFilePath = this.CreateCacheFileName(itemFilePath);

            if (Path.IsPathRooted(targetFilePath))
            {
                targetFilePath = PathEx.MakeRelative(
                    targetFilePath,
                    this.dependenciesOutput.BaseDirectoryAbsolutePath
                    );
            }

            return this.dependenciesOutput.CreateOutputItem(key, targetFilePath, isAutoNameCorrection);
        }

        public void InvalidateCacheItem(IDependentOutputItem cacheItem)
        {
            Ensure.Argument.NotNull(cacheItem);
            this.validCacheFilePaths.Remove(cacheItem.FilePath);
            this.garbageCacheFilePaths.Add(cacheItem.AbsoluteFilePath);
        }

        public void RegisterDependedFile(IDependentOutput cache, string dependedFilePath)
        {
            Ensure.Argument.NotNull(cache);
            Ensure.Argument.NotNull(dependedFilePath);

            string relativePath = PathEx.MakeRelative(
                dependedFilePath,
                this.dependenciesOutput.BaseDirectoryAbsolutePath
                );

            if (cache.Dependencies.Contains(relativePath))
            {
                return;
            }

            cache.Dependencies.Add(this.dependenciesOutput.CreateDependedFileInfo(relativePath));
        }

        public void RegisterDependedFile(IDependentOutput cache, string dependedFilePath, HashCode hashCode)
        {
            Ensure.Argument.NotNull(cache);
            Ensure.Argument.NotNull(dependedFilePath);

            string relativePath = PathEx.MakeRelative(
                dependedFilePath,
                this.dependenciesOutput.BaseDirectoryAbsolutePath
                );

            if (cache.Dependencies.Contains(relativePath))
            {
                cache.Dependencies[relativePath].CurrentHashCode = hashCode;
                return;
            }

            cache.Dependencies.Add(this.dependenciesOutput.CreateDependedFileInfo(relativePath, hashCode));
        }

        public void UnregisterDependedFile(IDependentOutput cache, string dependedFilePath)
        {
            Ensure.Argument.NotNull(cache);
            Ensure.Argument.NotNull(dependedFilePath);

            string relativePath = PathEx.MakeRelative(
                dependedFilePath,
                this.dependenciesOutput.BaseDirectoryAbsolutePath
                );

            IDependedFileInfo target = cache.Dependencies.FirstOrDefault(
                dependency => dependency.FilePath == relativePath);

            if (target != null)
            {
                cache.Dependencies.Remove(target);
            }
        }

        public void RegisterDependedOutputItem(IDependentOutput cache, IDependentOutput outputItem)
        {
            Ensure.Argument.NotNull(cache);
            Ensure.Argument.NotNull(outputItem);

            this.RegisterDependedOutputItem(cache, outputItem.Key);
        }

        public void RegisterDependedOutputItem(IDependentOutput cache, string outputItemKey)
        {
            Ensure.Argument.NotNull(cache);
            Ensure.Argument.NotNull(outputItemKey);

            if (cache.Dependencies.Contains(outputItemKey))
            {
                return;
            }

            cache.Dependencies.Add(
                this.dependenciesOutput.CreateDependedOutputItemInfo(outputItemKey)
                );
        }

        private void Initialize(string dependFilePath, string outputDirectoryPath, string cacheDiretoryPath)
        {
            Assertion.Argument.StringNotEmpty(dependFilePath);
            Assertion.Argument.StringNotEmpty(outputDirectoryPath);
            Assertion.Argument.NotNull(cacheDiretoryPath);

            this.dependFilePath = dependFilePath;

            this.dependenciesOutput = new DependencyManager();
            this.dependenciesOutput.ManagementFilePath = dependFilePath;
            this.dependenciesOutput.BaseDirectoryPath = cacheDiretoryPath;

            this.dependenciesOutput.UserParameters.Add(
                UserParameterKeyOutputDirectoryPath, outputDirectoryPath
                );

            this.dependenciesInput = new DependencyManager();
            this.dependenciesInput.ManagementFilePath = dependFilePath;
        }

        private IDependentOutput GetCacheInternal(
            DependencyManager dependencyManager,
            FileID fileID)
        {
            Assertion.Argument.NotNull(dependencyManager);
            Assertion.Argument.NotNull(fileID);

            if (!dependencyManager.Outputs.ContainsUserKey(UserKeyFileID, fileID.Value))
            {
                return null;
            }

            return dependencyManager.Outputs[UserKeyFileID, fileID.Value];
        }

        private IEnumerable<FileID> EnumerateFileIDs(DependencyManager dependencyManager)
        {
            Assertion.Argument.NotNull(dependencyManager);

            if (!dependencyManager.Outputs.ContainsUserKeyMap(UserKeyFileID))
            {
                yield break;
            }

            foreach (string fileIDValue in dependencyManager.Outputs.GetUserKeys(UserKeyFileID))
            {
                yield return new FileID(fileIDValue);
            }
        }

        private void RegisterFileID(FileID fileID, IDependentOutput cache)
        {
            Assertion.Argument.NotNull(fileID);
            Assertion.Argument.NotNull(cache);
            this.dependenciesOutput.SetUserKey(UserKeyFileID, fileID.Value, cache);

            RegisterFormatlessFileID(fileID);
        }

        private void RegisterFormatlessFileID(FileID fileID)
        {
            Assertion.Argument.NotNull(fileID);

            FileID formatlessFileID = fileID.Clone();
            formatlessFileID.Format = "*";

            string value = formatlessFileID.Value;

            if (this.formatlessFileIDs.Contains(value)) { return; }

            this.formatlessFileIDs.Add(value);
        }

        /// <summary>
        /// 出力ファイル名を作成します。
        /// </summary>
        /// <param name="baseName">ファイルのベースネーム</param>
        /// <returns>新しいファイル名</returns>
        private string CreateCacheFileName(string baseName)
        {
            if (null == baseName) { throw new ArgumentNullException("baseName"); }
            if (0 == baseName.Length) { throw new ArgumentException("baseName"); }

            string directory = Path.GetDirectoryName(baseName);
            string workName = baseName;
            int count = 0;

            while (this.validCacheFilePaths.Contains(workName))
            {
                count++;

                workName = Path.Combine(
                    directory,
                    string.Format("{0}_{1}{2}",
                        Path.GetFileNameWithoutExtension(baseName),
                        count.ToString(),
                        Path.GetExtension(baseName)
                        ));
            }

            return workName;
        }

        /// <summary>
        /// 出力フォルダに直接出力する項目について不要なものを削除します。
        /// </summary>
        /// <param name="newOutputDirectoryPath">出力先フォルダを指定します。</param>
        private void CollectGarbageDirectOutputs(string newOutputDirectoryPath)
        {
            Assertion.Argument.StringNotEmpty(newOutputDirectoryPath);

            if (!this.dependenciesInput.UserParameters.
                ContainsKey(UserParameterKeyOutputDirectoryPath))
            {
                return;
            }

            string oldOutputDirectoryPath =
                this.dependenciesInput.UserParameters[UserParameterKeyOutputDirectoryPath];

            if (oldOutputDirectoryPath == newOutputDirectoryPath)
            {
                return;
            }

            List<IDependentOutput> removeOutputs = new List<IDependentOutput>();

            foreach (IDependentOutput output in this.dependenciesInput.Outputs.Values)
            {
                foreach (IDependentOutputItem outputItem in output.OutputItems)
                {
                    if (outputItem.FilePath.StartsWith(oldOutputDirectoryPath))
                    {
                        removeOutputs.Add(output);
                    }
                }
            }

            foreach (IDependentOutput output in removeOutputs)
            {
                this.dependenciesInput.Remove(output);
            }
        }

        /// <summary>
        /// 依存出力を初期化します。
        /// </summary>
        private void InitializeDependenciesOutput()
        {
            foreach (FileID fileID in this.EnumerateFileIDs(this.dependenciesInput).ToArray())
            {
                IDependentOutput output = this.GetCacheInternal(this.dependenciesInput, fileID);

                if (output == null)
                {
                    continue;
                }

                // 以前はここで Dirty なキャッシュを弾いていましたが、
                // 後のタイミングでハッシュコード比較をするために Dirty なキャッシュも残すように変更しました。
                // ※後でも Dirty フラグを評価しているので、問題はないはずです。
                this.validCaches.Add(fileID, output);

                foreach (IDependentOutputItem outputItem in output.OutputItems)
                {
                    this.validCacheFilePaths.Add(outputItem.FilePath);
                }
            }
        }
    }
}
