﻿namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.FileFormats.NintendoSdkBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoSdkBinary.WaveSound2Elements;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;
    using System;
    using System.Linq;

    /// <summary>
    /// クリップ対応ウェーブサウンドバイナリ DOM の構築をサポートします。
    /// </summary>
    internal class WaveSound2FileBuilder
    {
        private const float ClipVolumeMaxValue = 2.0f;
        private const float ClipPanMaxValue = 1.0f;
        private const byte ClipVolumeBinaryMaxValue = byte.MaxValue;
        private const byte ClipPanBinaryMaxValue = 127;

        public WaveSound2FileBuilder(string signature, BinaryVersion version)
        {
            Ensure.Argument.NotNull(signature);
            this.Signature = signature;
            this.Version = version;
        }

        public string Signature { get; set; }

        public BinaryVersion Version { get; set; }

        public WaveSound2Binary Build(WaveSound waveSound, WaveArchiveBase waveArchive)
        {
            Ensure.Argument.NotNull(waveSound);
            Assertion.Argument.NotNull(waveArchive);

            Ensure.Operation.True(waveSound.IsConvertTarget());

            var binary = new WaveSound2Binary();
            this.BuildInfoBlock(binary.InfoBlock.Body, waveSound, waveArchive);

            return binary;
        }

        public void BuildInfoBlock(InfoBlockBody body, WaveSound waveSound, WaveArchiveBase waveArchive)
        {
            Assertion.Argument.NotNull(body);
            Assertion.Argument.NotNull(waveSound);

            foreach (var trackBinary in waveSound
                .GetTrackConvertModels()
                .Select(track => this.CreateTrackBinary(track, waveArchive)))
            {
                body.Tracks.Add(trackBinary);
            }
        }

        private TrackBinary CreateTrackBinary(WaveSoundTrackConvertModel track, WaveArchiveBase waveArchive)
        {
            Assertion.Argument.NotNull(track);

            var result = new TrackBinary();

            foreach (var clip in track.Children.Cast<WaveSoundClipConvertModel>())
            {
                result.ClipTable.Items.Add(
                    this.CreateClipBinary(clip, waveArchive));
            }

            return result;
        }

        private ClipBinary CreateClipBinary(WaveSoundClipConvertModel clip, WaveArchiveBase waveArchive)
        {
            Assertion.Argument.NotNull(clip);

            return new ClipBinary()
            {
                WaveIndex = this.GetWaveIndex(waveArchive, clip.GetOutputTarget()),
                Position = (UInt32)clip.Model.Position,
                Duration = (UInt32)clip.Model.Duration,
                StartOffset = (UInt32)clip.Model.StartOffset,
                Pitch = clip.Model.Pitch,
                Volume = (byte)(clip.Model.Volume / ClipVolumeMaxValue * ClipVolumeBinaryMaxValue),
                Pan = (byte)(clip.Model.Pan / ClipPanMaxValue * ClipPanBinaryMaxValue),
            };
        }

        private uint GetWaveIndex(WaveArchiveBase waveArchive, IOutput output)
        {
            Assertion.Argument.NotNull(waveArchive);
            Assertion.Argument.NotNull(output);

            uint waveIndex = (uint)waveArchive.GetItemOutputTargets().IndexOf(output);

            if (waveIndex == uint.MaxValue)
            {
                throw new Exception("failed to get wave index.");
            }

            return waveIndex;
        }
    }
}
