﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using Core;
    using Projects;
    using ToolDevelopmentKit;

    internal class WaveArchiveProcessorFactory : ComponentSetup<SoundArchiveContext, WaveArchiveBase>
    {
        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, WaveArchiveBase component)
        {
            IOutput output = component.GetOutputTarget();
            ICollection<GroupBase> attachedGroups = component.GetAttachedGroups();

            // 自動生成されていない場合、または
            // 自動生成されたけども、どの埋め込みグループにも属していない場合は、
            // オリジナルファイルの ConversionProcessor を生成します。
            if (!component.IsAutoGenerated() || attachedGroups.Count == 0)
            {
                IOutputItem outputItem = this.GetOutputItem(component, null);
                context.AddComponentProcessor(output, string.Empty, CreateProcessor(component, outputItem));
                return;
            }

            // 自動生成され、埋め込みグループに属する場合は、
            // グループ内で１つの波形アーカイブにまとめられるので、
            // グループに最適化されたファイルの ConversionProcessor を生成します。
            if (attachedGroups.Count != 1)
            {
                throw new Exception("internal error : group shared wave archive must be attached to a group.");
            }

            GroupBase group = attachedGroups.First();
            IOutputItem outputItemForGroup = this.GetOutputItem(component, group);

            context.AddComponentProcessor(output, group.Name, CreateProcessor(component, outputItemForGroup));
        }

        private WaveArchiveProcessor CreateProcessor(WaveArchiveBase component, IOutputItem outputItem)
        {
            Assertion.Argument.NotNull(component);
            return new WaveArchiveProcessor(component, outputItem);
        }

        private IOutputItem GetOutputItem(WaveArchiveBase waveArchive, GroupBase group)
        {
            Assertion.Argument.NotNull(waveArchive);

            if (group == null)
            {
                return waveArchive.GetOutputTarget().ItemDictionary[string.Empty];
            }

            return waveArchive.GetOutputTarget().ItemDictionary[group.Name];
        }
    }
}
