﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using Projects;
    using ToolDevelopmentKit;

    /// <summary>
    /// シーケンスサウンドに関する ConversionProcessor を生成します。
    /// </summary>
    internal class SequenceSoundProcessorFactory : ComponentSetup<SoundArchiveContext, SequenceSoundBase>
    {
        public const string OutputID_Intermediate = ComponentContext.IntermediateOutputPrefix + "Intermediate";

        private delegate IComponentProcessor CreateProcessorHandler(
            SoundArchiveContext context, SequenceSoundBase component, IOutputItem output);

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, SequenceSoundBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            IOutput output = component.GetOutputTarget();

            // SmfProcessor
            if (component.FileType == SequenceSoundFileType.Smf)
            {
                this.RegisterSmfProcessor(context, component);
            }

            // SequenceSoundProcessor
            this.RegisterProcessor(context, component);
        }

        private void RegisterSmfProcessor(SoundArchiveContext context, SequenceSoundBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            IOutputItem intermediateOutputItem = this.GetIntermediateOutputTargetItem(component);

            if (intermediateOutputItem == null)
            {
                return;
            }

            IOutput output = component.GetOutputTarget();

            if (context.ContainsProcessor(output, SequenceSoundProcessorFactory.OutputID_Intermediate))
            {
                context.GetProcessor(
                    output, SequenceSoundProcessorFactory.OutputID_Intermediate)
                    .Components.Add(component);
                return;
            }

            context.AddComponentProcessor(
                    output,
                    SequenceSoundProcessorFactory.OutputID_Intermediate,
                    this.CreateSmfProcessor(context, component, intermediateOutputItem)
                    );
        }

        private void RegisterProcessor(SoundArchiveContext context, SequenceSoundBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            IOutputItem outputItem = this.GetOutputTargetItem(component);

            if (outputItem == null)
            {
                return;
            }

            IOutput output = component.GetOutputTarget();

            if (context.ContainsProcessor(output, string.Empty))
            {
                context.GetProcessor(output, string.Empty).Components.Add(component);
                return;
            }

            context.AddComponentProcessor(
                    output,
                    string.Empty,
                    this.CreateProcessor(context, component, outputItem)
                    );
        }

        private IComponentProcessor CreateSmfProcessor(
            SoundArchiveContext context, SequenceSoundBase component, IOutputItem output)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(output);

            return new SmfProcessor(component, output)
            {
                ConverterExePath = context.Traits.SmfConverterPath,
                TimeBase = context.Project.SmfConvertTimeBase,
            };
        }

        private IComponentProcessor CreateProcessor(
            SoundArchiveContext context, SequenceSoundBase component, IOutputItem output)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(output);

            return new SequenceSoundProcessor(component, output)
            {
                ConverterExePath = context.Traits.SequenceSoundTextConverterPath,
            };
        }

        private IOutputItem GetOutputTargetItem(SequenceSoundBase sequenceSound)
        {
            Assertion.Argument.NotNull(sequenceSound);
            return sequenceSound.GetOutputTarget().ItemDictionary[string.Empty];
        }

        private IOutputItem GetIntermediateOutputTargetItem(SequenceSoundBase sequenceSound)
        {
            Assertion.Argument.NotNull(sequenceSound);

            IOutputItem intermediateOutputItem = null;
            sequenceSound.GetOutputTarget().ItemDictionary.TryGetValue(
                SequenceSoundProcessorFactory.OutputID_Intermediate, out intermediateOutputItem);

            return intermediateOutputItem;
        }
    }
}
