﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Resources;
    using NintendoWare.ToolDevelopmentKit;

    internal class WaveArchiveOutputFactory : ComponentSetup<SoundArchiveContext, WaveArchiveBase>
    {
        private FileManager fileManager;

        public WaveArchiveOutputFactory(FileManager fileManager)
        {
            Ensure.Argument.NotNull(fileManager);
            this.fileManager = fileManager;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, WaveArchiveBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            IOutput outputTarget = this.fileManager.GetOutput(component);
            component.SetOutputTarget(outputTarget);

            context.AddFile(component, outputTarget);

            ICollection<GroupBase> attachedGroups = component.GetAttachedGroups();

            // 自動生成されていない場合、または
            // 自動生成されたけども、どの埋め込みグループにも属していない場合は、
            // オリジナルファイルを生成します。
            if (!component.IsAutoGenerated() || attachedGroups.Count == 0)
            {
                this.AddOutput(context, outputTarget, component);
                return;
            }

            // 自動生成され、埋め込みグループに属する場合は、
            // グループ内で１つの波形アーカイブにまとめられるので、
            // グループに最適化されたファイルを１つだけ生成します。
            if (attachedGroups.Count != 1)
            {
                throw new Exception("internal error : group shared wave archive must be attached to a group.");
            }

            this.AddGroupDependedOutput(context, outputTarget, component, attachedGroups.First());
        }

        private void AddOutput(SoundArchiveContext context, IOutput outputTarget, WaveArchiveBase waveArchive)
        {
            Assertion.Argument.NotNull(outputTarget);
            Assertion.Argument.NotNull(waveArchive);

            string fileName = CreateFileName(waveArchive);

            if (fileName.Length >= context.Traits.MaxFileName - 1)
            {
                context.Logger.AddLine(
                    new Logs.ErrorLine(
                        string.Format(MessageResource.Message_FilePathTooLong, fileName),
                        waveArchive));
            }

            this.fileManager.RegisterCacheItem(outputTarget, string.Empty, fileName);
        }

        private void AddGroupDependedOutput(SoundArchiveContext context, IOutput outputTarget, WaveArchiveBase waveArchive, GroupBase group)
        {
            Assertion.Argument.NotNull(outputTarget);
            Assertion.Argument.NotNull(waveArchive);
            Assertion.Argument.NotNull(group);

            string fileName = CreateFileName(waveArchive);

            if (fileName.Length >= context.Traits.MaxFileName - 1)
            {
                context.Logger.AddLine(
                    new Logs.ErrorLine(
                        string.Format(MessageResource.Message_FilePathTooLong, fileName),
                        waveArchive));
            }

            this.fileManager.RegisterCacheItem(outputTarget, group.Name, fileName);
        }

        /// <summary>
        /// コンポーネントのファイル名を作成します。
        /// </summary>
        /// <param name="component">コンポーネントを指定します。</param>
        /// <returns>ファイル名を返します。</returns>
        private string CreateFileName(WaveArchiveBase component)
        {
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(this.fileManager.BinaryOutputTraits.WaveArchiveBinaryFileExtension);
            Assertion.Argument.StringNotEmpty(this.fileManager.BinaryOutputTraits.WaveArchiveBinaryFileExtension);

            if (component.Name.Length == 0)
            {
                throw new Exception("internal error : invalid name.");
            }

            return component.Name + "." + this.fileManager.BinaryOutputTraits.WaveArchiveBinaryFileExtension;
        }
    }
}
