﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Resources;
    using Projects;
    using ToolDevelopmentKit;

    internal class SoundSetBankOutputFactory : ComponentSetup<SoundArchiveContext, SoundSetBankBase>
    {
        private FileManager fileManager;

        public SoundSetBankOutputFactory(FileManager fileManager)
        {
            Ensure.Argument.NotNull(fileManager);
            this.fileManager = fileManager;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, SoundSetBankBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            // メインサウンドアーカイブのバンクを追加サウンドアーカイブで参照する場合はバンク情報のみを出力するので、スキップする
            if (context.IsExternalSoundArchiveBank(component))
            {
                return;
            }

            IOutput outputTarget = this.fileManager.GetOutput(component);
            component.SetOutputTarget(outputTarget);

            context.AddFile(component, outputTarget);

            foreach (KeyValuePair<string, WaveArchiveBase> item in component.GetWaveArchiveDictionary())
            {
                // キャッシュを有効化するために GetOutput() しておく
                foreach (var waveArchiveItem in item.Value.GetItems())
                {
                    this.fileManager.GetOutput(waveArchiveItem);
                }

                if (item.Key.Length == 0)
                {
                    this.AddOutput(context, outputTarget, component);
                }
                else
                {
                    GroupBase group = context.Groups.GetValue(item.Key);
                    this.AddGroupDependedOutput(context, outputTarget, component, group);
                }
            }

            this.AddOutputInclude(context, outputTarget, component);
        }

        private void AddOutput(SoundArchiveContext context, IOutput outputTarget, SoundSetBankBase soundSetBank)
        {
            Assertion.Argument.NotNull(outputTarget);
            Assertion.Argument.NotNull(soundSetBank);

            string fileName = this.CreateFileName(soundSetBank);

            if (fileName.Length >= context.Traits.MaxFileName - 1)
            {
                context.Logger.AddLine(
                    new Logs.ErrorLine(
                        string.Format(MessageResource.Message_FilePathTooLong, fileName),
                        soundSetBank));
            }

            this.fileManager.RegisterCacheItem(outputTarget, string.Empty, fileName);
        }

        private void AddGroupDependedOutput(SoundArchiveContext context, IOutput outputTarget, SoundSetBankBase soundSetBank, GroupBase group)
        {
            Assertion.Argument.NotNull(outputTarget);
            Assertion.Argument.NotNull(soundSetBank);
            Assertion.Argument.NotNull(group);

            string fileName = this.CreateFileNameForGroup(soundSetBank, group);

            if (fileName.Length >= context.Traits.MaxFileName - 1)
            {
                context.Logger.AddLine(
                    new Logs.ErrorLine(
                        string.Format(MessageResource.Message_FilePathTooLong, fileName),
                        soundSetBank));
            }

            this.fileManager.RegisterCacheItem(outputTarget, group.Name, fileName);
        }

        private void AddOutputInclude(SoundArchiveContext context, IOutput outputTarget, SoundSetBankBase soundSetBank)
        {
            Assertion.Argument.NotNull(outputTarget);
            Assertion.Argument.NotNull(soundSetBank);

            string filePath = this.CreateIncludeFileFullPath(soundSetBank);

            // .xinl は、.xbnk と同じ場所に出力されるため、フルパスで登録しているので、
            // ConversionTraits.MaxPath と比較する
            // フルパスで登録するのを止めてもよいが、相対パスを許容するなら、
            // プロジェクトフォルダ外が指定されることも考慮しないといけない。
            if (filePath.Length >= ConversionTraits.MaxPath)
            {
                context.Logger.AddLine(
                    new Logs.ErrorLine(
                        string.Format(MessageResource.Message_FilePathTooLong, filePath),
                        soundSetBank));
            }

            // Include ファイルは複数の CacheItem から出力される可能性があるため、
            // isAutoNameCorrection を false にします。
            this.fileManager.RegisterCacheItem(
                outputTarget,
                SoundSetBankProcessorFactory.OutputID_Include,
                filePath,
                false);
        }

        /// <summary>
        /// コンポーネントのファイル名を作成します。
        /// </summary>
        /// <param name="component">コンポーネントを指定します。</param>
        /// <returns>ファイル名を返します。</returns>
        private string CreateFileName(SoundSetBankBase component)
        {
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(this.fileManager.BinaryOutputTraits.BankBinaryFileExtension);
            Assertion.Argument.StringNotEmpty(this.fileManager.BinaryOutputTraits.BankBinaryFileExtension);

            if (component.Name.Length == 0)
            {
                throw new Exception("internal error : invalid name.");
            }

            return component.Name + "." + this.fileManager.BinaryOutputTraits.BankBinaryFileExtension;
        }

        /// <summary>
        /// コンポーネントのファイル名を作成します。
        /// </summary>
        /// <param name="component">コンポーネントを指定します。</param>
        /// <param name="group">コンポーネントが含まれるグループを指定します。</param>
        /// <returns>ファイル名を返します。</returns>
        private string CreateFileNameForGroup(SoundSetBankBase component, GroupBase group)
        {
            Assertion.Argument.NotNull(component);
            Assertion.Argument.NotNull(group);
            Assertion.Argument.NotNull(this.fileManager.BinaryOutputTraits.BankBinaryFileExtension);
            Assertion.Argument.StringNotEmpty(this.fileManager.BinaryOutputTraits.BankBinaryFileExtension);

            if (component.Name.Length == 0)
            {
                throw new Exception("internal error : invalid name.");
            }

            return string.Format("{0}.{1}{2}.{3}",
                component.Name,
                group.Name,
                SoundArchiveContext.AutoGeneratedNamePostfix,
                this.fileManager.BinaryOutputTraits.BankBinaryFileExtension);
        }

        /// <summary>
        /// コンポーネントのインクルードファイルパスを作成します。
        /// </summary>
        /// <param name="component">コンポーネントを指定します。</param>
        /// <returns>ファイルパスを返します。</returns>
        private string CreateIncludeFileFullPath(SoundSetBankBase component)
        {
            Assertion.Argument.NotNull(component);
            Assertion.Argument.StringNotEmpty(this.fileManager.IntermeidateOutputTraits.BankIncludeFileExtension);

            return Path.ChangeExtension(
                component.FilePath,
                this.fileManager.IntermeidateOutputTraits.BankIncludeFileExtension);
        }
    }
}
