﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using Logs;
    using Projects;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.Collections;
    using FileFormats.NintendoSdk;
    using System.Text.RegularExpressions;

    internal class SoundSetBankNameResolver : NameResolver<SoundSetBankBase>
    {
        public SoundSetBankNameResolver(ComponentDictionary componentDictionary)
            : base(componentDictionary)
        {
        }

        private bool IsInstrumentIncluded<Type>(Type item, ItemFilter filter) where Type : Component
        {
            if (filter == null || filter.Patterns == null)
            {
                return true;
            }

            foreach (var pattern in filter.Patterns)
            {
                Regex regexPattern = new Regex(pattern);

                if (regexPattern.Match(item.Name).Success)
                {
                    return true;
                }
            }
            return false;
        }

        private bool IsInstrumentExcluded<Type>(Type item, ItemFilter filter) where Type : Component
        {
            if (filter == null || filter.Patterns == null)
            {
                return false;
            }

            foreach (var pattern in filter.Patterns)
            {
                Regex regexPattern = new Regex(pattern);

                if (regexPattern.Match(item.Name).Success)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="componentDictionary">コンポーネントディクショナリを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected override void RunInternal(
            SoundArchiveContext context, ComponentDictionary componentDictionary, SoundSetBankBase component)
        {
            // 自動設定の場合は ItemGenerator にて処理します。
            if (WaveArchiveConsts.IsConstValue(component.WaveArchiveReference)) { return; }

            // 部分コンバートの場合は、自動生成波形アーカイブを利用するので、ここでは処理しません。
            if (context.Settings.IsConvertParts) { return; }

            WaveArchiveBase waveArchive = GetWaveArchive(componentDictionary, component);

            if (waveArchive == null || !waveArchive.IsConvertTarget())
            {
                context.Logger.AddLine(
                    new ErrorLine(
                        string.Format(Resources.MessageResource.Message_WaveArchiveNotFound, component.WaveArchiveReference),
                        component)
                    );
                return;
            }

            if (context.AddonSoundSet == null)
            {
                // メインサウンドアーカイブから追加サウンドアーカイブのバンクを参照している場合はエラーにする
                if (context.Project.IsItemInAddonSoundArchive(waveArchive))
                {
                    context.Logger.AddLine(
                        new ErrorLine(
                            string.Format(Resources.MessageResource.Message_WaveArchiveNotFoundInSoundArchive, waveArchive.Name),
                            component)
                        );
                    return;
                }
            }
            else
            {
                // 追加サウンドアーカイブからメインサウンドアーカイブのバンクを参照している場合は、
                // 波形アーカイブは参照せず、文字列テーブルにバンク名を抽出するだけなので、依存解決はスキップする
                if (context.Project.IsItemInSoundArchive(waveArchive))
                {
                    return;
                }
            }

            if (!context.BankDictionary.ContainsKey(component))
            {
                throw new Exception("internal error : bank not found.");
            }

            // 逆参照の一覧に追加します。
            WaveArchiveEx.ItemList waveArchiveItems = waveArchive.GetItems();
            Bank bank = context.BankDictionary[component];

            foreach (VelocityRegion velocityRegion in bank.GetVelocityRegions())
            {
                if (!velocityRegion.IsHierarchicalConvertTarget())
                {
                    continue;
                }

                if (waveArchiveItems.Contains(velocityRegion))
                {
                    continue;
                }

                waveArchiveItems.Add(velocityRegion);
            }
        }

        private WaveArchiveBase GetWaveArchive(ComponentDictionary componentDictionary, SoundSetBankBase component)
        {
            Assertion.Argument.NotNull(componentDictionary);
            Assertion.Argument.NotNull(component);

            if (!componentDictionary.Contains(component.WaveArchiveReference)) { return null; }
            return componentDictionary[component.WaveArchiveReference][0] as WaveArchiveBase;
        }
    }
}
