﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.Collections.Generic;
    using Logs;
    using Projects;
    using ToolDevelopmentKit;

    internal class GroupNameResolver : NameResolver<GroupBase>
    {
        public GroupNameResolver(ComponentDictionary componentDictionary)
            : base(componentDictionary)
        {
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="componentDictionary">コンポーネントディクショナリを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected override void RunInternal(
            SoundArchiveContext context, ComponentDictionary componentDictionary, GroupBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(componentDictionary);
            Assertion.Argument.NotNull(component);

            foreach (GroupItemBase item in component.GetAllItems())
            {
                if (item.Target == null)
                {
                    context.Logger.AddLine(
                       new ErrorLine(
                           string.Format(Resources.MessageResource.Message_GroupItemTargetNotFound, item.TargetItemReference),
                           component)
                       );
                    continue;
                }

                // メインサウンドアーカイブのグループから追加サウンドアーカイブのアイテムを参照していたらエラー
                if (context.AddonSoundSet == null)
                {
                    SoundArchiveOutputTypes outputType;

                    if (context.Project.SoundArchiveOutputTypes.TryGetValue(item.Target.SoundSet, out outputType) &&
                        outputType == SoundArchiveOutputTypes.AddonSoundArchive)
                    {
                        context.Logger.AddLine(
                           new ErrorLine(
                               string.Format(
                                   Resources.MessageResource.Message_GroupHasNoAddonSoundArchiveItems,
                                   component.Name,
                                   item.Target.SoundSet.Name,
                                   item.Target.Name),
                               component)
                           );
                        continue;
                    }
                }

                if (context.Project.DoWarnDisableGroupItemTargets && !item.Target.IsConvertTarget())
                {
                    context.Logger.AddLine(
                       new WarningLine(
                           string.Format(Resources.MessageResource.Message_GroupItemTargetIsDisabled, item.TargetItemReference),
                           component)
                       );
                    continue;
                }
            }

            // グループの逆参照を登録します。
            foreach (SoundSetItem item in component.GetAllItemTargets(context.Traits.IsWaveSound2BinaryEnabled))
            {
                AddGroupDereference(context, item, component);
            }
        }

        /// <summary>
        /// グループの逆参照を追加します。
        /// </summary>
        /// <param name="context">サウンドアーカイブコンテキストを指定します。</param>
        /// <param name="groupItemTarget">グループアイテムの参照先を指定します。</param>
        /// <param name="group">グループを指定します。</param>
        private void AddGroupDereference(SoundArchiveContext context, SoundSetItem groupItemTarget, GroupBase group)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(groupItemTarget);
            Assertion.Argument.NotNull(group);

            ICollection<GroupBase> attachedGroups = groupItemTarget.GetAttachedGroups();

            if (!attachedGroups.Contains(group))
            {
                attachedGroups.Add(group);
            }
        }
    }
}
