﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.Linq;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;
    using Nintendo.AudioToolkit.DomainModels;

    internal class WaveSoundItemGenerator : ComponentSetup<SoundArchiveContext, WaveSound>
    {
        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, WaveSound component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            // WSDSET に含まれるWSD ごとに出力する場合（マルチクリップ WSD）
            if (context.Traits.IsWaveSound2BinaryEnabled)
            {
                this.GenerateWaveSoundTrackConvertModels(component);
                this.GenerateWaveSoundClipConvertModels(component);

                if (context.Settings.IsConvertParts)
                {
                    // TODO : 部分コンバート処理 this.GenerateOriginalWaveSoundSet() 呼び出しの WaveSoundResoruce 対応が必要
                }
            }
            // WSDSET で１ファイルを出力する場合（旧来の WSD）
            else
            {
                if (context.Settings.IsConvertParts)
                {
                    this.GenerateOriginalWaveSoundSet(context, component);
                }
            }
        }

        private void GenerateWaveSoundTrackConvertModels(WaveSound waveSound)
        {
            var tracks = waveSound.GetTrackConvertModels();
            tracks.Clear();

            if (waveSound.IsWaveSound2BinaryRequired())
            {
                foreach (var track in waveSound.WaveSoundResource
                    .Tracks
                    .Where(track => track.IsEnabled))
                {
                    tracks.Add(new WaveSoundTrackConvertModel(track));
                }
            }
        }

        private void GenerateWaveSoundClipConvertModels(WaveSound waveSound)
        {
            var tracks = waveSound.GetTrackConvertModels();
            var clips = waveSound.GetClipConvertModels();
            clips.Clear();

            foreach (var track in tracks)
            {
                foreach (var clipModel in track.Model.Clips)
                {
                    var clip = new WaveSoundClipConvertModel(clipModel)
                    {
                        Encoding = waveSound.Encoding,
                    };

                    clips.Add(clip);
                    track.Children.Add(clip);
                }
            }
        }

        /// <summary>
        /// 部分コンバート（sndedit）用に WaveSound を差し替えます。
        /// </summary>
        /// <param name="context">対象のコンテキスト。</param>
        /// <param name="waveSound">
        /// コンテキストに含まれる WaveSound。
        /// この関数において、単一の WaveSound を含む WaveSoundSet に差し替えられます。
        /// </param>
        /// <remarks>
        /// 指定 WaveSound のみを含む bfwsd, bfwar を生成するための対処です。
        /// もし、元の WaveSound をそのまま利用すると、以降の処理工程において、
        /// その親 WaveSoundSet に含まれる兄弟 WaveSound を bfwsd, bfwar に含めようとしてエラーが発生してしまいます。
        /// </remarks>
        private void GenerateOriginalWaveSoundSet(SoundArchiveContext context, WaveSound waveSound)
        {
            var tempWaveSound = new WaveSound();

            foreach (var key in waveSound.Parameters.Keys)
            {
                if (!tempWaveSound.Parameters.ContainsKey(key))
                {
                    tempWaveSound.Parameters.AddValue(key, waveSound.Parameters[key]);
                }
                else
                {
                    tempWaveSound.Parameters[key].Value = waveSound.Parameters[key].Value;
                }
            };

            tempWaveSound.Name = waveSound.Name + SoundArchiveContext.AutoGeneratedNamePostfix;

            var tempWaveSoundSet = new WaveSoundSetBase()
            {
                Name = string.Format(
                    "WaveSoundSet_{0}{1}",
                    waveSound.Name,
                    SoundArchiveContext.AutoGeneratedNamePostfix),
            };

            tempWaveSoundSet.Children.Add(tempWaveSound);

            context.WaveSounds.Remove(waveSound);
            context.WaveSounds.Add(tempWaveSound);
            context.WaveSoundSets.Add(tempWaveSoundSet);

            tempWaveSound.SetOriginalModel(waveSound);
            tempWaveSoundSet.SetOriginalModel(waveSound.Parent as SoundSetItem);

            this.GenerateOriginalWaveArchive(context, tempWaveSoundSet);
        }

        private void DelegateWaves(SoundArchiveContext context, WaveSoundSetBase waveSoundSet, WaveArchiveBase waveArchive)
        {
            Assertion.Argument.NotNull(waveSoundSet);
            Assertion.Argument.NotNull(waveArchive);

            WaveArchiveEx.ItemList waveArchiveItems = waveArchive.GetItems();

            foreach (var waveComponent in waveSoundSet.Children
                .Cast<WaveSound>()
                .Where(waveSound => waveSound.IsConvertTarget())
                .SelectMany(waveSound => waveSound.EnumerateWaveComponents(context.Traits.IsWaveSound2BinaryEnabled)))
            {
                if (!waveComponent.IsConvertTarget())
                {
                    continue;
                }

                if (waveArchiveItems.Contains(waveComponent))
                {
                    continue;
                }

                waveArchiveItems.Add(waveComponent);
            }
        }

        private void GenerateOriginalWaveArchive(SoundArchiveContext context, WaveSoundSetBase waveSoundSet)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(waveSoundSet);

            WaveArchiveBase waveArchive = new WaveArchiveBase()
            {
                Name = string.Format(
                    "{0}_WaveArchive{1}",
                    waveSoundSet.Name, SoundArchiveContext.AutoGeneratedNamePostfix),
            };

            waveSoundSet.GetWaveArchiveDictionary().Add(string.Empty, waveArchive);
            DelegateWaves(context, waveSoundSet, waveArchive);

            waveArchive.ID = new BinaryItemID(ItemType.WaveArchive, (uint)context.WaveArchives.Count);
            context.WaveArchives.Add(waveArchive);
        }
    }
}
