﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using NintendoWare.SoundFoundation.Core;
    using Projects;
    using System.Linq;
    using System.Security.Cryptography;
    using System.Text;
    using ToolDevelopmentKit;

    /// <summary>
    /// ウェーブサウンドセットの依存ファイルをリンクします。
    /// </summary>
    internal class WaveSoundSetDependedFileLinker : ComponentSetup<SoundArchiveContext, WaveSoundSetBase>
    {
        private static readonly HashAlgorithm hashAlgorithm = new SHA1CryptoServiceProvider();

        private SoundProjectService _soundProjectService;
        private FileManager fileManager;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="soundProjectService">SoundProjectService を指定します。</param>
        /// <param name="fileManager">ファイルマネージャを指定します。</param>
        public WaveSoundSetDependedFileLinker(SoundProjectService soundProjectService, FileManager fileManager)
        {
            Ensure.Argument.NotNull(fileManager);
            this.fileManager = fileManager;
            _soundProjectService = soundProjectService;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, WaveSoundSetBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            if (context.Traits.IsWaveSound2BinaryEnabled)
            {
                // WSD ごとに出力する場合
                foreach (var waveSound in component.Children.Cast<WaveSound>().Where(waveSound => waveSound.IsConvertTarget()))
                {
                    Ensure.Operation.True(waveSound.IsWaveSound2BinaryRequired(), "WaveSound must has WaveSoundResource.");

                    IOutput output = waveSound.GetOutputTarget();

                    // 共有波形アーカイブに対応するため、全てのサウンドセットに依存させます。
                    // ただし、ハッシュ値を指定することで、関係のない修正に影響されないようにします。

                    HashCode warcsHashCode = this.ComputeWaveArchivesHash(component);
                    HashCode wsdSetHashCode = component.GetComponentHashCode(hashAlgorithm, null);

                    HashCode hashCode = new HashCode(
                        hashAlgorithm.ComputeHash(
                            wsdSetHashCode.Value.Concat(warcsHashCode.Value).ToArray()
                        ));

                    foreach (string soundSetFilePath in (_soundProjectService.SoundSetDocuments
                                                         .Where(d => context.IsTargetSoundSet(d.SoundSet))
                                                         .Select(d => d.Resource.Key)))
                    {
                        this.fileManager.RegisterDependedFile(output, soundSetFilePath, hashCode);
                    }
                }
            }
            else
            {
                // WSDSET で１つ出力する場合
                IOutput output = component.GetOutputTarget();

                // 共有波形アーカイブに対応するため、全てのサウンドセットに依存させます。
                // ただし、ハッシュ値を指定することで、関係のない修正に影響されないようにします。

                HashCode warcsHashCode = this.ComputeWaveArchivesHash(component);
                HashCode wsdSetHashCode = component.GetComponentHashCode(hashAlgorithm, null);

                HashCode hashCode = new HashCode(
                    hashAlgorithm.ComputeHash(
                        wsdSetHashCode.Value.Concat(warcsHashCode.Value).ToArray()
                    ));

                foreach (string soundSetFilePath in (_soundProjectService.SoundSetDocuments
                                                     .Where(d => context.IsTargetSoundSet(d.SoundSet))
                                                     .Select(d => d.Resource.Key)))
                {
                    this.fileManager.RegisterDependedFile(output, soundSetFilePath, hashCode);
                }
            }
        }

        private HashCode ComputeWaveArchivesHash(WaveSoundSetBase component)
        {
            StringBuilder hashSourceBuilder = new StringBuilder();

            // 関係のない修正が波形アーカイブIDをずらす可能性があるため、
            // ハッシュ値に波形アーカイブIDを含めます。
            foreach (var item in component.GetWaveArchiveDictionary())
            {
                hashSourceBuilder.Append(item.Key + item.Value.ID.ToString());
            }

            // 内包する波形リストのパスを含めます。
            foreach (var item in component.GetWaveArchiveDictionary())
            {
                this.AppendWaveArchiveHashString(item.Value, hashSourceBuilder);
            }

            string waveArchiveIDHashSource = hashSourceBuilder.ToString();

            if (waveArchiveIDHashSource.Length == 0)
            {
                return HashCode.Empty;
            }

            return new HashCode(
                hashAlgorithm.ComputeHash(Encoding.Unicode.GetBytes(waveArchiveIDHashSource)));
        }

        private void AppendWaveArchiveHashString(WaveArchiveBase component, StringBuilder hashSourceBuilder)
        {
            foreach (IOutput waveOutputTarget in component.GetItemOutputTargets())
            {
                if (waveOutputTarget.ItemDictionary.Values.Count == 0)
                {
                    continue;
                }

                hashSourceBuilder.Append(waveOutputTarget.ItemDictionary.Values.First().Path);
            }
        }
    }
}
