﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Linq;
    using System.Security.Cryptography;
    using System.Text;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// グループの依存ファイルをリンクします。
    /// </summary>
    internal class GroupDependedFileLinker : ComponentSetup<SoundArchiveContext, GroupBase>
    {
        private static readonly HashAlgorithm hashAlgorithm = new SHA1CryptoServiceProvider();

        private SoundProjectService _soundProjectService;
        private FileManager fileManager;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="soundProjectService">SoundProjectService を指定します。</param>
        /// <param name="fileManager">ファイルマネージャを指定します。</param>
        public GroupDependedFileLinker(SoundProjectService soundProjectService, FileManager fileManager)
        {
            Ensure.Argument.NotNull(fileManager);

            _soundProjectService = soundProjectService;
            this.fileManager = fileManager;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, GroupBase component)
        {
            Assertion.Argument.NotNull(context);

            if (component.OutputType != GroupOutputType.UserManagement)
            {
                return;
            }

            IOutput outputTarget = component.GetOutputTarget();

            // サウンドセットファイルに依存します。
            string soundSetFilePath = this.GetSoundSetFilePath(component);
            this.fileManager.RegisterDependedFile(outputTarget, soundSetFilePath, this.ComputeHash(context, component));

            // すべてのアイテム出力に依存します。
            foreach (var item in component.GetAllItemTargets(context.Traits.IsWaveSound2BinaryEnabled))
            {
                IOutput itemOutput = item.GetOutputTarget();

                if (itemOutput == null)
                {
                    throw new Exception("item output not found.");
                }

                this.fileManager.RegisterDependedOutputItem(outputTarget, itemOutput);
            }
        }

        /// <summary>
        /// 指定サウンドセットアイテムが属するサウンドセットのエントリーを取得します。
        /// </summary>
        /// <param name="soundSetItem">サウンドセットアイテムを指定します。</param>
        /// <returns>サウンドセットのファイルパスを返します。</returns>
        private string GetSoundSetFilePath(SoundSetItem soundSetItem)
        {
            Assertion.Argument.NotNull(soundSetItem);

            SoundSet soundSet = soundSetItem.SoundSet;
            Ensure.Operation.ObjectNotNull(soundSet);

            return _soundProjectService.SoundSetDocuments
                .Where(d => d.SoundSet == soundSet)
                .Select(d => d.Resource.Key)
                .First();
        }

        private HashCode ComputeHash(SoundArchiveContext context, GroupBase group)
        {
            StringBuilder hashSourceBuilder = new StringBuilder();

            // アイテム ID, ファイル ID
            foreach (var item in group.GetAllItemTargets(context.Traits.IsWaveSound2BinaryEnabled).OrderBy(item => item.ID.Value))
            {
                hashSourceBuilder.Append(
                    string.Format("{0}@{1:X},{2:X};", item.Name, item.ID.Value, context.GetFileID(item))
                    );
            }

            string hashSource = hashSourceBuilder.ToString();

            if (hashSource.Length == 0)
            {
                return HashCode.Empty;
            }

            return new HashCode(
                hashAlgorithm.ComputeHash(Encoding.Unicode.GetBytes(hashSource)));
        }
    }
}
