﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.IO;
    using NintendoWare.SoundFoundation.Binarization;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using ToolDevelopmentKit;

    internal class WaveSoundProcessor : ComponentProcessor<SoundArchiveContext, WaveSoundSetBase>
    {
        private WaveArchiveBase waveArchive;

        public WaveSoundProcessor(WaveSoundSetBase component, WaveArchiveBase waveArchive, IOutputItem outputItem)
            : base(component, outputItem)
        {
            Ensure.Argument.NotNull(waveArchive);
            this.waveArchive = waveArchive;
        }

        protected override void ProcessInternal(SoundArchiveContext context)
        {
            try
            {
                using (Stream stream = this.OutputTargetItem.OpenWrite())
                {
                    Write(context, context.CreateBinaryWriter(stream));
                }
            }
            catch
            {
                try
                {
                    File.Delete(this.OutputTargetItem.Path);
                }
                catch
                {
                }

                throw;
            }
        }

        protected override void PostProcessInternal(SoundArchiveContext context)
        {
            this.SetBinaryPaths(context);
        }

        protected override void OutputLog(SoundArchiveContext context, Component[] components)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(components);

            context.Logger.AddLine(new InformationLine(
                string.Format("[WSD] {0} ({1}) > {2}",
                components[0].Name,
                this.waveArchive.Name,
                Path.GetFileName(this.OutputTargetItem.Path)
                )));
        }

        private void Write(SoundArchiveContext context, BinaryWriter writer)
        {
            Assertion.Argument.NotNull(writer);

            WaveSoundBinary file = new WaveSoundFileBuilder(
                context.Traits.BinaryFileInfo.WaveSoundSignature,
                context.Traits.BinaryFileInfo.WaveSoundVersion).
                Build(this.TargetComponent as WaveSoundSetBase, this.waveArchive);

            DomElement fileElement = new DomBuilder().Build(file);
            new DomWriter(writer).Run(new DomObjectWriter(), fileElement);
        }

        private void SetBinaryPaths(SoundArchiveContext context)
        {
            if (!context.Settings.IsConvertParts)
            {
                return;
            }

            var waveSoundSet = this.TargetComponent as WaveSoundSetBase;
            var waveArchive = waveSoundSet.GetWaveArchiveDictionary()[string.Empty];

            // 部分コンバートでは、必ず 1WSD : 1WSDSET
            Ensure.Operation.True(waveSoundSet.Children.Count == 1);

            var waveSound = (waveSoundSet.Children[0] as SoundSetItem).GetOriginalModel() as WaveSoundBase;
            Ensure.Operation.ObjectNotNull(waveSound);

            // bxwsd ファイルパスを設定します。
            waveSound.SetBinaryFilePathForPartsConvert(this.OutputTargetItem.Path);

            // bxwar ファイルパスを設定します。
            waveSound.SetWaveArchiveBinaryFilePathForPartsConvert(
                waveArchive.GetOutputTarget().ItemDictionary[string.Empty].Path);
        }
    }
}
