﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Binarization;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// サウンドアーカイブを作成します。
    /// </summary>
    internal class SoundArchiveProcessor : ComponentProcessor<SoundArchiveContext, SoundProject>
    {
        private const string CockpitExtension = "cockpit";

        private Dictionary<IOutputItem, object> fileEntities;
        private HashSet<IOutputItem> embededGroupItemOutputs;
        private HashSet<IOutputItem> userManagermentGroupItemOutputs;

        private IOutputItem mapOutputItem;

        public SoundArchiveProcessor(
            SoundProject component, IOutputItem outputItem, IOutputItem mapOutputItem)
            : base(component, outputItem)
        {
            Ensure.Argument.NotNull(mapOutputItem);
            this.mapOutputItem = mapOutputItem;
        }

        protected override void ProcessInternal(SoundArchiveContext context)
        {
            Ensure.Argument.NotNull(context);

            try
            {
                if (File.Exists(this.OutputTargetItem.Path))
                {
                    File.Delete(this.OutputTargetItem.Path);
                }

                using (Stream stream = this.OutputTargetItem.OpenWrite())
                {
                    Write(context, context.CreateBinaryWriter(stream));
                }
            }
            catch
            {
                try
                {
                    if (File.Exists(this.OutputTargetItem.Path))
                    {
                        File.Delete(this.OutputTargetItem.Path);
                    }
                }
                catch
                {
                }

                throw;
            }
        }

        protected override void OutputLog(SoundArchiveContext context, Component[] components)
        {
            context.Logger.AddLine(new InformationLine(Resources.MessageResource.Message_SoundArchiveCreating));
        }

        private void Write(SoundArchiveContext context, BinaryWriter writer)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(writer);

            try
            {
                this.fileEntities = new Dictionary<IOutputItem, object>();
                this.embededGroupItemOutputs = new HashSet<IOutputItem>();
                this.userManagermentGroupItemOutputs = new HashSet<IOutputItem>();

                // 先にグループを展開しておきます。
                // ここで開かれたオリジナルファイルは、サウンドアーカイブの
                // ファイルブロックにファイル（実体）が重複して含まれないようにします。
                BuildGroups(context);

                List<object> soundArchiveFileEntities = new List<object>();

                // まだ開いていないオリジナルファイル（実体）を追加します。
                foreach (ComponentFile file in context.Files)
                {
                    // 外部ファイルやグループ依存ファイルはサウンドアーカイブに含まないのでスキップします。
                    if (file.ExternalFilePath.Length > 0)
                    {
                        continue;
                    }

                    IOutputItem originalOutputItem = null;
                    file.OutputTarget.ItemDictionary.TryGetValue(string.Empty, out originalOutputItem);

                    if (originalOutputItem == null)
                    {
                        continue;
                    }

                    // すでにグループに埋め込まれている場合は、
                    // 二重に開かないようにすることで、グループ内のファイル（実体）を直接参照します。
                    // リンク扱いの場合は、サウンドアーカイブにファイル（実体）を追加します。
                    if (this.embededGroupItemOutputs.Contains(originalOutputItem)) { continue; }

                    // ユーザー管理グループにのみ存在するファイルは、サウンドアーカイブから除外します。
                    if (this.userManagermentGroupItemOutputs.Contains(originalOutputItem)) { continue; }

                    soundArchiveFileEntities.Add(this.GetFileEntity(originalOutputItem));
                }

                string preprocessedTag = this.GetPreprocessedTag(context.ProjectFilePath);

                var builder = new SoundArchiveFileBuilder(
                        context.Traits.BinaryFileInfo.SoundArchiveSignature,
                        context.Traits.BinaryFileInfo.SoundArchiveVersion)
                {
                    PreprocessedTag = preprocessedTag,
                };

                SoundArchiveBinary soundArchiveFile = builder.
                    Build(context, this.fileEntities, soundArchiveFileEntities.ToArray());

                DomElement soundArchiveFileElement = null;

                {
                    var domBuilder = new DomBuilder();
                    soundArchiveFileElement = domBuilder.Build(soundArchiveFile);

                    domBuilder = null;
                }

                // DomBuilder.Build() 後の不要メモリを回収
                GC.Collect();
                GC.WaitForPendingFinalizers();
                GC.Collect();

                DomWriter domWriter = new DomWriter(writer);
                domWriter.Run(new DomObjectWriter(), soundArchiveFileElement);

                // バイナリXMLを出力します。
                using (Stream stream = this.mapOutputItem.OpenWrite())
                {
                    new SoundArchiveBinaryXmlExporter(
                        context, domWriter, this.fileEntities,
                        Path.GetDirectoryName(this.mapOutputItem.Path)).Export(stream);
                }
            }
            finally
            {
                foreach (object obj in this.fileEntities.Values)
                {
                    if (obj is ByteStream)
                    {
                        (obj as ByteStream).Stream.Close();
                    }
                }
            }
        }

        private bool IsGroupFile(ComponentFile file)
        {
            Assertion.Argument.NotNull(file);
            return (file.Components.Count == 1) && (file.Components[0] is GroupBase);
        }

        private void BuildGroups(SoundArchiveContext context)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Operation.True(this.fileEntities != null);

            List<ComponentFile> linkGroupFiles = new List<ComponentFile>();

            // 埋め込みグループを先にビルドします。
            // リンクグループは埋め込みグループ内のファイルを参照する可能性があるため。
            foreach (ComponentFile file in context.Files)
            {
                if (!IsGroupFile(file)) { continue; }

                IOutputItem groupOutputItem = null;
                file.OutputTarget.ItemDictionary.TryGetValue(string.Empty, out groupOutputItem);

                if (groupOutputItem == null ||
                    this.fileEntities.ContainsKey(groupOutputItem))
                {
                    throw new Exception("internal error");
                }

                GroupBase group = file.Components[0] as GroupBase;
                Assertion.Operation.ObjectNotNull(group);

                switch (group.OutputType)
                {
                    case GroupOutputType.Link:
                        // リンクグループは、上記理由により、後で処理します。
                        linkGroupFiles.Add(file);
                        break;

                    case GroupOutputType.Embedding:
                        {
                            object fileObject = BuildEmbeddingGroup(context, group);
                            this.fileEntities.Add(groupOutputItem, fileObject);
                        }
                        break;

                    case GroupOutputType.UserManagement:
                        this.SkipUserManagementGroupItemFiles(context, file, group);
                        break;
                }
            }

            foreach (ComponentFile file in linkGroupFiles)
            {
                Assertion.Operation.True(IsGroupFile(file));

                IOutputItem groupOutputItem = null;
                file.OutputTarget.ItemDictionary.TryGetValue(string.Empty, out groupOutputItem);

                Assertion.Operation.ObjectNotNull(groupOutputItem);

                object fileObject = BuildLinkGroup(context, file.Components[0] as GroupBase);
                this.fileEntities.Add(groupOutputItem, fileObject);
            }
        }

        private GroupBinary BuildEmbeddingGroup(SoundArchiveContext context, GroupBase group)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(group);
            Assertion.Argument.True(group.OutputType == GroupOutputType.Embedding);
            Assertion.Operation.True(this.fileEntities != null);

            var groupFileEntityDictionary = new Dictionary<IOutputItem, object>();
            var groupFileEntities = new List<object>();

            // 埋め込みグループにファイル（実体）を追加します。
            // 埋め込みグループは各々が実体をもつため CreateFileEntity() します。
            // 埋め込みグループはグループアイテムテーブルから実体を参照する際、
            // 必ず自身が保持する実体を参照するために groupFileEntities と groupFileEntityDictionary に
            // 同じ内容を追加しています。
            foreach (ComponentFile file in group.GetItemFiles())
            {
                IOutputItem outputItem = GetOutputItemForGroup(file, group);

                if (outputItem != null)
                {
                    object fileEntity = this.CreateFileEntity(outputItem);

                    this.embededGroupItemOutputs.Add(outputItem);
                    groupFileEntities.Add(fileEntity);
                    groupFileEntityDictionary.Add(outputItem, fileEntity);
                }
            }

            return new GroupFileBuilder(
                context.Traits.BinaryFileInfo.GroupSignature,
                context.Traits.BinaryFileInfo.GroupVersion).
                Build(context, group, groupFileEntityDictionary, groupFileEntities.ToArray());
        }

        private GroupBinary BuildLinkGroup(SoundArchiveContext context, GroupBase group)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(group);
            Assertion.Argument.True(group.OutputType == GroupOutputType.Link);
            Assertion.Operation.True(this.fileEntities != null);

            foreach (ComponentFile file in group.GetItemFiles())
            {
                IOutputItem originalOutputItem = null;
                file.OutputTarget.ItemDictionary.TryGetValue(string.Empty, out originalOutputItem);

                if (originalOutputItem == null)
                {
                    // 有効なオリジナルファイルが含まれているはずなので、例外とします。
                    throw new Exception("internal error : link group must has original files.");
                }

                if (this.userManagermentGroupItemOutputs.Contains(originalOutputItem))
                {
                    // TODO : ★警告する

                    // ユーザー管理グループファイル一覧から削除し、サウンドアーカイブにファイルを含めるようにする
                    this.userManagermentGroupItemOutputs.Remove(originalOutputItem);
                }

                // 参照解決のため、グループビルド前にファイル（実体）を開いておきます。
                // リンクグループにはファイル（実体）を埋め込みません。
                this.GetFileEntity(originalOutputItem);
            }

            return new GroupFileBuilder(
                context.Traits.BinaryFileInfo.GroupSignature,
                context.Traits.BinaryFileInfo.GroupVersion).
                Build(context, group, this.fileEntities, new object[0]);
        }

        private void SkipUserManagementGroupItemFiles(SoundArchiveContext context, ComponentFile groupFile, GroupBase group)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(groupFile);
            Assertion.Argument.NotNull(group);
            Assertion.Argument.True(group.OutputType == GroupOutputType.UserManagement);
            Assertion.Operation.True(this.fileEntities != null);

            // ユーザー管理グループファイル自身をサウンドアーカイブに含めない
            {
                IOutputItem groupOutputItem = null;
                groupFile.OutputTarget.ItemDictionary.TryGetValue(string.Empty, out groupOutputItem);
                Ensure.Operation.ObjectNotNull(groupOutputItem);

                this.userManagermentGroupItemOutputs.Add(groupOutputItem);
            }

            // ユーザー管理グループに含まれるオリジナルファイルは、
            // サウンドアーカイブに不要な可能性があるので、
            // 保持しておいて、リンクグループから参照されていないことを BuildLinkGroup() で確認する
            foreach (ComponentFile file in group.GetItemFiles())
            {
                IOutputItem originalOutputItem = null;
                file.OutputTarget.ItemDictionary.TryGetValue(string.Empty, out originalOutputItem);

                if (originalOutputItem != null)
                {
                    this.userManagermentGroupItemOutputs.Add(originalOutputItem);
                }
            }
        }

        private IOutputItem GetOutputItemForGroup(ComponentFile file, GroupBase group)
        {
            Assertion.Argument.NotNull(file);
            Assertion.Argument.NotNull(group);

            IOutputItem groupDependedOutputItem = null;
            if (file.OutputTarget.ItemDictionary.TryGetValue(group.Name, out groupDependedOutputItem))
            {
                return groupDependedOutputItem;
            }

            return file.OutputTarget.ItemDictionary[string.Empty];
        }

        private object GetFileEntity(IOutputItem outputItem)
        {
            Assertion.Argument.NotNull(outputItem);

            object fileEntity = null;
            this.fileEntities.TryGetValue(outputItem, out fileEntity);

            if (fileEntity != null)
            {
                return fileEntity;
            }

            return CreateFileEntity(outputItem);
        }

        private object CreateFileEntity(IOutputItem outputItem)
        {
            Assertion.Argument.NotNull(outputItem);

            var fileEntity = new FileObject(outputItem.Path);

            if (!this.fileEntities.ContainsKey(outputItem))
            {
                this.fileEntities.Add(outputItem, fileEntity);
            }

            return fileEntity;
        }

        private string GetPreprocessedTag(string projectFilePath)
        {
            string cockpitFilePath = Path.ChangeExtension(projectFilePath, CockpitExtension);

            if (!File.Exists(cockpitFilePath))
            {
                return string.Empty;
            }

            using (var fileStream = File.Open(cockpitFilePath, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                using (var reader = new StreamReader(fileStream))
                {
                    return reader.ReadLine();
                }
            }
        }
    }
}
