﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Binarization;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// グループを作成します。
    /// </summary>
    internal class GroupProcessor : ComponentProcessor<SoundArchiveContext, GroupBase>
    {
        private Dictionary<IOutputItem, object> fileEntities;

        public GroupProcessor(GroupBase component, IOutputItem outputItem)
            : base(component, outputItem)
        {
        }

        protected override void ProcessInternal(SoundArchiveContext context)
        {
            Ensure.Argument.NotNull(context);

            try
            {
                if (File.Exists(this.OutputTargetItem.Path))
                {
                    File.Delete(this.OutputTargetItem.Path);
                }

                using (Stream stream = this.OutputTargetItem.OpenWrite())
                {
                    Write(context, context.CreateBinaryWriter(stream));
                }
            }
            catch
            {
                try
                {
                    if (File.Exists(this.OutputTargetItem.Path))
                    {
                        File.Delete(this.OutputTargetItem.Path);
                    }
                }
                catch
                {
                }

                throw;
            }
        }

        protected override void OutputLog(SoundArchiveContext context, Component[] components)
        {
            context.Logger.AddLine(
                new InformationLine(
                    string.Format("[GROUP] {0} > {1}",
                    this.TargetComponent.Name,
                    Path.GetFileName(this.OutputTargetItem.Path)
                    ))
                );
        }

        private void Write(SoundArchiveContext context, BinaryWriter writer)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(writer);

            try
            {
                this.fileEntities = new Dictionary<IOutputItem, object>();

                IOutputItem groupOutputItem = null;
                this.TargetComponent.GetOutputTarget().ItemDictionary.TryGetValue(string.Empty, out groupOutputItem);

                if (groupOutputItem == null ||
                    this.fileEntities.ContainsKey(groupOutputItem))
                {
                    throw new Exception("internal error");
                }

                GroupBase group = this.TargetComponent as GroupBase;
                Assertion.Operation.ObjectNotNull(group);

                if (group.OutputType != GroupOutputType.UserManagement)
                {
                    throw new Exception("internal error");
                }

                var groupBinary = this.BuildEmbeddingGroup(context, group);

                DomElement groupFileElement = new DomBuilder().Build(groupBinary);
                DomWriter domWriter = new DomWriter(writer);
                domWriter.Run(new DomObjectWriter(), groupFileElement);
            }
            finally
            {
                foreach (object obj in this.fileEntities.Values)
                {
                    if (obj is ByteStream)
                    {
                        (obj as ByteStream).Stream.Close();
                    }
                }
            }
        }

        private bool IsGroupFile(ComponentFile file)
        {
            Assertion.Argument.NotNull(file);
            return (file.Components.Count == 1) && (file.Components[0] is GroupBase);
        }

        private GroupBinary BuildEmbeddingGroup(SoundArchiveContext context, GroupBase group)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(group);
            Assertion.Argument.True(group.OutputType == GroupOutputType.UserManagement);
            Assertion.Operation.True(this.fileEntities != null);

            var groupFileEntityDictionary = new Dictionary<IOutputItem, object>();
            var groupFileEntities = new List<object>();

            // 埋め込みグループにファイル（実体）を追加します。
            // 埋め込みグループは各々が実体をもつため CreateFileEntity() します。
            // 埋め込みグループはグループアイテムテーブルから実体を参照する際、
            // 必ず自身が保持する実体を参照するために groupFileEntities と groupFileEntityDictionary に
            // 同じ内容を追加しています。
            foreach (ComponentFile file in group.GetItemFiles())
            {
                IOutputItem outputItem = GetOutputItemForGroup(file, group);

                if (outputItem != null)
                {
                    object fileEntity = this.CreateFileEntity(outputItem);

                    groupFileEntities.Add(fileEntity);
                    groupFileEntityDictionary.Add(outputItem, fileEntity);
                }
            }

            return new GroupFileBuilder(
                context.Traits.BinaryFileInfo.GroupSignature,
                context.Traits.BinaryFileInfo.GroupVersion).
                Build(context, group, groupFileEntityDictionary, groupFileEntities.ToArray());
        }

        private IOutputItem GetOutputItemForGroup(ComponentFile file, GroupBase group)
        {
            Assertion.Argument.NotNull(file);
            Assertion.Argument.NotNull(group);

            IOutputItem groupDependedOutputItem = null;
            if (file.OutputTarget.ItemDictionary.TryGetValue(group.Name, out groupDependedOutputItem))
            {
                return groupDependedOutputItem;
            }

            return file.OutputTarget.ItemDictionary[string.Empty];
        }

        private object CreateFileEntity(IOutputItem outputItem)
        {
            Assertion.Argument.NotNull(outputItem);

            var fileEntity = new FileObject(outputItem.Path);

            if (!this.fileEntities.ContainsKey(outputItem))
            {
                this.fileEntities.Add(outputItem, fileEntity);
            }

            return fileEntity;
        }
    }
}
