﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Xml.Serialization;
    using NintendoWare.SoundFoundation.Binarization;
    using NintendoWare.SoundFoundation.Core.IO;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveElements;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.SoundArchiveXmlElements;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using ToolDevelopmentKit;

    internal class SoundArchiveBinaryXmlExporter
    {
        private const uint InvalidFileID = 0xFFFFFFFF;

        private SoundArchiveContext context;
        private DomWriter domWriter;
        private string baseDirectoryPath;
        private IDictionary<IOutputItem, object> fileEntities;

        public SoundArchiveBinaryXmlExporter(
            SoundArchiveContext context, DomWriter domWriter,
            IDictionary<IOutputItem, object> fileEntities, string baseDirectoryPath)
        {
            Ensure.Argument.NotNull(context);
            Ensure.Argument.NotNull(domWriter);
            Ensure.Argument.NotNull(fileEntities);
            Ensure.Argument.NotNull(baseDirectoryPath);

            this.context = context;
            this.domWriter = domWriter;
            this.fileEntities = fileEntities;
            this.baseDirectoryPath = baseDirectoryPath;
        }

        //-----------------------------------------------------------------
        // 出力
        //-----------------------------------------------------------------

        public void Export(Stream stream)
        {
            Ensure.Argument.NotNull(stream);

            SoundArchiveBinaryXml soundArchiveXml = new SoundArchiveBinaryXml()
            {
                Name = this.context.GetSoundArchiveName(),
                Sounds = new List<SoundXml>(),
                SoundGroups = new List<SoundGroupXml>(),
                SoundSetBanks = new List<SoundSetBankXml>(),
                Players = new List<PlayerXml>(),
                WaveArchives = new List<WaveArchiveXml>(),
                Groups = new List<GroupXml>(),
                Files = new List<FileXml>(),
                ElementMap = new ElementMapXml(),
                IsAddonSoundArchive = this.context.AddonSoundSet != null,
            };

            this.BuildSounds(soundArchiveXml);
            this.BuildSoundGroups(soundArchiveXml);
            this.BuildSoundSetBanks(soundArchiveXml);
            this.BuildPlayers(soundArchiveXml);
            this.BuildWaveArchives(soundArchiveXml);
            this.BuildGroups(soundArchiveXml);
            this.BuildFiles(soundArchiveXml);
            this.BuildWaveFiles(soundArchiveXml);
            this.BuildMap(soundArchiveXml);

            new XmlSerializer(typeof(SoundArchiveBinaryXml)).Serialize(stream, soundArchiveXml);
        }

        //-----------------------------------------------------------------
        // ビルド
        //-----------------------------------------------------------------

        private void BuildSounds(SoundArchiveBinaryXml soundArchiveXml)
        {
            Assertion.Argument.NotNull(soundArchiveXml);

            foreach (StreamSoundBase sound in this.context.StreamSounds)
            {
                ComponentFile file = this.context.GetFile(sound.GetOutputTarget());

                StreamSoundXml soundXml = new StreamSoundXml()
                {
                    ID = sound.ID,
                    Name = sound.Name,
                    FileID = (uint)this.context.Files.IndexOf(file),
                };

                soundArchiveXml.Sounds.Add(soundXml);
            }

            foreach (WaveSoundBase sound in this.context.WaveSounds)
            {
                WaveSoundXml soundXml = new WaveSoundXml()
                {
                    ID = sound.ID,
                    Name = sound.Name,
                };

                soundArchiveXml.Sounds.Add(soundXml);
            }

            foreach (SequenceSoundBase sound in this.context.SequenceSounds)
            {
                ComponentFile file = this.context.GetFile(sound.GetOutputTarget());

                SequenceSoundXml soundXml = new SequenceSoundXml()
                {
                    ID = sound.ID,
                    Name = sound.Name,
                    FileID = (uint)this.context.Files.IndexOf(file),
                };

                soundArchiveXml.Sounds.Add(soundXml);
            }
        }

        private void BuildSoundGroups(SoundArchiveBinaryXml soundArchiveXml)
        {
            Assertion.Argument.NotNull(soundArchiveXml);

            if (!context.Traits.IsWaveSound2BinaryEnabled)
            {
                foreach (WaveSoundSetBase soundGroup in this.context.WaveSoundSets)
                {
                    ComponentFile file = this.context.GetFile(soundGroup.GetOutputTarget());

                    SoundGroupXml soundGroupXml = new WaveSoundGroupXml()
                    {
                        ID = soundGroup.ID,
                        Name = soundGroup.Name,
                    };

                    if (file != null)
                    {
                        soundGroupXml.FileID = (uint)this.context.Files.IndexOf(file);
                    }

                    soundArchiveXml.SoundGroups.Add(soundGroupXml);
                }
            }

            foreach (SequenceSoundSetBase soundGroup in this.context.SequenceSoundSets)
            {
                SoundGroupXml soundGroupXml = new SequenceSoundGroupXml()
                {
                    ID = soundGroup.ID,
                    Name = soundGroup.Name,
                };

                soundArchiveXml.SoundGroups.Add(soundGroupXml);
            }
        }

        private void BuildSoundSetBanks(SoundArchiveBinaryXml soundArchiveXml)
        {
            Assertion.Argument.NotNull(soundArchiveXml);

            foreach (SoundSetBankBase soundSetBank in this.context.SoundSetBanks)
            {
                uint fileID = InvalidFileID;
                var output = soundSetBank.GetOutputTarget();

                if (output != null)
                {
                    fileID = (uint)this.context.Files.IndexOf(this.context.GetFile(output));
                }
                else
                {
                    // メインサウンドアーカイブのバンクを追加サウンドアーカイブで参照する場合はバンク情報のみを出力するので、スキップする
                    if (!soundArchiveXml.IsAddonSoundArchive)
                    {
                        // 追加サウンドアーカイブでない場合は、ここに来ることはないはず。
                        throw new InvalidOperationException("invalid bank fileID");
                    }
                }

                SoundSetBankXml soundSetBankXml = new SoundSetBankXml()
                {
                    ID = soundSetBank.ID,
                    Name = soundSetBank.Name,
                    FileID = fileID,
                };

                soundArchiveXml.SoundSetBanks.Add(soundSetBankXml);
            }
        }

        private void BuildPlayers(SoundArchiveBinaryXml soundArchiveXml)
        {
            Assertion.Argument.NotNull(soundArchiveXml);

            foreach (PlayerBase player in this.context.Players)
            {
                PlayerXml playerXml = new PlayerXml()
                {
                    ID = player.ID,
                    Name = player.Name,
                };

                soundArchiveXml.Players.Add(playerXml);
            }
        }

        private void BuildWaveArchives(SoundArchiveBinaryXml soundArchiveXml)
        {
            Assertion.Argument.NotNull(soundArchiveXml);

            foreach (WaveArchiveBase waveArchive in this.context.WaveArchives)
            {
                ComponentFile file = this.context.GetFile(waveArchive.GetOutputTarget());

                WaveArchiveXml waveArchiveXml = new WaveArchiveXml()
                {
                    ID = waveArchive.ID,
                    Name = waveArchive.Name,
                    FileID = (uint)this.context.Files.IndexOf(file),
                };

                soundArchiveXml.WaveArchives.Add(waveArchiveXml);
            }
        }

        private void BuildGroups(SoundArchiveBinaryXml soundArchiveXml)
        {
            Assertion.Argument.NotNull(soundArchiveXml);

            foreach (GroupBase group in this.context.Groups)
            {
                GroupXml groupXml = null;

                if (group.OutputType == GroupOutputType.UserManagement)
                {
                    IOutputItem outputItem = group.GetOutputTarget().ItemDictionary[string.Empty];
                    uint size = 0;

                    try
                    {
                        size = (uint)new System.IO.FileInfo(outputItem.Path).Length;
                    }
                    catch { }

                    groupXml = new GroupXml()
                    {
                        ID = group.ID,
                        Name = group.Name,
                        FileID = 0xFFFFFFFF,
                        OutputType = group.OutputType.ToText(),
                        Size = size,
                        TotalSize = size,
                        Items = new List<GroupItemXml>(),
                    };
                }
                else
                {
                    ComponentFile file = this.context.GetFile(group.GetOutputTarget());
                    IOutputItem outputItem = file.OutputTarget.ItemDictionary[string.Empty];
                    ObjectReference reference = this.GetOutputItemReference(outputItem);

                    groupXml = new GroupXml()
                    {
                        ID = group.ID,
                        Name = group.Name,
                        FileID = (uint)this.context.Files.IndexOf(file),
                        OutputType = group.OutputType.ToText(),
                        Size = reference.Size,
                        TotalSize = reference.Size,
                        Items = new List<GroupItemXml>(),
                    };
                }

                this.BuildGroupItems(group, groupXml);

                soundArchiveXml.Groups.Add(groupXml);
            }
        }

        private void BuildGroupItems(GroupBase group, GroupXml groupXml)
        {
            Assertion.Argument.NotNull(group);
            Assertion.Argument.NotNull(groupXml);

            ulong totalSize = 0;

            foreach (ComponentFile file in group.GetItemFiles())
            {
                ObjectReference reference = this.GetOutputItemReference(this.GetOutputItemForGroup(file, group));

                groupXml.Items.Add(
                    new GroupItemXml()
                    {
                        FileID = (uint)this.context.Files.IndexOf(file),
                    }
                    );

                totalSize += reference.Size;
            }

            if (group.OutputType == GroupOutputType.Link)
            {
                groupXml.TotalSize += totalSize;
            }
        }

        private void BuildFiles(SoundArchiveBinaryXml soundArchiveXml)
        {
            Assertion.Argument.NotNull(soundArchiveXml);

            uint fileID = 0;

            foreach (ComponentFile file in this.context.Files)
            {
                Ensure.Operation.True(file.Components.Count > 0);

                IOutputItem outputItem = this.GetOutputItem(file);
                ObjectReference reference =
                    (outputItem != null) ? this.GetOutputItemReference(outputItem) : null;

                if (file.IsExternal)
                {
                    var filePath = Path.Combine(
                        Path.GetDirectoryName(this.context.ProjectFilePath),
                        this.context.Project.OutputDirectoryPath);

                    filePath = Path.Combine(
                        filePath,
                        file.ExternalFilePath);

                    soundArchiveXml.Files.Add(
                        new ExternalFileXml()
                        {
                            ID = fileID,
                            Name = this.GetFileName(file, outputItem, false),
                            ExternalPath = file.ExternalFilePath,
                            Size = (File.Exists(filePath)) ?
                                (ulong)new System.IO.FileInfo(filePath).Length : ObjectReference.InvalidValue,
                        }
                        );
                }
                else
                {
                    soundArchiveXml.Files.Add(
                        new InternalFileXml()
                        {
                            ID = fileID,
                            Name = this.GetFileName(file, outputItem, false),
                            Size = (reference != null) ? reference.Size : ObjectReference.InvalidValue,
                        }
                        );
                }

                fileID++;
            }
        }

        private void BuildWaveFiles(SoundArchiveBinaryXml soundArchiveXml)
        {
            Assertion.Argument.NotNull(soundArchiveXml);

            if (soundArchiveXml.WaveFiles == null)
            {
                soundArchiveXml.WaveFiles = new List<WaveFileXml>();
            }

            foreach (ComponentFile file in this.context.WaveFiles)
            {
                Ensure.Operation.True(file.Components.Count > 0);

                IOutputItem outputItem = this.GetOutputItem(file);

                var waveFileXml = new WaveFileXml()
                {
                    FilePath = this.CreateRelativePath(file.Components[0].GetFilePathForConvert()),
                    BinaryFilePath = this.GetFileName(file, outputItem, false),
                    ReferenceCount = file.Components.Count,
                    ArchiveCount = file.Archives.Count,
                    Users = new List<WaveFileUserComponentXml>(),
                    WaveArchives = new List<WaveArchiveXml>(),
                };

                foreach (var component in file.Components)
                {
                    if (component is SoundSetItem)
                    {
                        waveFileXml.Users.Add(
                            new WaveFileUserWaveSoundXml()
                            {
                                ID = component.ID,
                                Name = component.Name,
                            });
                    }
                    else if (component is VelocityRegion)
                    {
                        var velocityRegion = component as VelocityRegion;
                        var keyRegion = velocityRegion.Parent as KeyRegion;

                        waveFileXml.Users.Add(
                            new WaveFileUserVelocityRegionXml()
                            {
                                BankFilePath = this.CreateRelativePath(this.context.BankFilePathDictionary[velocityRegion.Bank]),
                                ProgramNo = velocityRegion.Instrument.ProgramNo,
                                Key = keyRegion.KeyMin,
                                Velocity = velocityRegion.VelocityMin,
                            });
                    }
                }

                foreach (var waveArchive in file.Archives.OfType<WaveArchiveBase>())
                {
                    waveFileXml.WaveArchives.Add(new WaveArchiveXml()
                    {
                        ID = waveArchive.ID,
                        Name = waveArchive.Name,
                        FileID = (uint)this.context.Files.IndexOf(file),
                    });
                }

                soundArchiveXml.WaveFiles.Add(waveFileXml);
            }
        }

        private void BuildMap(SoundArchiveBinaryXml soundArchiveXml)
        {
            Assertion.Argument.NotNull(soundArchiveXml);

            SoundArchiveBinary binary = this.domWriter.Context.RootElement.Value as SoundArchiveBinary;

            soundArchiveXml.ElementMap.Root = this.CreateMapItem("SoundArchiveBinaryFile", binary);
            soundArchiveXml.ElementMap.Root.Items = new List<ElementMapItemXml>();
            soundArchiveXml.ElementMap.Root.Items.Add(this.CreateMapItem("BinaryHeader", binary.Header));

            ElementMapItemXml stringBlockXml = this.CreateMapItem("StringBlock", binary.StringBlock);
            if (stringBlockXml.AddressTextSpecified)
            {
                soundArchiveXml.ElementMap.Root.Items.Add(stringBlockXml);
            }

            soundArchiveXml.ElementMap.Root.Items.Add(this.CreateMapItem("InfoBlock", binary.InfoBlock));
            soundArchiveXml.ElementMap.Root.Items.Add(this.CreateMapItem(binary.FileBlock));
        }

        private ElementMapItemXml CreateMapItem(FileBlock fileBlock)
        {
            Assertion.Argument.NotNull(fileBlock);

            ElementMapItemXml mapItemXml = this.CreateMapItem("FileBlock", fileBlock);
            mapItemXml.Items = new List<ElementMapItemXml>();

            Dictionary<object, ComponentFile> outputItems = new Dictionary<object, ComponentFile>();

            foreach (ComponentFile file in this.context.Files)
            {
                Ensure.Operation.True(file.Components.Count > 0);

                IOutputItem outputItem = this.GetOutputItem(file);
                if (outputItem == null)
                {
                    continue;
                }

                object fileEntity = null;
                if (!this.fileEntities.TryGetValue(outputItem, out fileEntity))
                {
                    continue;
                }

                outputItems.Add(fileEntity, file);
            }

            foreach (object fileObject in fileBlock.Body.Items)
            {
                ComponentFile file = outputItems[fileObject];
                IOutputItem outputItem = this.GetOutputItem(file);
                ObjectReference reference = this.GetOutputItemReference(outputItem);

                ElementMapItemXml childMapItemXml = new FileElementMapItemXml()
                {
                    Address = (UInt32)reference.Address,
                    Size = (UInt32)reference.Size,
                    FileID = (uint)this.context.Files.IndexOf(file),
                    Name = this.GetFileName(file, outputItem, true),
                };

                if (file.Components[0] is GroupBase)
                {
                    GroupBase group = file.Components[0] as GroupBase;

                    switch (group.OutputType)
                    {
                        case GroupOutputType.Embedding:
                        case GroupOutputType.UserManagement:
                            childMapItemXml.Items = new List<ElementMapItemXml>();
                            this.BuildGroupMapItems(childMapItemXml, group);
                            break;
                    }
                }

                mapItemXml.Items.Add(childMapItemXml);
            }

            return mapItemXml;
        }

        private void BuildGroupMapItems(ElementMapItemXml mapItemXml, GroupBase group)
        {
            Assertion.Argument.NotNull(mapItemXml);
            Assertion.Argument.NotNull(group);

            foreach (ComponentFile file in group.GetItemFiles())
            {
                IOutputItem outputItem = this.GetOutputItemForGroup(file, group);
                ObjectReference reference = this.GetOutputItemReference(outputItem);

                mapItemXml.Items.Add(
                    new FileElementMapItemXml()
                    {
                        Address = (UInt32)reference.Address,
                        Size = (UInt32)reference.Size,
                        FileID = (uint)this.context.Files.IndexOf(file),
                        Name = this.GetFileName(file, outputItem, true),
                    }
                    );
            }
        }

        private ElementMapItemXml CreateMapItem(string name, object value)
        {
            Assertion.Argument.NotNull(name);
            Assertion.Argument.NotNull(value);

            ObjectReference reference = this.GetObjectReference(value);

            return new ElementMapItemXml()
            {
                Name = name,
                Address = (UInt32)((reference != null) ? reference.Address : ObjectReference.InvalidValue),
                Size = (UInt32)((reference != null) ? reference.Size : ObjectReference.InvalidValue),
            };
        }

        //-----------------------------------------------------------------
        // ユーティリティメソッド
        //-----------------------------------------------------------------

        private ObjectReference GetObjectReference(object value)
        {
            Assertion.Argument.NotNull(value);
            return this.domWriter.Context.ReferenceResolver.GetObjectReference(value);
        }

        private ObjectReference GetOutputItemReference(IOutputItem outputItem)
        {
            Assertion.Argument.NotNull(outputItem);

            object fileEntity = null;
            if (!this.fileEntities.TryGetValue(outputItem, out fileEntity))
            {
                return ObjectReference.Null;
            }

            return this.domWriter.Context.ReferenceResolver.GetObjectReference(fileEntity);
        }

        private IOutputItem GetOutputItem(ComponentFile file)
        {
            Assertion.Argument.NotNull(file);

            if (file.OutputTarget.ItemDictionary.Count == 0)
            {
                return null;
            }

            IOutputItem outputItem = null;
            file.OutputTarget.ItemDictionary.TryGetValue(string.Empty, out outputItem);

            if (outputItem != null)
            {
                return outputItem;
            }

            return file.OutputTarget.ItemDictionary.First().Value;
        }

        private IOutputItem GetOutputItemForGroup(ComponentFile file, GroupBase group)
        {
            Assertion.Argument.NotNull(file);
            Assertion.Argument.NotNull(group);

            IOutputItem outputItem = null;
            file.OutputTarget.ItemDictionary.TryGetValue(group.Name, out outputItem);

            if (outputItem != null)
            {
                return outputItem;
            }

            file.OutputTarget.ItemDictionary.TryGetValue(string.Empty, out outputItem);
            return outputItem;
        }

        private bool IsGroupFile(ComponentFile file)
        {
            Assertion.Argument.NotNull(file);
            return (file.Components.Count == 1) && (file.Components[0] is GroupBase);
        }

        private int GetMultipleOutputCount(ComponentFile file)
        {
            Assertion.Argument.NotNull(file);
            return (from string key in file.OutputTarget.ItemDictionary.Keys
                    where !key.StartsWith(ComponentContext.IntermediateOutputPrefix)
                    select key).Count();
        }

        private string GetFileName(ComponentFile file, IOutputItem outputItem, bool canAllowMultipleOutputs)
        {
            Assertion.Argument.NotNull(file);

            // 出力がない or グループファイルは名前を表示します。
            if (outputItem == null ||
                this.IsGroupFile(file))
            {
                return file.Components[0].Name;
            }

            // File リスト等、複数出力を許さない場合は、名前を表示します。
            if (!canAllowMultipleOutputs)
            {
                int outputCount = this.GetMultipleOutputCount(file);

                if (outputCount > 1)
                {
                    return file.Components[0].Name + string.Format(" ({0} files)", outputCount);
                }
            }

            return this.CreateRelativePath(outputItem.Path);
        }

        private string CreateRelativePath(string filePath)
        {
            Assertion.Argument.NotNull(filePath);

            if (filePath.Length == 0)
            {
                return string.Empty;
            }

            return PathEx.MakeRelative(filePath, this.baseDirectoryPath);
        }
    }
}
