﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System;
    using System.Runtime.InteropServices;
    using NintendoWare.ToolDevelopmentKit;

    internal class SamplingrateConverter : Filter
    {
        private int outputFrameCount = 0;
        private double magnification = 1.0;

        public SamplingrateConverter()
            : base(new InputConnectionPoint())
        {
        }

        /// <summary>
        /// 入力接続ポイントを取得します。
        /// </summary>
        private InputConnectionPoint Input
        {
            get { return this.Inputs[0]; }
        }

        /// <summary>
        /// 入力ストリーム情報を取得します。
        /// </summary>
        private WaveStreamInformation InputStreamInformation
        {
            get { return this.Inputs[0].StreamInformation; }
        }

        /// <summary>
        /// 出力ストリーム情報を取得します。
        /// </summary>
        private WaveStreamInformation OutputStreamInformation
        {
            get { return this.Outputs[0].StreamInformation; }
        }

        /// <summary>
        /// 入力フォーマットを取得します。
        /// </summary>
        private WaveFormat InputFormat
        {
            get { return this.Inputs[0].StreamInformation.Format; }
        }

        /// <summary>
        /// 出力フォーマットを取得します。
        /// </summary>
        private WaveFormat OutputFormat
        {
            get { return this.Outputs[0].StreamInformation.Format; }
        }

        protected override void ValidateInputs(InputConnectionPoint[] inputs)
        {
            Ensure.Operation.True(
                                  inputs[0].StreamInformation.Format.ChannelCount == 1,
                                  "InputChannel for SamplingrateConverter must be 1."
                                  );
        }

        /// <summary>
        /// 出力接続ポイントを生成します。
        /// </summary>
        /// <returns>出力接続ポイントの配列を返します。</returns>
        protected override OutputConnectionPoint[] CreateOutputs()
        {
            if (this.InputFormat.HasLoop == true)
            {
                int loopLength = this.InputFormat.LoopEndFrame - this.InputFormat.LoopStartFrame;

                if (loopLength % 2 != 0)
                {
                    this.magnification = (loopLength + 1) / (double)loopLength;

                    this.outputFrameCount =
                        CalculateCeiling(this.InputStreamInformation.FrameCount);

                    int outputLength = CalculateCeiling(this.InputStreamInformation.Length);

                    return new OutputConnectionPoint[]
                        {
                            new OutputConnectionPoint(
                                new WaveStreamInformation(
                                    outputLength,
                                    this.outputFrameCount,
                                    this.GetOutputFormat())
                             )
                        };
                }
            }

            return new OutputConnectionPoint[]
                {
                    new OutputConnectionPoint(
                        new WaveStreamInformation(
                            this.InputStreamInformation.Length,
                            this.InputStreamInformation.FrameCount,
                            this.GetOutputFormat())
                        )
                };
        }

        /// <summary>
        /// パケットを処理します。
        /// </summary>
        /// <param name="input">入力接続ポイントを指定します。</param>
        /// <param name="packet">パケットを指定します。</param>
        protected override void Process(InputConnectionPoint input, Packet packet)
        {
            Assertion.Argument.NotNull(packet);

            if (this.InputFormat.HasLoop == false)
            {
                this.SendToAll(packet);
                return;
            }

            int outputLength = CalculateCeiling(packet.Samples.Length);
            byte[] outputSamples = new byte[outputLength];

            Packet outputPacket = Packet.Create(outputSamples);

            GCHandle inputHandle = GCHandle.Alloc(packet.Samples, GCHandleType.Pinned);
            GCHandle outputHandle = GCHandle.Alloc(outputPacket.Samples, GCHandleType.Pinned);

            try
            {
                int outputLoopStartFrame;
                int outputSampleRate;

                int frameCount = packet.Samples.Length / this.InputFormat.FrameLength;
                Resample(
                         outputHandle.AddrOfPinnedObject(),
                         outputLength / this.OutputFormat.FrameLength,
                         out outputLoopStartFrame,
                         out outputSampleRate,
                         inputHandle.AddrOfPinnedObject(),
                         frameCount,
                         this.InputFormat.LoopStartFrame,
                         this.InputFormat.SamplingRate
                         );
                this.SendToAll(outputPacket);
            }
            finally
            {
                inputHandle.Free();
                outputHandle.Free();
            }
        }

        [DllImport("WaveCodecGeneric.dll", CallingConvention = CallingConvention.Cdecl)]
        private static extern void Resample(
                                            IntPtr output,
                                            int outputFrameCount,
                                            [Out] out int outputLoopStartFrame,
                                            [Out] out int outputSampleRate,
                                            IntPtr input,
                                            int inputFrameCount,
                                            int inputLoopStartFrame,
                                            int inputSampleRate);

        /// <summary>
        /// 出力フォーマットを取得します。
        /// </summary>
        /// <returns>出力フォーマットを返します。</returns>
        private WaveFormat GetOutputFormat()
        {
            if (this.InputFormat.HasLoop == false)
            {
                return new WaveFormat(this.InputFormat);
            }
            else
            {
                return new WaveFormat(this.InputFormat)
                {
                    SamplingRate = Calculate(this.InputFormat.SamplingRate),
                    OriginalLoopStartFrame = Calculate(this.InputFormat.OriginalLoopStartFrame),
                    OriginalLoopEndFrame = Calculate(this.InputFormat.OriginalLoopEndFrame),
                    LoopStartFrame = Calculate(this.InputFormat.LoopStartFrame),
                    LoopEndFrame = Calculate(this.InputFormat.LoopEndFrame),
                };
            }
        }

        /// <summary>
        /// リサンプリング後の値を四捨五入して計算します。
        /// </summary>
        /// <returns>リサンプリング後の値を返します。</returns>
        private int Calculate(int value)
        {
            return (int)Math.Floor(value * magnification + 0.5);
        }

        /// <summary>
        /// リサンプリング後の値を Ceiling して計算します。
        /// </summary>
        /// <returns>リサンプリング後の値を返します。</returns>
        private int CalculateCeiling(int value)
        {
            return (int)Math.Ceiling(value * magnification);
        }
    }
}
